/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonExample.C
 *    An example how to use polygons.
 *
 * @author Jens Kessler
 * @date   2010/10/1
 */

#include <iostream>

#include <boost/geometry.hpp>

#include <serialization/Print.h>

#include <geometry/Geometry.h>

using namespace std;
using namespace mira;
using namespace boost::geometry;

#if BOOST_VERSION >= 107100

/**
 * New improved simplify algorithm for rings only compiles for Dimension=2.
 * Use generic algorithm instead, by defining dispatch as in boost 1.65.
 */
namespace boost { namespace geometry {
namespace dispatch {

template <>
struct simplify<Polygon3f, ring_tag>
	: detail::simplify::simplify_range
		<
			core_detail::closure::minimum_ring_size
				<
					geometry::closure<Polygon3f>::value
				>::value
		>
{};

}
}}

#endif // BOOST_VERSION >= 107100

int main()
{
	Point3f A(1.0, 0.0, 0.0);
	Point3f B(1.0, 1.0, 0.0);
	Point3f C(0.0, 1.0, 0.0);
	Point3f D(0.0, 0.0, 0.0);

	Polygon3f poly;

	poly.push_back(A);
	poly.push_back(B);
	poly.push_back(C);
	poly.push_back(D);
	poly.push_back(A);

	Polygon3f triangle = createTriangle(Point3f(2.0, 2.0, 0.0),
	                                    Point3f(-2.0, 2.0, 0.0),
	                                    Point3f(-2.0,-2.0,0.0));

	cout << "Defined points:" << endl;
	cout << "  A: " << dsv(A) << endl;
	cout << "  B: " << dsv(B) << endl;
	cout << "  C: " << dsv(C) << endl;
	cout << "  D: " << dsv(D) << endl;
	cout << "Constructed polygon:" << endl << "  " << dsv(poly) << endl;
	cout << "Constructed triangle: " << endl << "  " << dsv(triangle) << endl;

	// bounding box
	Box3f box;
	envelope(poly, box);
	cout << "Bounding box of polygon:" << endl << "  " << dsv(box) << endl;

	envelope(triangle, box);
	cout << "Bounding box of triangle:" << endl << "  " << dsv(box) << endl;

	// length of a polygon
	cout << "Length of polygon:" << endl << "  " << length(poly) << endl;

	// simplification
	Polygon3f simple = poly;
	simplify(poly,simple,0.5);
	cout << "Simplification of polygon:" << endl << "  " << dsv(simple) << endl;

	///////////////////////////////////////////////////////////////////////////

	Point2f X(1.0, 0.0);
	Point2f Y(1.0, 1.0);
	Point2f Z(0.0, 1.0);

	Polygon2f poly2D;
	poly2D.push_back(X);
	poly2D.push_back(Y);
	poly2D.push_back(Z);
	poly2D.push_back(X);

	cout << "Working on 2D polygon:" << endl << "  " << dsv(poly2D) << endl;

	// center of mass
	Point2f center;
	centroid(poly2D, center); // centroid is only available in 2D shapes
	cout << "Centroid (2D only):" << endl << "  " << dsv(center) << endl;

	// convex hull
	Polygon2f hull;
	convex_hull(poly2D, hull);
	cout << "Convex hull (2D only):" << endl << "  " << dsv(hull) << endl;

	///////////////////////////////////////////////////////////////////////////

	typedef std::vector<boost::geometry::model::polygon<Point2f>> PolygonSet;

	Rect2f rect(Point2f(0.5, 0.5), Point2f(1.5, 2.5));

	// union of geometries
	PolygonSet tUnion;
	union_(rect, poly2D, tUnion);
	std::cout << std::endl;
	std::cout << "Poly2D: " << std::endl << "  " << dsv(poly2D) << std::endl;
	std::cout << "Rect: " << std::endl << "  " << dsv(rect) << std::endl;
	std::cout << "Union of Rect and Poly2D: " << std::endl;
	for (PolygonSet::iterator it = tUnion.begin(); it != tUnion.end(); ++it)
		std::cout << "  " << dsv(*it) << std::endl;

	// intersect geometries
	PolygonSet tIntersect;
	intersection(rect, poly2D, tIntersect);
	std::cout << std::endl;
	std::cout << "Poly2D: " << std::endl << "  " << dsv(poly2D) << std::endl;
	std::cout << "Rect: " << std::endl << "  " << dsv(rect) << std::endl;
	std::cout << "Intersection of Rect and Poly2D" << std::endl;
	for (PolygonSet::iterator it = tIntersect.begin(); it != tIntersect.end(); ++it)
		std::cout << "  " << dsv(*it) << std::endl;

	return 0;
}
