/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeEditorApp.C
 *    Standalone application of the tape editor.
 *
 * @author Erik Einhorn
 * @date   2013/02/06
 */

#include <QApplication>
#include <QMainWindow>
#include <QToolBar>
#include <QMenuBar>
#include <QMenu>

#include <fw/Framework.h>
#include <fw/FrameworkWithGui.h>

#include <TapeEditor.h>

using namespace mira;


class MainWindow : public QMainWindow
{
public:

	MainWindow()
	{
		editor = new TapeEditor(this);
		setCentralWidget(editor);

		QToolBar* toolbar = addToolBar("Main Toolbar");
		QMenuBar* menu = menuBar();
		QMenu* m;
		m = menu->addMenu("&Tape");
		QAction* a;
		a = editor->getOpenAction();
		a->setText("&Open ...");
		a->setShortcut(QKeySequence("Ctrl+O"));
		m->addAction(a);
		toolbar->addAction(a);

		a = editor->getSaveAction();
		a->setText("&Save As ...");
		a->setShortcut(QKeySequence("Ctrl+S"));
		m->addAction(a);
		toolbar->addAction(a);

		m->addSeparator();
		a = m->addAction("E&xit");
		connect(a, SIGNAL(triggered()), this, SLOT(close()));


		m = menu->addMenu("&Edit");
		toolbar->addSeparator();

		a = editor->getUndoAction();
		a->setShortcut(QKeySequence("Ctrl+Z"));
		m->addAction(a);
		toolbar->addAction(a);

		a = editor->getRedoAction();
		a->setShortcut(QKeySequence("Ctrl+Y"));
		m->addAction(a);
		toolbar->addAction(a);

		m->addSeparator();
		toolbar->addSeparator();

		a = editor->getEraseMessagesAction();
		a->setShortcut(QKeySequence("Del"));
		m->addAction(a);
		toolbar->addAction(a);

		a = editor->getEraseTimeESAction();
		a->setShortcut(QKeySequence("Backspace"));
		m->addAction(a);
		toolbar->addAction(a);

		a = editor->getEraseTimeSEAction();
		a->setShortcut(QKeySequence("Shift+Backspace"));
		m->addAction(a);
		toolbar->addAction(a);

		m->addSeparator();
		a = editor->getTrimGapsAction();
		m->addAction(a);
	}


	TapeEditor* editor;
};

void printUsage()
{
	std::cout << "\n" <<
	"The mira tape editor is a graphical tool for editing, modifying and merging\n"
	"tape files.\n"
	"In order to merge tape files, you can load several of them at once.\n"
	"The tape file(s) can be specified optionally as command line parameters.\n\n"
	"Usage:\n"
	"      mirataped [tapefile] ([tapefile2] ... )\n"
	<< std::endl;
}

int main(int argc, char** argv)
{
	FrameworkWithGui::Application qapp(argc,argv);

	// initializing the framework, initializes the classfactory and allows us to use codecs, etc.
	Framework fw(argc,argv);


	// parse command line parameters
	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();
	if ( vmap.count("help") ) {
		printUsage();
		return 0;
	}

	// collect the passed file names
	QStringList fileList;
	// use the filenames that the Framework parses as "config files"
	if ( vmap.count("config-file") ) {
		std::vector<std::string> files = vmap["config-file"].as< std::vector<std::string>>();
		foreach(const std::string& f, files)
			fileList.append(QString::fromStdString(f));
	}

	MainWindow* window = new MainWindow;
	window->showMaximized();

	if(!fileList.empty())
		window->editor->openFiles(fileList);

	int res = qapp.exec();
	delete window;

	return res;
}
