/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeCommand.C
 *    Description.
 *
 * @author Tim Langner
 * @date   2012/02/03
 */

#include <TapeCommand.h>
#include <TapeEditor.h>

#include <QProgressDialog>

namespace mira {


///////////////////////////////////////////////////////////////////////////////

TapeCommand::TapeCommand(TapeEditor* editor) : mEditor(editor), mProgress(NULL)
{
}

TapeCommand::~TapeCommand()
{
	delete mProgress;
}

void TapeCommand::showProgress(int value, int maximum)
{
	if(mProgress==NULL) {
		mProgress = new QProgressDialog(mEditor);
		mProgress->setWindowModality(Qt::WindowModal);
		mProgress->setCancelButton(0);
		mProgress->setWindowTitle(" ");
		mProgress->setAutoReset(false);
		mProgress->setLabelText("Processing ...");
	}

	if(maximum>=0)
		mProgress->setMaximum(maximum);

	mProgress->setValue(value);

	// delete dialog, if value reached max. (this is more than "auto reset", which hides the dialog only)
	if(mProgress->value()>=mProgress->maximum()) {
		delete mProgress;
		mProgress=NULL;
	}
}

void TapeCommand::closeProgress()
{
	delete mProgress;
}


///////////////////////////////////////////////////////////////////////////////

void CreateChannelCommand::exec()
{
	mEditor->addChannel(mChannel, mType, TypeMetaPtr());
	mEditor->updateContents(true);
}

void CreateChannelCommand::undo()
{
	mEditor->getChannels().erase(mChannel);
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void DeleteChannelCommand::exec()
{
	mInfo = mEditor->getChannels()[mChannel];
	mEditor->getChannels().erase(mChannel);
	mEditor->updateContents(true);
}

void DeleteChannelCommand::undo()
{
	mEditor->getChannels()[mChannel] = mInfo;
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void RenameChannelCommand::exec()
{
	TapeChannelInfo& info = mEditor->getChannels()[mOldChannel];
	info.name = mNewChannel;
	mEditor->getChannels()[mNewChannel] = info;
	mEditor->getChannels().erase(mOldChannel);
	mEditor->updateContents(true);
}

void RenameChannelCommand::undo()
{
	TapeChannelInfo& info = mEditor->getChannels()[mNewChannel];
	info.name = mOldChannel;
	mEditor->getChannels()[mOldChannel] = info;
	mEditor->getChannels().erase(mNewChannel);
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void AddMessageCommand::exec()
{
	mMessage = mEditor->getChannels()[mChannel].data[mTime] = mMessage;
	mEditor->updateContents(true);
}

void AddMessageCommand::undo()
{
	mEditor->getChannels()[mChannel].data.erase(mTime);
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void EditMessageCommand::exec()
{
	if (mNewTime == mOldTime)
		mEditor->getChannels()[mChannel].data[mNewTime] = mNewMessage;
	else
	{
		mEditor->getChannels()[mChannel].data.insert(std::make_pair(mNewTime, mNewMessage));
		mEditor->getChannels()[mChannel].data.erase(mOldTime);
	}
	mEditor->updateContents(true);
}

void EditMessageCommand::undo()
{
	if (mNewTime == mOldTime)
		mEditor->getChannels()[mChannel].data[mOldTime] = mOldMessage;
	else
	{
		mEditor->getChannels()[mChannel].data.insert(std::make_pair(mOldTime, mOldMessage));
		mEditor->getChannels()[mChannel].data.erase(mNewTime);
	}
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void EraseMessageCommand::exec()
{
	mMessage = mEditor->getChannels()[mChannel].data[mTime];
	mEditor->getChannels()[mChannel].data.erase(mTime);
	mEditor->updateContents(true);
}

void EraseMessageCommand::undo()
{
	mEditor->getChannels()[mChannel].data[mTime] = mMessage;
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void EraseMessagesCommand::exec()
{
	mMessages.clear();
	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		if (!channel.second.selected)
			continue;
		auto iStart = channel.second.data.lower_bound(mStart);
		auto iEnd = channel.second.data.upper_bound(mEnd);
		mMessages[channel.first].data.insert(iStart, iEnd);
		channel.second.data.erase(iStart, iEnd);
	}
	mEditor->updateContents(true);
}

void EraseMessagesCommand::undo()
{
	foreach(TapeChannelInfoMap::value_type& channel, mMessages)
	{
		mEditor->getChannels()[channel.first].data.insert(channel.second.data.begin(),
		                                                  channel.second.data.end());
		mEditor->getChannels()[channel.first].selected = true;
	}
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void EraseTimeEndToStartCommand::exec()
{
	mMessages.clear();

	int64 erasedDuration = mEnd - mStart;

	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		if (!channel.second.selected)
			continue;
		auto iStart = channel.second.data.lower_bound(mStart);
		auto iEnd = channel.second.data.upper_bound(mEnd);
		mMessages[channel.first].data.insert(iStart, iEnd);
		channel.second.data.erase(iStart, iEnd);
		for ( ; iEnd != channel.second.data.end();)
		{
			channel.second.data.insert(std::make_pair(iEnd->first - erasedDuration, iEnd->second));
			auto tmp = iEnd;
			++iEnd;
			channel.second.data.erase(tmp);
		}
	}
	mEditor->setSelectionEnd(mStart);
	mEditor->updateContents(true);
}

void EraseTimeEndToStartCommand::undo()
{
	int64 erasedDuration = mEnd - mStart;

	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, mMessages)
	{
		auto& c = mEditor->getChannels()[channel.first];
		auto iStart = c.data.lower_bound(mStart);
		TapeChannelInfo::DataMap tmpMap;
		auto it = iStart;
		for ( ; it != c.data.end(); ++it)
			tmpMap.insert(std::make_pair(it->first + erasedDuration, it->second));
		c.data.erase(iStart, c.data.end());
		c.data.insert(channel.second.data.begin(), channel.second.data.end());
		c.data.insert(tmpMap.begin(), tmpMap.end());
		c.selected = true;
	}
	mEditor->setSelectionStart(mStart);
	mEditor->setSelectionEnd(mEnd);
	mEditor->updateContents(true);
}

//////////////////////////////////////////////////////////////////////////////

void EraseTimeStartToEndCommand::exec()
{
	mMessages.clear();

	int64 erasedDuration = mEnd - mStart;

	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		if (!channel.second.selected)
			continue;
		auto iStart = channel.second.data.lower_bound(mStart);
		auto iEnd = channel.second.data.upper_bound(mEnd);

		TapeChannelInfo::DataMap tmpMap;
		auto it = channel.second.data.begin();
		for ( ; it != iStart; ++it)
			tmpMap.insert(std::make_pair(it->first + erasedDuration, it->second));
		mMessages[channel.first].data.insert(channel.second.data.begin(), iEnd);
		channel.second.data.erase(channel.second.data.begin(), iEnd);
		channel.second.data.insert(tmpMap.begin(), tmpMap.end());
	}
	mEditor->setSelectionStart(mEnd);
	mEditor->updateContents(true);
}

void EraseTimeStartToEndCommand::undo()
{
	int64 erasedDuration = mEnd - mStart;

	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, mMessages)
	{
		auto& c = mEditor->getChannels()[channel.first];
		auto iEnd = c.data.upper_bound(mEnd);
		auto it = c.data.begin();
		for ( ; it != iEnd;)
		{
			c.data.insert(std::make_pair(it->first - erasedDuration, it->second));
			auto tmp = it;
			++it;
			channel.second.data.erase(tmp);
		}
		c.data.insert(channel.second.data.begin(), channel.second.data.end());
		c.selected = true;
	}
	mEditor->setSelectionStart(mStart);
	mEditor->setSelectionEnd(mEnd);
	mEditor->updateContents(true);
}

///////////////////////////////////////////////////////////////////////////////

void TrimGapsCommand::exec()
{
	mMessages.clear();
	TapeChannelInfoMap& channels = mEditor->getChannels();

	auto masterIt = channels.find(mChannelID);
	if(masterIt==channels.end())
		return; // no such channel, abort

	// prepare for undo
	foreach(TapeChannelInfoMap::value_type& channel, channels)
		mMessages[channel.first].data = channel.second.data;

	int64 maxGap_2 = mMaxGap/2;

	auto firstIt = masterIt->second.data.lower_bound(0);

	// loop through the messages of the master channel and detect gaps
	int64 gapStart=firstIt->first;

	int i=1;
	showProgress(0,masterIt->second.data.size());
	//for(auto it=masterIt->second.data.begin();it!=masterIt->second.data.end();++it,++i)
	for(auto it=firstIt;it!=masterIt->second.data.end();++it,++i)
	{
		showProgress(i);
		int64 gap = it->first - gapStart;

		if(gap>mMaxGap) {
			// found a gap
			int64 start = gapStart+maxGap_2;
			int64 end=it->first-maxGap_2;
			int64 erasedDuration = end-start;

			foreach(TapeChannelInfoMap::value_type& channel, channels)
			{
				auto itStart = channel.second.data.lower_bound(start);
				auto itEnd = channel.second.data.upper_bound(end);
				channel.second.data.erase(itStart, itEnd);

				// move the data forward
				for ( ; itEnd != channel.second.data.end();)
				{
					channel.second.data.insert(std::make_pair(itEnd->first - erasedDuration, itEnd->second));
					auto tmp = itEnd;
					++itEnd;
					channel.second.data.erase(tmp);
				}

			}

			// restore iterator to continue
			it = masterIt->second.data.upper_bound(gapStart+maxGap_2);
		}
		gapStart=it->first;
	}
	closeProgress();

	mEditor->updateContents(true);
}

void TrimGapsCommand::undo()
{
	foreach(TapeChannelInfoMap::value_type& channel, mMessages)
	{
		mEditor->getChannels()[channel.first].data = channel.second.data;
		mEditor->getChannels()[channel.first].selected = true;
	}
	mEditor->updateContents(true);

}


///////////////////////////////////////////////////////////////////////////////

} // namespace

