/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TypedTapeDataRenderer.h
 *    .
 *
 * @author Tim Langner
 * @date   2012/01/08
 */

#ifndef _MIRA_TYPEDTAPEDATARENDERER_H_
#define _MIRA_TYPEDTAPEDATARENDERER_H_

#include <QMenu>
#include <QAction>
#include <QFileDialog>

#include <widgets/MultiInputDialog.h>

#include <TapeDataRenderer.h>
#include <TapeEditor.h>
#include <TapeCommand.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class TypedTapeDataRenderer : public TapeDataRenderer
{
public:

	TypedTapeDataRenderer()
	{
		mFeatures = INFO | CREATE | EDIT;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, TapeDataRenderer);
		r.property("Member", mMember, "Member to render");
	}

	virtual void fillContextMenu(QMenu* menu, TapeChannelInfo& info,
	                             TapeChannelInfo::DataMap::iterator& message,
	                             int64 time);
	virtual void executeAction(QAction* action, TapeChannelInfo& info,
	                           TapeChannelInfo::DataMap::iterator& message,
	                           int64 time);

	virtual Typename getDatatype() { return typeName<T>(); }

	virtual QString getMessageInfo(TapeChannelInfo& info,
	                               TapeChannelInfo::DataMap::iterator& message);

	virtual void addMessage(TapeChannelInfo& info, int64 time);

	virtual void editMessage(TapeChannelInfo& info,
	                         TapeChannelInfo::DataMap::iterator& message);

protected:

	std::string mMember;
};

///////////////////////////////////////////////////////////////////////////////

template<typename T>
void TypedTapeDataRenderer<T>::fillContextMenu(QMenu* menu, TapeChannelInfo& info,
                                               TapeChannelInfo::DataMap::iterator& message,
                                               int64 time)
{
	TapeDataRenderer::fillContextMenu(menu, info, message, time);
	menu->addAction("Export all messages as CSV");
	int64 selected = mEditor->getSelectionEnd() - mEditor->getSelectionStart();
	menu->addAction("Export selected messages as CSV")->setEnabled(selected > 0);
}

template<typename T>
void TypedTapeDataRenderer<T>::executeAction(QAction* action, TapeChannelInfo& info,
                                      TapeChannelInfo::DataMap::iterator& message,
                                      int64 time)
{
	auto iStart = info.data.begin();
	auto iEnd = info.data.end();

	if (action->text() == "Export all messages as CSV" ||
		action->text() == "Export selected messages as CSV")
	{
		QString fileName = QFileDialog::getSaveFileName(mEditor, "Filename", "", "CSV file (*.csv)");
		if (fileName.isEmpty())
			return;
		if (action->text() == "Export selected messages as CSV")
		{
			iStart = info.data.lower_bound(mEditor->getSelectionStart());
			iEnd = info.data.upper_bound(mEditor->getSelectionEnd());
		}
		std::ofstream output(fileName.toStdString().c_str());
		for ( ; iStart != iEnd; ++iStart)
		{
			T data;
			BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&iStart->second.getData()));
			bs.deserialize(data, false);
			JSONSerializer js;
			json::Value json = js.serialize(data);
			if (!mMember.empty())
				json = json::getElement(json, mMember);
			output << iStart->first << ";" << iStart->second.getFrameID() << ";"
					<< iStart->second.getSequenceID() << ";" << json::write(json) << std::endl;
		}
		output.close();
	}
	else
		TapeDataRenderer::executeAction(action, info, message, time);
}


template<typename T>
QString TypedTapeDataRenderer<T>::getMessageInfo(TapeChannelInfo& info,
                                                 TapeChannelInfo::DataMap::iterator& message)
{
	try
	{
		T data;
		BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
		bs.deserialize(data, false);
		JSONSerializer js;
		json::Value json = js.serialize(data);
		if (!mMember.empty())
			json = json::getElement(json, mMember);
		return json::write(json, true).c_str();
	}
	catch(Exception& ex)
	{
		return ex.message().c_str();
	}
	return "";
}

template<typename T>
void TypedTapeDataRenderer<T>::addMessage(TapeChannelInfo& info, int64 time)
{
	MultiInputDialog dialog("Creating message");
	dialog.addInputField("time", "Time offset us", QString("%1").arg(time));
	dialog.addInputField("frame", "Frame ID", "");
	dialog.addInputField("sequence", "Sequence ID", "0");
	std::string defaultData = MakeString() << print(T());
	dialog.addInputField("data", "Data", defaultData.c_str(), true);
	if (dialog.exec() == 0)
		return;

	time = fromString<int64>(dialog.getInput("time").toStdString());
	TapeChannelMessage message(true);
	message.setFrameID(dialog.getInput("frame").toStdString());
	message.setSequenceID(dialog.getInput("sequence").toInt());
	json::Value v;
	T data;
	json::read(dialog.getInput("data").toStdString(), v);
	JSONDeserializer js(v);
	js.deserialize(data);
	BinaryBufferSerializer bs(const_cast<Buffer<uint8>*>(&message.getData()));
	bs.serialize(data, false);
	mEditor->executeCommand(new AddMessageCommand(info.name, time, message, mEditor));
}

template<typename T>
void TypedTapeDataRenderer<T>::editMessage(TapeChannelInfo& info,
                                           TapeChannelInfo::DataMap::iterator& message)
{
	try
	{
		// deserialize the stamped message
		T data;
		BinaryBufferDeserializer bds(const_cast<Buffer<uint8>*>(&message->second.getData()));
		bds.deserialize(data, false);
		// serialize the data into json
		JSONSerializer js;
		json::Value json = js.serialize(data);
		std::string jsonStr = json::write(json, true);

		// create an input dialog and fill it with the data of the message
		MultiInputDialog dialog("Editing message");
		dialog.addInputField("time", "Time offset us", QString("%1").arg(message->first));
		dialog.addInputField("frame", "Frame ID", message->second.getFrameID().c_str());
		dialog.addInputField("sequence", "SequenceID", QString("%1").arg(message->second.getSequenceID()));
		dialog.addInputField("data", "Data", jsonStr.c_str(), true);
		if (dialog.exec() == 0)
			return;

		int64 time = fromString<int64>(dialog.getInput("time").toStdString());
		std::string newFrameID = dialog.getInput("frame").toStdString();
		uint32 newSequenceID = dialog.getInput("sequence").toInt();
		std::string newJsonStr = dialog.getInput("data").toStdString();
		// did the user modify anything? if not nothing changed
		if (time == message->first &&
			newFrameID == message->second.getFrameID() &&
			newSequenceID == message->second.getSequenceID() &&
			newJsonStr == jsonStr)
			return;

		TapeChannelMessage newMessage(true);
		newMessage.setFrameID(newFrameID);
		newMessage.setSequenceID(newSequenceID);
		// deserialize the edited data back into the data
		T newData;
		json::Value v;
		json::read(newJsonStr, v);
		JSONDeserializer jds(v);
		jds.deserialize(newData);
		BinaryBufferSerializer bs(const_cast<Buffer<uint8>*>(&newMessage.getData()));
		bs.serialize(newData, false);
		mEditor->executeCommand(new EditMessageCommand(info.name,
		                                               message->first, message->second,
		                                               time, newMessage,
		                                               mEditor));
	}
	catch(Exception&)
	{
	}
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
