/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeCommand.h
 *    .
 *
 * @author Tim Langner
 * @date   2012/02/03
 */

#ifndef _MIRA_TAPECOMMAND_H_
#define _MIRA_TAPECOMMAND_H_

#include <TapeChannelInfo.h>


class QProgressDialog;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class TapeEditor;

class TapeCommand
{
public:

	TapeCommand(TapeEditor* editor);
	virtual ~TapeCommand();

	virtual void exec() = 0;
	virtual void undo() = 0;

protected:

	void showProgress(int value, int maximum=-1);
	void closeProgress();

protected:

	TapeEditor* mEditor;
	QProgressDialog* mProgress;
};

///////////////////////////////////////////////////////////////////////////////

class CreateChannelCommand : public TapeCommand
{
public:
	CreateChannelCommand(const std::string& channelID, const std::string& type,
	                     TapeEditor* editor) :
		TapeCommand(editor),
		mChannel(channelID),
		mType(type) {}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannel;
	std::string mType;
};

//////////////////////////////////////////////////////////////////////////////

class DeleteChannelCommand : public TapeCommand
{
public:
	DeleteChannelCommand(const std::string& channelID, TapeEditor* editor) :
		TapeCommand(editor),
		mChannel(channelID) {}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannel;
	TapeChannelInfo mInfo;
};


//////////////////////////////////////////////////////////////////////////////

class RenameChannelCommand : public TapeCommand
{
public:
	RenameChannelCommand(const std::string& oldChannelID,
	                     const std::string& newChannelID,
	                     TapeEditor* editor) :
		TapeCommand(editor),
		mOldChannel(oldChannelID),
		mNewChannel(newChannelID) {}

	virtual void exec();
	virtual void undo();

protected:
	std::string mOldChannel;
	std::string mNewChannel;
};

//////////////////////////////////////////////////////////////////////////////

class AddMessageCommand : public TapeCommand
{
public:
	AddMessageCommand(const std::string& channelID,
	                  int64 time,
	                  const TapeChannelMessage& message,
	                  TapeEditor* editor) :
		TapeCommand(editor),
		mChannel(channelID),
		mTime(time),
		mMessage(message)
	{}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannel;
	int64 mTime;
	TapeChannelMessage mMessage;
};

//////////////////////////////////////////////////////////////////////////////

class EditMessageCommand : public TapeCommand
{
public:
	EditMessageCommand(const std::string& channelID,
	                  int64 oldTime,
	                  const TapeChannelMessage& oldMessage,
	                  int64 newTime,
	                  const TapeChannelMessage& newMessage,
	                  TapeEditor* editor) :
		TapeCommand(editor),
		mChannel(channelID),
		mOldTime(oldTime),
		mOldMessage(oldMessage),
		mNewTime(newTime),
		mNewMessage(newMessage)
	{}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannel;
	int64 mOldTime;
	TapeChannelMessage mOldMessage;
	int64 mNewTime;
	TapeChannelMessage mNewMessage;
};

//////////////////////////////////////////////////////////////////////////////

class EraseMessageCommand : public TapeCommand
{
public:
	EraseMessageCommand(const std::string& channelID,
	                    int64 time,
	                    TapeEditor* editor) :
		TapeCommand(editor),
		mChannel(channelID),
		mTime(time) {}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannel;
	int64 mTime;
	TapeChannelMessage mMessage;
};

//////////////////////////////////////////////////////////////////////////////

class EraseMessagesCommand : public TapeCommand
{
public:
	EraseMessagesCommand(int64 start,
	                     int64 end,
	                     TapeEditor* editor) :
		TapeCommand(editor),
		mStart(start),
		mEnd(end) {}

	virtual void exec();
	virtual void undo();

protected:
	int64 mStart;
	int64 mEnd;
	TapeChannelInfoMap mMessages;
};

//////////////////////////////////////////////////////////////////////////////

class EraseTimeEndToStartCommand : public TapeCommand
{
public:
	EraseTimeEndToStartCommand(int64 start,
	                           int64 end,
	                           TapeEditor* editor) :
		TapeCommand(editor),
		mStart(start),
		mEnd(end) {}

	virtual void exec();
	virtual void undo();

protected:
	int64 mStart;
	int64 mEnd;
	TapeChannelInfoMap mMessages;
};

//////////////////////////////////////////////////////////////////////////////

class EraseTimeStartToEndCommand : public TapeCommand
{
public:
	EraseTimeStartToEndCommand(int64 start,
	                           int64 end,
	                           TapeEditor* editor) :
		TapeCommand(editor),
		mStart(start),
		mEnd(end) {}

	virtual void exec();
	virtual void undo();

protected:
	int64 mStart;
	int64 mEnd;
	TapeChannelInfoMap mMessages;
};

//////////////////////////////////////////////////////////////////////////////

class TrimGapsCommand : public TapeCommand
{
public:
	TrimGapsCommand(const std::string& channelID,
	                int64 maxGap,
	                TapeEditor* editor) :
		TapeCommand(editor),
		mChannelID(channelID), mMaxGap(maxGap) {}

	virtual void exec();
	virtual void undo();

protected:
	std::string mChannelID;
	int64 mMaxGap;
	TapeChannelInfoMap mMessages;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
