/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AddVersionTest.C
 *    Test cases for miratape addversion.
 *
 * @author Christof Schroeter
 * @date   2018/05/15
 */

#include <boost/test/unit_test.hpp>

#include <math/UniformRandomGenerator.h>
#include <platform/Process.h>
#include <utils/PathFinder.h>
#include <utils/Stamped.h>

#include <serialization/MetaSerializer.h>

#include <fw/Tape.h>

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/pair>

namespace mira { namespace tapetest {

////////////////////////////////////////////////////////////////////////////////

bool gEnableVersion;

class ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		if (gEnableVersion) {        // we have written this without version and
			                         // check reading it with and without version
			int v = r.version(3, this);
			BOOST_CHECK_EQUAL(v, 2); // because we will set the version to 2 in
			                         // the tape after it is written
		}

		r.member("Int", mInt, "");
		r.member("Float", mFloat, "");
		r.member("Int8", mInt8, "");
		r.member("Bool", mBool, "");
	}

	bool operator==(const ClassA& other) const
	{
		return (mInt == other.mInt) &&
		       (std::abs(mFloat  - other.mFloat) < 0.0001f) &&
		       (mInt8 == other.mInt8) &&
		       (mBool == other.mBool);
	}

public:

	int mInt;
	float mFloat;
	uint8 mInt8;
	bool mBool;
};

std::ostream& operator<<(std::ostream& stream, const ClassA& o)
{
	return stream << "[ " << o.mInt << ", " << o.mFloat << ", "
	                      << o.mInt8 << ", " << (o.mBool?"true":"false") << " ]";
}
////////////////////////////////////////////////////////////////////////////////

void checkTape(const Tape& tape, std::vector<std::string> channelnames, int messages)
{
	Tape::ChannelMap channels = tape.getChannels();
	BOOST_CHECK_EQUAL(channels.size(), channelnames.size());
	foreach(auto c, channelnames)
		BOOST_CHECK_EQUAL(channels[c].messages.size(), messages);
}

template <typename T>
void writeObject(Tape* tape, const T& object,
                 const std::string& channel, MetaTypeDatabase& metaDB)
{
	static MetaSerializer metaSerializer(metaDB);

	Stamped<T> o(object, Time::now(), "");
	TypeMetaPtr meta = metaSerializer.addMeta(object);
	tape->write(channel, o, false, meta, metaDB);
}

template <typename T>
void readMessage(Tape* tape, const Tape::MessageIndex& index, Stamped<T>& oValue)
{
	Buffer<uint8> buffer;
	Duration time;
	tape->readMessage(index, oValue.frameID, oValue.sequenceID, buffer, time);
	oValue.timestamp = tape->getStartTime() + time;

	BinaryBufferDeserializer d(&buffer);
	d.deserialize(oValue.internalValueRep(), false);
}

template <typename T>
void readMessage(Tape* tape, const std::string& channel, int index, Stamped<T>& oValue)
{
	Tape::ChannelMap channels = tape->getChannels();
	readMessage(tape, channels[channel].messages[index], oValue);
}

uint8 getFormatVersion(Tape* tape, const Tape::MessageIndex& index)
{
	Buffer<uint8> buffer;
	Duration time;
	std::string frameID;
	uint32 sequenceID;
	tape->readMessage(index, frameID, sequenceID, buffer, time);

	BinaryBufferIstream stream(&buffer);
	return BinaryBufferDeserializer::getDataFormatVersion(stream);
}

uint8 getFormatVersion(Tape* tape, const std::string& channel, int index)
{
	Tape::ChannelMap channels = tape->getChannels();
	return getFormatVersion(tape, channels[channel].messages[index]);
}

////////////////////////////////////////////////////////////////////////////////

/// add version to the class that is the channel type
void addVersionToSimpleClass(const Path& tapename, uint8 formatversion)
{
	UniformRandomGenerator<int> rndI(1, 255);
	rndI.seed(123456);
	UniformRandomGenerator<float> rndF(1.0f, 255.0f);
	rndI.seed(654321);

	std::vector<ClassA> objectsA;

	for (int n = 0; n < 10; ++n) {
		ClassA objectA;
		objectA.mInt = rndI();
		objectA.mFloat = rndF();
		objectA.mInt8 = rndI();
		objectA.mBool = rndI() % 2;

		objectsA.push_back(objectA);
	}

	// ClassA does not have a version in the tape. We can read it if we do not expect a version.
	gEnableVersion = false;

	{
		Stamped<ClassA> o;

		Tape t;
		t.open(tapename, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		for (int n = 0; n < 10; ++n) {
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelA", n), formatversion);
			readMessage(&t, "ChannelA", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsA[n]);
		}
	}

	// Now we expect a version in ClassA from the tape (we added one to ClassA's definition).
	// Reading the original tape should fail.
	gEnableVersion = true;

	{
		Stamped<ClassA> o;

		Tape t;
		t.open(tapename, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		try {
			for (int n = 0; n < 10; ++n) {
				readMessage(&t, "ChannelA", n, o);

				// it's ok to fail reading with an XIO
				// if, however, reading succeeds, the result must not be what was written!
				BOOST_CHECK(!(o.value() == objectsA[n]));
			}
		}
		catch(XIO& ex) {}
	}

	// add the version number to all occurrences of ClassA in the tape
	executeProcess("miratape copy " + tapename.string() + " addedversion.tape"
	               " --addversion mira::tapetest::ClassA=2");

	// After manipulating the tape, we can read it with the new definition of ClassA.
	{
		Stamped<ClassA> o;

		Tape t;
		t.open("addedversion.tape", Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelA"}), 10);

		for (int n = 0; n < 10; ++n) {
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelA", n), formatversion);
			readMessage(&t, "ChannelA", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsA[n]);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(SimpleAddVersionTest)
{
	// the test tape is pre-created, because we cannot (do not want to) write v0 tapes anymore
	// the test method makes sure random numbers are in sync with the tape by using specific seeds!
	addVersionToSimpleClass(findProjectFile("tools/miratape/tests/etc/simple.v0.tape"), 0);
}

////////////////////////////////////////////////////////////////////////////////

class ClassB : public ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Int16", mInt16, "");
		MIRA_REFLECT_BASE(r, ClassA);
	}

	bool operator==(const ClassB& other) const
	{
		return (mInt16 == other.mInt16) && ClassA::operator==(other);
	}

public:

	int16 mInt16;
};

std::ostream& operator<<(std::ostream& stream, const ClassB& o)
{
	return stream << "[ " << (ClassA)o << ", " << o.mInt16 << " ]";
}

class ClassC : public ClassA
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, ClassA);
		r.member("Int64", mInt64, "");
	}

	bool operator==(const ClassC& other) const
	{
		return (mInt64 == other.mInt64) && ClassA::operator==(other);
	}

public:

	int64 mInt64;
};

std::ostream& operator<<(std::ostream& stream, const ClassC& o)
{
	return stream << "[ " << (ClassA)o << ", " << o.mInt64 << " ]";
}

class ClassD
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Pair", mPair, "");
	}

	bool operator==(const ClassD& other) const
	{
		return (mPair == other.mPair);
	}

public:
	std::pair<ClassB, ClassC> mPair;
};

std::ostream& operator<<(std::ostream& stream, const ClassD& o)
{
	return stream << print(o.mPair);
}

class ClassE
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Vector", mVector, "");
	}

	bool operator==(const ClassE& other) const
	{
		return (mVector == other.mVector);
	}

public:
	std::vector<ClassD> mVector;
};

std::ostream& operator<<(std::ostream& stream, const ClassE& o)
{
	return stream << print(o.mVector);
}

////////////////////////////////////////////////////////////////////////////////

// add version to a class that is a part of an arbitrarily complex structured channel type
void addVersionToComplexClass(const Path& tapename, uint8 formatversion)
{
	UniformRandomGenerator<int> rndI(1, 255);
	rndI.seed(123456);
	UniformRandomGenerator<float> rndF(1.0f, 255.0f);
	rndI.seed(654321);

	std::vector<ClassE> objectsE;

	for (int n = 0; n < 10; ++n) {
		ClassE objectE;

		int l = rndI();
		for (int i = 0; i < l; ++i) {
			ClassD objectD;
			objectD.mPair.first.mInt = rndI();
			objectD.mPair.first.mFloat = rndF();
			objectD.mPair.first.mInt8 = rndI();
			objectD.mPair.first.mBool = rndI() % 2;
			objectD.mPair.first.mInt16 = rndI();
			objectD.mPair.second.mInt = rndI();
			objectD.mPair.second.mFloat = rndF();
			objectD.mPair.second.mInt8 = rndI();
			objectD.mPair.second.mBool = rndI() % 2;
			objectD.mPair.second.mInt64 = rndI();

			objectE.mVector.push_back(objectD);
		}

		objectsE.push_back(objectE);
	}

	gEnableVersion = false;

	{
		Stamped<ClassE> o;

		Tape t;
		t.open(tapename, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		for (int n = 0; n < 5; ++n) {
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE1", n), formatversion);
			readMessage(&t, "ChannelE1", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);

			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE2", n), formatversion);
			readMessage(&t, "ChannelE2", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n+5]);
		}
	}

	gEnableVersion = true;

	{
		Stamped<ClassE> o;

		Tape t;
		t.open(tapename, Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		try {
			for (int n = 0; n < 5; ++n) {
				readMessage(&t, "ChannelE1", n, o);

				// it's ok to fail reading with an XIO
				// if, however, reading succeeds, the result must not be what was written!
				BOOST_CHECK(!(o.value() == objectsE[n]));

				readMessage(&t, "ChannelE2", n, o);
				BOOST_CHECK(!(o.value() == objectsE[n+5]));
			}

		}
		catch(XIO& ex) {}
	}

	// add the version number to all occurrences of ClassB and ClassC in the tape
	// (both inherit the version member from ClassA)
	executeProcess("miratape copy " + tapename.string() + " addedversion.tape.tmp"
	               " --addversion mira::tapetest::ClassB=2,1");
	executeProcess("miratape copy addedversion.tape.tmp addedversion.tape"
	               " --addversion mira::tapetest::ClassC=2");

	{
		Stamped<ClassE> o;

		Tape t;
		t.open("addedversion.tape", Tape::READ);
		checkTape(t, std::vector<std::string>({"ChannelE1", "ChannelE2"}), 5);

		for (int n = 0; n < 5; ++n) {
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE1", n), formatversion);
			readMessage(&t, "ChannelE1", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n]);

			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE2", n), formatversion);
			readMessage(&t, "ChannelE2", n, o);
			BOOST_CHECK_EQUAL(o.value(), objectsE[n+5]);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(ComplexAddVersionTest)
{
	// the test tape is pre-created, because we cannot (do not want to) write v0 tapes anymore
	// the test method makes sure random numbers are in sync with the tape by using specific seeds!
	addVersionToComplexClass(findProjectFile("tools/miratape/tests/etc/complex.v0.tape"), 0);
}

////////////////////////////////////////////////////////////////////////////////

// add format version to start of serialized data
BOOST_AUTO_TEST_CASE(AddFormatVersionTest)
{
	{
		Path tapename = findProjectFile("tools/miratape/tests/etc/simple.v0.tape");

		executeProcess("miratape copy " + tapename.string() + " simple.v1.tape"
			           " --addformatversion 1");

		Tape t;
		t.open("simple.v1.tape", Tape::READ);

		for (int n = 0; n < 10; ++n)
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelA", n), 1);
	}
	{
		Path tapename = findProjectFile("tools/miratape/tests/etc/complex.v0.tape");

		executeProcess("miratape copy " + tapename.string() + " complex.v1.tape"
			           " --addformatversion 1");

		Tape t;
		t.open("complex.v1.tape", Tape::READ);

		for (int n = 0; n < 5; ++n) {
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE1", n), 1);
			BOOST_CHECK_EQUAL(getFormatVersion(&t, "ChannelE2", n), 1);
		}
	}

	addVersionToSimpleClass("simple.v1.tape", 1);
	addVersionToComplexClass("complex.v1.tape", 1);
}

////////////////////////////////////////////////////////////////////////////////

}}
