/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ExportDialog.h
 *    $Dialog for selecting packages for export$.
 *
 * @author Ronny Stricker
 * @date   2012/07/05
 */

#ifndef EXPORTDIALOG_H_
#define EXPORTDIALOG_H_

#include <ui_ExportDialog.h>

#include <QDialog>
#include <QLabel>
#include <QCheckBox>

#include "core/Package.h"
#include "core/Database.h"


namespace mira {

//////////////////////////////////////////////////////////////////////////////

class ExportDialog : public QDialog, protected Ui::ExportDialog
 {
     Q_OBJECT

public:
     ExportDialog( Database& ioDB, std::map<Path,std::string>& pathMap, QWidget *parent = 0)
     	 : QDialog( parent ), mDB( ioDB )
     {
    	 Ui::ExportDialog::setupUi(this);

    	 int line = 0;

    	 mTblRepos->setRowCount( ioDB.getRepositoryMap().size() );
    	 foreach( auto repo, ioDB.getRepositoryMap() ) {
    		 QCheckBox* tBox = new QCheckBox();
			 tBox->setCheckState( Qt::Checked );
			 connect( tBox, SIGNAL( toggled( bool ) ), this, SLOT( refreshPackageList() ) );
			 mTblRepos->setCellWidget(line,0, tBox );

			 QLabel* tLabel = new QLabel( QString::fromStdString( repo.first ) );
			 mTblRepos->setCellWidget(line,1, tLabel );

			 tLabel = new QLabel( QString::fromStdString( repo.second->url ) );
			 mTblRepos->setCellWidget(line,2, tLabel );

			 ++line;
    	 }

    	 // setup the path table
    	 line = 0;
    	 mTblPath->setRowCount( pathMap.size() );
    	 foreach( auto path, pathMap ) {
    		 QCheckBox* tBox = new QCheckBox();
			 tBox->setCheckState( Qt::Checked );
			 connect( tBox, SIGNAL( toggled( bool ) ), this, SLOT( refreshPackageList() ) );
			 mTblPath->setCellWidget(line,0, tBox );

			 QLabel* tLabel = new QLabel( QString::fromStdString( path.first.string() ) );
			 mTblPath->setCellWidget(line,1, tLabel );

			 assert( path.second.find(']'));
			 std::string tPathId = path.second.substr(0,
					 path.second.find(']'));

			 mPathId.push_back( fromString<int>( tPathId ) );

			 ++line;
    	 }

    	 // setup the package table
    	 foreach( PackageGroup* group, mDB.getRootPackage()->mSubPackages ) {

			 Package* package = dynamic_cast<Package*>( group );
			 if ( package ) {
				 PackageElement elem;
				 elem.package = package;

				 assert( package->mLocalPath.string().find(']'));
				 std::string tPathId = package->mLocalPath.string().substr(0,
						 package->mLocalPath.string().find(']'));
				 elem.pathId = fromString<int>( tPathId );

				 mPkgList.push_back( elem );
			 }
    	 }

#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
    	 mTblRepos->horizontalHeader()->setSectionResizeMode(0,QHeaderView::ResizeToContents);
    	 mTblRepos->horizontalHeader()->setSectionResizeMode(1,QHeaderView::ResizeToContents);
    	 mTblRepos->horizontalHeader()->setSectionResizeMode(2,QHeaderView::Stretch);

    	 mTblPkgExport->horizontalHeader()->setSectionResizeMode(0,QHeaderView::ResizeToContents);
    	 mTblPkgExport->horizontalHeader()->setSectionResizeMode(1,QHeaderView::Stretch);

    	 for ( int i=2; i<mTblPkgExport->columnCount(); ++i )
    		 mTblPkgExport->horizontalHeader()->setSectionResizeMode(i,QHeaderView::ResizeToContents);

    	 mTblPath->horizontalHeader()->setSectionResizeMode(0,QHeaderView::ResizeToContents);
		 mTblPath->horizontalHeader()->setSectionResizeMode(1,QHeaderView::Stretch);
#else
    	 mTblRepos->horizontalHeader()->setResizeMode(0,QHeaderView::ResizeToContents);
    	 mTblRepos->horizontalHeader()->setResizeMode(1,QHeaderView::ResizeToContents);
    	 mTblRepos->horizontalHeader()->setResizeMode(2,QHeaderView::Stretch);

    	 mTblPkgExport->horizontalHeader()->setResizeMode(0,QHeaderView::ResizeToContents);
    	 mTblPkgExport->horizontalHeader()->setResizeMode(1,QHeaderView::Stretch);

    	 for ( int i=2; i<mTblPkgExport->columnCount(); ++i )
    		 mTblPkgExport->horizontalHeader()->setResizeMode(i,QHeaderView::ResizeToContents);

    	 mTblPath->horizontalHeader()->setResizeMode(0,QHeaderView::ResizeToContents);
		 mTblPath->horizontalHeader()->setResizeMode(1,QHeaderView::Stretch);
#endif
		line = 0;

		mTblPkgExport->setRowCount( mPkgList.size() );
		foreach( PackageElement& elem, mPkgList ) {

			 Package* package = elem.package;

			 QCheckBox* tBox = new QCheckBox();
			 tBox->setCheckState( Qt::Checked );
			 mTblPkgExport->setCellWidget(line,0, tBox );

			 QLabel* tLabel = new QLabel( QString::fromStdString( "<b>"+package->mName+"</b>" ) );
			 mTblPkgExport->setCellWidget(line,1, tLabel );

			 tLabel = new QLabel( QString::fromStdString( "<b> "+package->getTypeString()+" </b>" ) );
			 mTblPkgExport->setCellWidget(line,2, tLabel );

			 tLabel = new QLabel( QString::fromStdString( "<b> "+package->mVersion.str()+" </b>" ) );
			 mTblPkgExport->setCellWidget(line,3, tLabel );

			 tLabel = new QLabel( QString::fromStdString( " "+Package::getDevelStateString(*package)+" " ) );
			 mTblPkgExport->setCellWidget(line,4, tLabel );

			 ++line;
		 }
     }

     virtual ~ExportDialog()
     {

     }
     
public slots:
	virtual void accept()
	{
		for ( int row=0; row<int(mPkgList.size()); ++row ) {
			QCheckBox* tBox = dynamic_cast<QCheckBox*>( mTblPkgExport->cellWidget(row,0) );
			bool checked = tBox->checkState() == Qt::Checked;
			if ( mTblPkgExport->isRowHidden( row ) || !checked ) {
				mDB.removePackage( mPkgList[row].package );
			}
		}

		for ( int row=0; row<mTblRepos->rowCount(); ++row) {
			QCheckBox* tBox = dynamic_cast<QCheckBox*>( mTblRepos->cellWidget(row,0) );

			if ( tBox->checkState() != Qt::Checked ) {
				QLabel* tLabel = dynamic_cast<QLabel*>( mTblRepos->cellWidget(row,1) );
				assert( mDB.repos.find( tLabel->text().toStdString() )
						!= mDB.repos.end() );
				mDB.repos.erase( mDB.repos.find( tLabel->text().toStdString() ) );
			}
		}

		QDialog::accept();
	}

	void refreshPackageList()
	{
		std::map<int,bool> tPathMap;
		std::map<Url,bool> tRepoMap;

		// check which repos are enabled
		for ( int i=0; i<mTblRepos->rowCount(); ++i ) {
			QCheckBox* tBox = dynamic_cast<QCheckBox*>( mTblRepos->cellWidget(i,0) );
			bool checked = tBox->checkState() == Qt::Checked;

			QLabel* tLabel = dynamic_cast<QLabel*>( mTblRepos->cellWidget(i,2) );
			Url tUrl = tLabel->text().toStdString();

			tRepoMap[ tUrl ] = checked;
		}

		// check which paths are enabled
		for ( int i=0; i<mTblPath->rowCount(); ++i ) {
			QCheckBox* tBox = dynamic_cast<QCheckBox*>( mTblPath->cellWidget(i,0) );
			bool checked = tBox->checkState() == Qt::Checked;

			tPathMap[ mPathId[i] ] = checked;
		}

		// now we can hide/show the package items
		int column = 0;
		foreach(  PackageElement& elem, mPkgList ) {
			bool hidden = (tRepoMap.find(elem.package->mCurrentRepo) != tRepoMap.end()
							&& !tRepoMap[ elem.package->mCurrentRepo ]) ||
							(tPathMap.find(elem.pathId) != tPathMap.end()
									&& !tPathMap[elem.pathId] );
			mTblPkgExport->setRowHidden( column, hidden );
			++column;
		}
	}

protected:
	Database& mDB;

	struct PackageElement {
		Package* package;
		int pathId;
	};

	std::vector<PackageElement> mPkgList;
	std::vector<int> mPathId;	///< store path ids associated with the corresponding
								/// table widget item

 };

//////////////////////////////////////////////////////////////////////////////

}

#endif /* EXPORTDIALOG_H_ */
