/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAPackage.h
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */

#ifndef _MIRA_MIRAPACKAGE_H_
#define _MIRA_MIRAPACKAGE_H_

#include <core/Database.h>

#define DATABASEXML_FILENAME "mirapackage.xml"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Class implements the main functionality for package administration
 * and installation. Basically, this class consists of a database and provides
 * additional functions for package handling.
 */
class MIRAPackage
{
public:
	MIRAPackage(PromptProvider* promptProvider);

	virtual ~MIRAPackage();

	/**
	 * @brief Try to load database.
	 * If the given path is empty, the function tries to load the database
	 * from the users home directory.
	 */
	void load( std::string path = "");

	/**
	 * @brief Try to store the database.
	 * If the given path is empty, the function tries to save the database
	 * into the users home directory.
	 */
	void save( std::string path = "");

	/**
	 * @brief Export database
	 * See Database::exportDB for further infos.
	 * @param path Path of the export file
	 */
	void exportDB( std::string const& path );

	/**
	 * @brief Import database
	 * The repositories are imported first of all. Afterwards the function
	 * triggers reindexing of the Database. The packages contained in the
	 * MIRA Package Export file are added to the install plan subsequently.
	 * @param path Path of the file
	 */
	void importDB( std::string const& path );

	/**
	 * @brief Select the active MIRA_PATH from the corresponding environment variable.
	 * If more than one path exists, the functions calls selectMIRAPath() for selection
	 * purpose.
	 * @throws XLogical if no path can be found or if selection is invalid.
	 */
	virtual void resolveMIRAPath( bool forceSelection = false );

	/**
	 * @brief Remove all repositories from the database.
	 */
	void clearRepositories();

	/**
	 * @brief Add repository to database.
	 */
	void addRepository( RepositoryPtr repo );

	/**
	 * @brief Create repository and add it to the database.
	 */
	void addRepository(std::string const& iName,
		std::string const& iDescription,
		std::string const& iUrl,
		std::string const& iIndexFile,
		uint32 priority, std::string const& iType );

	/**
	 * @brief reindex all repositories.
	 */
	void reindexWebRepositories( bool thorough);

	virtual void reindexWebRepository(RepositoryPtr repo, bool thorough);

	virtual void checkForInstalled();

	void touchCmakeLists( Path const& path ) const;

	/**
	 * @brief Display status message
	 */
	virtual void statusMessage( std::string const& message );

	/**
	 * @brief Show status message with current progress
	 * @param message Verbose message for current progress step
	 * @param title Title of the whole process
	 * @param value progress value
	 * @param maximum maximum progress value
	 */
	virtual void statusProgress( std::string const& message,
	                             std::string const& title, int value, int maximum );

	void progressReceiver( std::string const& file, int step, int total );

	void addPackageToCheckoutPlan( Package* package, bool checkDependencies = true, bool preselectDependencies = true );

	void addPackageForRemoval( Package* package );

	PackageGroup* checkForPrimaryPackageError();

	/**
	 * @brief Return the given packages as sequence ordered by dependencies.
	 * The first package in the sequence will be the one with no dependencies to
	 * the other packages while the last one requires the most dependencies.
	 */
	std::vector<Package*> getInstallSequence( std::vector<Package*> const& packages );

	/**
	 * Same as above but only given the names of the desired packages
	 */
	std::vector<std::string> getInstallSequence( std::vector<std::string> const& packageNames );

	/**
	 * @brief Return the given packages as sequence ordered by dependencies.
	 * Applies getInstallSequence to the packages which should be installed, if
	 * the action equals INSTALL. The reversed package sequence is returned if
	 * the action equals UNINSTALL. A random sequence is returned for every
	 * other action
	 */
	std::vector<Package*> getPackageSequence( Database::Action action );

	/**
	 * @brief Determine the installation path for a package update action.
	 * @return The value for mLocalPath
	 */
	Path getPathForPackageUpdate(const Package* package) const;

	virtual void doIt();

	virtual void update();

	virtual void installPackages( std::vector<std::string> const& packageNames ) = 0;

	virtual void deinstallPackages( std::vector<std::string> const& packageNames ) = 0;

protected:
	virtual void loadContent(const XMLDom& content);
	virtual void saveContent(XMLDom& content);

	virtual std::string selectMIRAPath( std::string const& defaultValue = "" )
	{
		return defaultValue;
	}

	virtual void resolveDependencies( Package* depTree, bool preselect = true );

	virtual bool confirmCheckoutPlan() = 0;
	virtual bool confirmExportPackages( Database& ioDB, std::map<Path,std::string>& oPathMap ) = 0;
	virtual bool confirmUpdatePlan( std::vector<std::pair<Package*,Package*> >& updatePlan ) = 0;

	/**
	 * @brief confirm the dependency tree given in rootPackage.
	 * The install changes will be marked in the given database.
	 * @param rootPackage tree structure to confirm
	 * @param database install actions will be written to this database
	 * @return true if the dependencies are confirmed
	 */
	virtual bool confirmDependencies( Package* rootPackage, Database* database ) = 0;

public:
	/** @brief do not expect userinput
	 */
	void setNonInteractive(bool i = true) { noninteractive = i; }

public:
	Database mDB;
	std::string mInstallRoot;
	std::string mDepInstallPath;	///< target path for dependent packages (installRoot is used if it is empty)

protected:
	bool noninteractive ; ///< when true, don't ask vor userinput

	PromptProvider* mPromptProvider;	///< provider for user interaction
};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_MIRAPACKAGE_H_ */
