/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ThreadInfoView.C
 *    Implementation of ThreadInfoView.h.
 *
 * @author Tim Langner
 * @date   2011/01/14
 */

#include <views/ThreadInfoView.h>

#include <QVBoxLayout>
#include <QHeaderView>
#include <QMenu>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

const Duration cRPCTimeout = Duration::seconds(3);

///////////////////////////////////////////////////////////////////////////////

ThreadInfoView::ThreadInfoView() : ui(NULL)
{
	mAutoHideServiceSelector = new QAction("Auto-hide service selector", this);
	mAutoHideServiceSelector->setCheckable(true);
	mAutoHideServiceSelector->setChecked(true);

	mShowUnknownThreads= new QAction("Show unknown threads", this);
	mShowUnknownThreads->setCheckable(true);
	mShowUnknownThreads->setChecked(false);

	mShowHiddenThreads= new QAction("Show hidden threads", this);
	mShowHiddenThreads->setCheckable(true);
	mShowHiddenThreads->setChecked(false);

	mLastTimingMeasurement = Time::invalid();
}

ThreadInfoView::~ThreadInfoView()
{
	mInfoWorkerThread.interrupt();
	mInfoWorkerThread.join();
}

QWidget* ThreadInfoView::createPartControl()
{
	ui = new UI(this);
	ui->mServiceBox->updateServices();

	ui->mServiceBox->setAutoHide(mAutoHideServiceSelector->isChecked());
	connect(mAutoHideServiceSelector, SIGNAL(toggled(bool)),
		ui->mServiceBox, SLOT(setAutoHide(bool)));
	connect(ui->mServiceBox, SIGNAL(selected(std::string)),
	        this, SLOT(serviceSelected()));

	QMenu* menu = getViewMenu();
	menu->addAction(mAutoHideServiceSelector);
	menu->addAction(mShowUnknownThreads);
	menu->addAction(mShowHiddenThreads);

	mInfoWorkerThread = boost::thread(boost::bind(&ThreadInfoView::updateInfo, this));
	
	startTimer(2000);

	return ui;
}

void ThreadInfoView::timerEvent(QTimerEvent *e)
{
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		ui->mServiceBox->updateServices();
	}
	update();
}

void ThreadInfoView::updateInfo()
{
	ThreadMonitor::instance().addThisThread("#ThreadInfoView");

	RPCManager& rpcman = MIRA_FW.getRPCManager();

	while (!boost::this_thread::interruption_requested())
	{
		try {
			std::string service;
			{
				boost::mutex::scoped_lock lock(mServiceMutex);
				service = ui->mServiceBox->getSelectedService();
			}
			if (!service.empty()) {
				auto f = rpcman.call<ThreadMonitor::BasicThreadInfoVector>(service,
				                                                           "getThreadInformation");
				if (f.timedWait(cRPCTimeout)) {
					boost::mutex::scoped_lock lock(mServiceMutex);
					// discard result if the service selection changed since calling the RPC function
					if (service == ui->mServiceBox->getSelectedService()) {
						ThreadMonitor::BasicThreadInfoVector threads = f.get();

						boost::mutex::scoped_lock lock(mInfoMutex);
						updateTiming(threads);
					}
				}
			}
		}
		catch(boost::thread_interrupted&) {
		        throw; // do NOT catch thread interuption!
		}
		catch(...) {}

		MIRA_SLEEP(2000);
	}
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
}


inline void resetItem(QTableWidget* table, int i, int j, const QVariant& v, int alignment=Qt::AlignLeft)
{
	QTableWidgetItem* item=table->item(i,j);
	if(item==NULL) {
		item = new QTableWidgetItem();
		item->setTextAlignment(alignment);
		table->setItem(i, j, item);
	}
	item->setData(Qt::DisplayRole, v);
}

void ThreadInfoView::serviceSelected()
{
	// service selection changed, clear the thread data and the
	// display table now and wait for it to update

	ui->table->setRowCount(0);

	boost::mutex::scoped_lock lock(mInfoMutex);
	updateTiming(ThreadMonitor::BasicThreadInfoVector());
}

void ThreadInfoView::update()
{
	if (ui==NULL)
		return;

	if(!isVisible()) // do not update if we are not visible to save performance
		return;

	boost::mutex::scoped_lock lock(mInfoMutex);

	ui->table->setRowCount(mThreadStats.size());

	// we need to disable sorting while inserting the data
	ui->table->setSortingEnabled(false);
	int i=0;
	foreach(auto& stat, mThreadStats)
	{
		resetItem(ui->table, i, 0, stat.second.info.identity.name.c_str(), Qt::AlignVCenter);

		double cpuUsage = 0.0;
		if(stat.second.valid)
			cpuUsage = round(stat.second.cpuUsage*1000.0)/10.0;
			//s = QString("%1").arg(stat.second.cpuUsage*100.0, 5, 'f', 1);

		resetItem(ui->table, i, 1, cpuUsage, Qt::AlignRight | Qt::AlignVCenter);

		resetItem(ui->table, i, 2, toString(stat.second.info.user_time + stat.second.info.kernel_time).c_str(), Qt::AlignVCenter);
		resetItem(ui->table, i, 3, toString(stat.second.info.user_time).c_str(), Qt::AlignVCenter);
		resetItem(ui->table, i, 4, toString(stat.second.info.kernel_time).c_str(), Qt::AlignVCenter);
		resetItem(ui->table, i, 5, toString(stat.second.info.identity.id).c_str(), Qt::AlignVCenter);
		resetItem(ui->table, i, 6, toString(stat.second.info.create_time).c_str(), Qt::AlignVCenter);
		++i;
	}
	ui->table->setSortingEnabled(true);
}

void ThreadInfoView::updateTiming(const ThreadMonitor::BasicThreadInfoVector& info)
{
	if(!mLastTimingMeasurement.isValid())
		mLastTimingMeasurement = Time::now();

	// get time, elapsed since last round
	Time now = Time::now();
	Duration dt = now - mLastTimingMeasurement;
	mLastTimingMeasurement = now;

	foreach(auto& p, mThreadStats)
		p.second.zombie = true;

	foreach(const ThreadMonitor::BasicThreadInfo& i, info)
	{
		Duration totalTime = i.kernel_time + i.user_time;
		ThreadID id = i.identity.id;
		bool hide = false;
		if(!i.known && !mShowUnknownThreads->isChecked())
			hide = true;
		if(i.identity.name.find("#") != std::string::npos && !mShowHiddenThreads->isChecked())
			hide = true;
		if (hide)
			continue;

		auto it = mThreadStats.find(id);
		if(it == mThreadStats.end())
		{
			mThreadStats.insert(std::make_pair(id,ThreadStat(totalTime))).first->second.info = i;
		}
		else {
			ThreadStat& stat = it->second;
			Duration thread_dt = totalTime - stat.lastTotalTime;

			stat.cpuUsage = (float)thread_dt.totalMicroseconds() /
			                (float)dt.totalMicroseconds();

			stat.lastTotalTime = totalTime;
			stat.valid = true;
			stat.zombie = false;
			stat.info = i;
		}
	}

	auto it = mThreadStats.begin();
	for(; it != mThreadStats.end();)
	{
		if (it->second.zombie)
		{
			auto tmp = it;
			++it;
			mThreadStats.erase(tmp);
		}
		else
			++it;
	}
}

///////////////////////////////////////////////////////////////////////////////


ThreadInfoView::UI::UI(QWidget* parent) : QWidget(parent)
{
	setupUi();
}

void ThreadInfoView::UI::setupUi()
{
	QVBoxLayout* verticalLayout = new QVBoxLayout(this);
	verticalLayout->setContentsMargins(0,3,0,0);

	mServiceBox = new ServiceSelectorBox(this);
	mServiceBox->setInterface("IThreadMonitor");
	mServiceBox->preferInitially(MIRA_FW.getGlobalID());
	verticalLayout->addWidget(mServiceBox);

	table = new QTableWidget(this);
	QFontMetrics fm(table->font());
	table->setColumnCount(7);
	table->sortByColumn(1, Qt::DescendingOrder);
	table->setSortingEnabled(true);
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	table->horizontalHeader()->resizeSection(0,fm.horizontalAdvance("ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
	table->horizontalHeader()->resizeSection(1,fm.horizontalAdvance("%CPUMMM"));
	table->horizontalHeader()->resizeSection(2,fm.horizontalAdvance("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(3,fm.horizontalAdvance("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(4,fm.horizontalAdvance("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(5,fm.horizontalAdvance("888888MM"));
#else
	table->horizontalHeader()->resizeSection(0,fm.width("ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
	table->horizontalHeader()->resizeSection(1,fm.width("%CPUMMM"));
	table->horizontalHeader()->resizeSection(2,fm.width("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(3,fm.width("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(4,fm.width("88:88:88.888888MM"));
	table->horizontalHeader()->resizeSection(5,fm.width("888888MM"));
#endif
	table->horizontalHeader()->setStretchLastSection(true);
	table->verticalHeader()->setDefaultSectionSize(fm.height()+6);
	table->setSelectionMode(QAbstractItemView::NoSelection);
	table->setAlternatingRowColors(true);

	QStringList l;
	l << "Name" << "%CPU" << "Total" << "User" << "System" << "ID" << "Start Time";
	table->setHorizontalHeaderLabels(l);

	verticalLayout->addWidget(table);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::ThreadInfoView, mira::ViewPart);
