/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LibrariesListView.C
 *    Implementation of LibrariesListView.h
 *
 * @author Christof Schröter
 * @date   2020/10/05
 */

#include <views/LibrariesListView.h>

#include <QHeaderView>
#include <QMenu>

#include <fw/Framework.h>

#include <widgets/QtUtils.h>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(6);

///////////////////////////////////////////////////////////////////////////////

LibrariesListView::LibrariesListView() : mira::EditorPart()
{
	mAutoHideServiceSelector = new QAction("Auto-hide service selector", this);
	mAutoHideServiceSelector->setCheckable(true);
	mAutoHideServiceSelector->setChecked(true);

	mShowLoadedOnly = new QAction("Show loaded libs only", this);
	mShowLoadedOnly->setCheckable(true);
	mShowLoadedOnly->setChecked(false);

	mServiceBox = NULL;
	mLibrariesView = NULL;
}

LibrariesListView::~LibrariesListView()
{
	mDataWorkerThread.interrupt();
	mDataWorkerThread.join();
}

QWidget* LibrariesListView::createPartControl()
{
	QWidget* topWidget = new QWidget(this);
	QVBoxLayout* topLayout = new QVBoxLayout;
	topWidget->setLayout(topLayout);

	mServiceBox = new ServiceSelectorBox(topWidget);
	mServiceBox->setInterface("ILibraryRegistry");
	mServiceBox->preferInitially(MIRA_FW.getGlobalID());
	mServiceBox->updateServices();

	mServiceBox->setAutoHide(mAutoHideServiceSelector->isChecked());
	connect(mAutoHideServiceSelector, SIGNAL(toggled(bool)),
		mServiceBox, SLOT(setAutoHide(bool)));
	connect(mServiceBox, SIGNAL(selected(std::string)),
	        this, SLOT(serviceSelected(const std::string&)));

	topLayout->addWidget(mServiceBox);

	connect(mShowLoadedOnly, SIGNAL(toggled(bool)),
	        this, SLOT(onShowLoadedOnly(bool)));

	QMenu* menu = getViewMenu();
	menu->addAction(mAutoHideServiceSelector);
	menu->addAction(mShowLoadedOnly);

	mLibrariesView = new QTreeWidget(topWidget);
	mLibrariesView->setColumnCount(4);
	mLibrariesView->setRootIsDecorated(false);
	mLibrariesView->setAlternatingRowColors(true);

	QStringList headers;
	headers << tr("Library") << tr("Version")
	        << tr("Manifest") << tr("Loaded");
	mLibrariesView->setHeaderLabels(headers);

#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	mLibrariesView->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setSectionResizeMode(1, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setSectionResizeMode(2, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setSectionResizeMode(3, QHeaderView::ResizeToContents);
#else
	mLibrariesView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(2, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(3, QHeaderView::ResizeToContents);
#endif

	mFilter = new TreeViewFilter(mLibrariesView, topWidget);
	mFilter->setFilterColumns(0, 1);
	mFilter->setFocus();

	topLayout->addWidget(mFilter);
	topLayout->addWidget(mLibrariesView);

	mDataWorkerThread = boost::thread(boost::bind(&LibrariesListView::updateLibraries, this));

	startTimer(5000);

	return topWidget;
}

void LibrariesListView::timerEvent(QTimerEvent* event)
{
	bool signalledChange;
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		signalledChange = mServiceBox->updateServices();
	}
	if (!signalledChange)
		updateDisplay(mServiceBox->getSelectedService());
}

void LibrariesListView::onShowLoadedOnly(bool loaded)
{
	std::string service;
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		service = mServiceBox->getSelectedService();
	}
	updateDisplay(service);
}

void LibrariesListView::serviceSelected(const std::string& service)
{
	updateDisplay(service);
}

void LibrariesListView::updateLibraries()
{
	ThreadMonitor::instance().addThisThread("#LibrariesListView");

	RPCManager& rpcman = MIRA_FW.getRPCManager();

	while (!boost::this_thread::interruption_requested())
	{
		try {
			std::string service;
			{
				boost::mutex::scoped_lock lock(mServiceMutex);
				service = mServiceBox->getSelectedService();
			}
			if (!service.empty()) {
				auto f = rpcman.call<LibraryRegistry::Register>(service, "getLibraries");

				if (f.timedWait(cRPCTimeout)) {
					boost::mutex::scoped_lock lock(mDataMutex);
					mRegistries[service] = f.get();
				}
			}
		}
		catch(boost::thread_interrupted&) {
		        throw; // do NOT catch thread interuption!
		}
		catch(...) {}

		MIRA_SLEEP(5000);
	}
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
}

void LibrariesListView::updateDisplay(const std::string& service)
{
	boost::mutex::scoped_lock lock(mDataMutex);
	const LibraryRegistry::Register& reg = mRegistries[service];

	int size = reg.size();
	if (mShowLoadedOnly->isChecked()) {
		size = 0;
		foreach(const auto& l, reg) {
			if (l.second.mIsLoaded)
				++size;
		}
	}

	for (int i = mLibrariesView->topLevelItemCount(); i < size; ++i)
		mLibrariesView->addTopLevelItem(new QTreeWidgetItem());

	while (mLibrariesView->topLevelItemCount() > size)
		delete mLibrariesView->topLevelItem(0);

	int row = 0;
	foreach(const auto& l, reg)
	{
		if (mShowLoadedOnly->isChecked() && !l.second.mIsLoaded)
			continue;

		QTreeWidgetItem* item = mLibrariesView->topLevelItem(row);
		item->setText(0, QString::fromStdString(l.first));
		item->setText(1, QString("%1.%2.%3").arg(l.second.mVersion.mMajor)
		                                    .arg(l.second.mVersion.mMinor)
		                                    .arg(l.second.mVersion.mPatch));
		item->setCheckState(2, l.second.mHasManifest ? Qt::Checked : Qt::Unchecked);
		item->setCheckState(3, l.second.mIsLoaded ? Qt::Checked : Qt::Unchecked);
		item->setFlags(item->flags() & ~Qt::ItemIsUserCheckable);

		QBrush brush;
		QString tooltip;
		if (l.second.mLoadError == LibraryInfo::NoError)
		{
			auto palette = mLibrariesView->palette();
			brush = (row % 2 == 0) ? palette.base() : palette.alternateBase();
		}
		else
		{
			brush = QBrush(Qt::red);
			tooltip = QString::fromStdString(l.second.reportError());
		}

		for (int c = 0; c < item->columnCount(); ++c) {
			item->setBackground(c, brush);
			item->setToolTip(c, tooltip);
		}
		++row;
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::LibrariesListView, mira::EditorPart);
