/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DefinitionsListView.C
 *    Implementation of DefinitionsListView.h
 *
 * @author Christian Reuther, Christof Schröter
 * @date   2017/04/25
 */

#include <views/DefinitionsListView.h>

#include <QMenu>
#include <QHeaderView>

#include <fw/Framework.h>

#include <widgets/TreeViewFilter.h>
#include <widgets/QtUtils.h>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(6);

///////////////////////////////////////////////////////////////////////////////

DefinitionsListView::DefinitionsListView() : mira::EditorPart()
{
	mAutoHideServiceSelector = new QAction("Auto-hide service selector", this);
	mAutoHideServiceSelector->setCheckable(true);
	mAutoHideServiceSelector->setChecked(true);

	mServiceBox = NULL;
	mVariablesView = NULL;
	mAliasView = NULL;
}

DefinitionsListView::~DefinitionsListView()
{
	mDataWorkerThread.interrupt();
	mDataWorkerThread.join();
}

QWidget* DefinitionsListView::createPartControl()
{
	QWidget* topWidget = new QWidget(this);
	QVBoxLayout* topLayout = new QVBoxLayout;
	topWidget->setLayout(topLayout);

	mServiceBox = new ServiceSelectorBox(topWidget);
	mServiceBox->setInterfaces({"INameRegistry", "IVariableRegistry"});
	mServiceBox->preferInitially(MIRA_FW.getGlobalID());
	mServiceBox->updateServices();

	mServiceBox->setAutoHide(mAutoHideServiceSelector->isChecked());
	connect(mAutoHideServiceSelector, SIGNAL(toggled(bool)),
		mServiceBox, SLOT(setAutoHide(bool)));
	connect(mServiceBox, SIGNAL(selected(std::string)),
	        this, SLOT(serviceSelected(const std::string&)));

	topLayout->addWidget(mServiceBox);

	QMenu* menu = getViewMenu();
	menu->addAction(mAutoHideServiceSelector);

	QTabWidget* tabWidget = new QTabWidget(topWidget);
	topLayout->addWidget(tabWidget);

	QWidget* aliasWidget = new QWidget(tabWidget);
	QVBoxLayout* aliasLayout = new QVBoxLayout;
	aliasWidget->setLayout(aliasLayout);

	mAliasView = new QTreeWidget(aliasWidget);
	mAliasView->setColumnCount(2);
	mAliasView->setRootIsDecorated(false);
	mAliasView->setAlternatingRowColors(false);

	QStringList headers;
	headers << tr("Resource name / used as") << tr("Source");
	mAliasView->setHeaderLabels(headers);

#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	mAliasView->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
	mAliasView->header()->setSectionResizeMode(1, QHeaderView::Stretch);
#else
	mAliasView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mAliasView->header()->setResizeMode(1, QHeaderView::Stretch);
#endif

	TreeViewFilter* aliasFilter = new TreeViewFilter(mAliasView, aliasWidget);
	aliasFilter->setExpandAll(true);
	aliasFilter->setFilterColumns(0, 1);
	aliasFilter->setFocus();

	mAliasView->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(mAliasView, SIGNAL(customContextMenuRequested(const QPoint&)),
	        this,       SLOT(popupMenu(const QPoint&)));

	aliasLayout->addWidget(aliasFilter);
	aliasLayout->addWidget(mAliasView);

	QWidget* variablesWidget = new QWidget(tabWidget);
	QVBoxLayout* varLayout = new QVBoxLayout;
	variablesWidget->setLayout(varLayout);

	mVariablesView = new QTreeWidget(variablesWidget);
	mVariablesView->setColumnCount(3);
	mVariablesView->setRootIsDecorated(false);
	mVariablesView->setAlternatingRowColors(true);

	headers.clear();
	headers << tr("Variable") << tr("Value") << tr("Source");
	mVariablesView->setHeaderLabels(headers);

#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	mVariablesView->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
	mVariablesView->header()->setSectionResizeMode(1, QHeaderView::ResizeToContents);
	mVariablesView->header()->setSectionResizeMode(2, QHeaderView::Stretch);
#else
	mVariablesView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mVariablesView->header()->setResizeMode(2, QHeaderView::Stretch);
#endif

	TreeViewFilter* varFilter = new TreeViewFilter(mVariablesView, variablesWidget);
	varFilter->setFilterColumns(0, 1);
	varFilter->setFocus();

	mVariablesView->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(mVariablesView, SIGNAL(customContextMenuRequested(const QPoint&)),
	        this,           SLOT(popupMenu(const QPoint&)));

	varLayout->addWidget(varFilter);
	varLayout->addWidget(mVariablesView);

	tabWidget->addTab(aliasWidget, "Aliases");
	tabWidget->addTab(variablesWidget, "Variables");

	mDataWorkerThread = boost::thread(boost::bind(&DefinitionsListView::updateData, this));

	startTimer(5000);

	return topWidget;
}

void DefinitionsListView::timerEvent(QTimerEvent* event)
{
	bool signalledChange;
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		signalledChange = mServiceBox->updateServices();
	}
	if (!signalledChange) {
		updateDisplay(mServiceBox->getSelectedService());
	}
}

void DefinitionsListView::serviceSelected(const std::string& service)
{
	updateDisplay(service);
}

void DefinitionsListView::updateData()
{
	ThreadMonitor::instance().addThisThread("#DefinitionsListView");

	RPCManager& rpcman = MIRA_FW.getRPCManager();

	while (!boost::this_thread::interruption_requested())
	{
		std::string service;
		{
			boost::mutex::scoped_lock lock(mServiceMutex);
			service = mServiceBox->getSelectedService();
		}

		if (!service.empty()) {
			try {
				auto f = rpcman.call<XMLVariablesMap>(service, "getVariables");

				if (f.timedWait(cRPCTimeout)) {
					boost::mutex::scoped_lock lock(mDataMutex);
					mVariables[service] = f.get();
				}
			}
			catch(boost::thread_interrupted&) {
			        throw; // do NOT catch thread interuption!
			}
			catch(...) {}

			try {
				auto f = rpcman.call<NameRegistry::AliasMap>(service, "getAliases");
				if (f.timedWait(cRPCTimeout)) {
					boost::mutex::scoped_lock lock(mDataMutex);
					mAliases[service] = f.get();
				}
			}
			catch(boost::thread_interrupted&) {
			        throw; // do NOT catch thread interuption!
			}
			catch(...) {}
		}

		MIRA_SLEEP(5000);
	}
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
}

void DefinitionsListView::updateDisplay(const std::string& service)
{
	updateVariablesDisplay(service);
	updateAliasesDisplay(service);
}
	
void DefinitionsListView::updateVariablesDisplay(const std::string& service)
{
	boost::mutex::scoped_lock lock(mDataMutex);
	const XMLVariablesMap& variables = mVariables[service];

	for (int i = mVariablesView->topLevelItemCount(); i < (int)variables.size(); ++i)
		mVariablesView->addTopLevelItem(new QTreeWidgetItem());

	while (mVariablesView->topLevelItemCount() > (int)variables.size())
		delete mVariablesView->topLevelItem(0);

	int row = 0;
	foreach(const auto& v, variables)
	{
		QTreeWidgetItem* item = mVariablesView->topLevelItem(row);
		item->setText(0, QString::fromStdString(v.first));
		item->setText(1, QString::fromStdString(v.second));
		QString src = QString::fromStdString(v.second.annotation);
		item->setText(2, src);
		item->setToolTip(0, src);
		item->setToolTip(1, src);
		item->setToolTip(2, src);

		++row;
	}
}

void DefinitionsListView::updateAliasesDisplay(const std::string& service)
{
	// Create mapping "to --> [from]", i.e. for each "real" resource, which "virtual"
	// resources point to it?
	std::map<std::string, std::vector<std::pair<std::string, std::string>>> mapping;

	{
		boost::mutex::scoped_lock lock(mDataMutex);
		const NameRegistry::AliasMap& aliases = mAliases[service];

		foreach(const auto& a, aliases)
			mapping[a.second.first.str()].push_back(std::make_pair(a.first.str(), a.second.second));
	}

	for (int i = mAliasView->topLevelItemCount(); i < (int)mapping.size(); ++i)
		mAliasView->addTopLevelItem(new QTreeWidgetItem());

	while (mAliasView->topLevelItemCount() > (int)mapping.size())
		delete mAliasView->topLevelItem(0);

	int topIdx = 0;
	int basePalette = 0;

	foreach(const auto& a, mapping)
	{
		const QBrush& brush = (basePalette ? palette().base() : palette().alternateBase());
		basePalette = 1 - basePalette;

		QTreeWidgetItem* fromItem = mAliasView->topLevelItem(topIdx++);
		fromItem->setText(0, QString::fromStdString(a.first));

		fromItem->setBackground(0, brush);
		fromItem->setBackground(1, brush);

		for (int i = fromItem->childCount(); i < (int)a.second.size(); ++i)
			fromItem->addChild(new QTreeWidgetItem(fromItem));

		while (fromItem->childCount() > (int)a.second.size())
			delete fromItem->child(0);

		int childIdx = 0;

		// Do all the "to" bits
		foreach(const auto& to, a.second)
		{
			QTreeWidgetItem* toItem = fromItem->child(childIdx++);
			toItem->setText(0, QString::fromStdString(to.first));
			QString src = QString::fromStdString(to.second);
			toItem->setText(1, src);

			toItem->setToolTip(0, src);
			toItem->setToolTip(1, src);

			toItem->setBackground(0, brush);
			toItem->setBackground(1, brush);
		}

		fromItem->setExpanded(true);
	}
}

void DefinitionsListView::popupMenu(const QPoint& pos)
{
	QMenu menu;

	QAction* exportVariablesAction = NULL;
	QAction* exportAliasAction = NULL;

	if (mVariablesView->isVisible())
		exportVariablesAction = menu.addAction("Export variables list");

	if (mAliasView->isVisible())
		exportAliasAction = menu.addAction("Export alias list");

	if (menu.isEmpty())
		return;

	QAction* a = menu.exec(QCursor::pos());
	if (a && (a == exportVariablesAction))
		exportVariablesList();
	else if (a && (a == exportAliasAction))
		exportAliasList();
}

void DefinitionsListView::exportVariablesList()
{
	std::string service;
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		service = mServiceBox->getSelectedService();
	}

	XMLVariablesMap variables;

	{
		boost::mutex::scoped_lock lock(mDataMutex);
		auto it = mVariables.find(service);
		if (it == mVariables.end() || it->second.empty())
			return;

		variables = it->second;
	}

	static QString saveDir = ".";
	QString fn = QtUtils::getSaveFileName(NULL, tr("Export variables"), saveDir,
	                                      tr("XML")+" (*.xml)",
	                                      NULL, QFileDialog::DontUseNativeDialog, QStringList("xml"));
	if (fn.isEmpty())
		return;

	XMLDom xml;

	foreach (const auto& entry, variables)
		xml.root().add_child("var").add_attribute(entry.first, entry.second.value);

	xml.saveToFile(fn.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(fn).path();
}

void DefinitionsListView::exportAliasList()
{
	std::string service;
	{
		boost::mutex::scoped_lock lock(mServiceMutex);
		service = mServiceBox->getSelectedService();
	}

	NameRegistry::AliasMap aliases;

	{
		boost::mutex::scoped_lock lock(mDataMutex);
		auto it = mAliases.find(service);
		if (it == mAliases.end() || it->second.empty())
			return;

		aliases = it->second;
	}

	static QString saveDir = ".";
	QString fn = QtUtils::getSaveFileName(NULL, tr("Export aliases"), saveDir,
	                                      tr("XML")+" (*.xml)",
	                                      NULL, QFileDialog::DontUseNativeDialog, QStringList("xml"));
	if (fn.isEmpty())
		return;

	XMLDom xml;

	foreach (const auto& entry, aliases) {
		std::cout << entry.first.str() << " : " << entry.second.first.str() << std::endl;
		ResourceName alias = entry.first;

		std::string leaf = alias.leaf();
		ResourceName parent = alias.parent();

		std::list<std::string> resNS;
		while (!parent.isRoot()) {
			resNS.push_front(parent.leaf());
			parent = parent.parent();
		}

		// Enter the resource alias' namespace tag(s), creating them if necessary
		XMLDom::sibling_iterator it = xml.root();
		foreach (const auto& s, resNS) {
			XMLDom::sibling_iterator nsIt = std::find(it.begin(), it.end(), "namespace");
			while (true) {
				if (nsIt == it.end()) {
					// namespace with name s does not exist, create and enter it
					it = it.add_child("namespace").add_attribute("name", s);
					break;
				}

				if (nsIt.get_attribute<std::string>("name") == s) {
					// namespace with name s found, enter it
					it = nsIt;
					break;
				}

				// keep searching
				nsIt = std::find(++nsIt, it.end(), "namespace");
			}
		}

		// keep usings before subnamespaces
		XMLDom::sibling_iterator nsIt = std::find(it.begin(), it.end(), "namespace");
		if (nsIt == it.end())
			it = it.add_child("using").add_attribute("name", entry.second.first.str());
		else
			it = nsIt.insert_before("using").add_attribute("name", entry.second.first.str());
			
		if (leaf != entry.second.first.leaf())
			it.add_attribute("as", leaf);
	}
	
	xml.saveToFile(fn.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(fn).path();
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::DefinitionsListView, mira::EditorPart);

