/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <boost/test/unit_test.hpp>

#include <robot/DifferentialRobotModel.h>
#include <serialization/JSONSerializer.h>

using namespace std;
using namespace mira;
using namespace mira::robot;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(deadReckoning_circle)
{
	Pose2 pose(0.0f, 0.0f, 0.0f);

	DifferentialRobotModel model;
	model.wheelDist = 0.5f;

	float angleDelta = 10.0f;
	float steps = (360.0f/angleDelta);
	float deltaR = two_pi<float>()*model.wheelDist/steps;

	struct {
		int steps;
		float x, y;
		float phi;
	} tPath[4] = {
		{ (int)(steps/4),  0.25f, 0.25f, deg2rad(90.0f) },
		{ (int)(steps/4),  0.00f, 0.50f, deg2rad(180.0f) },
		{ (int)(steps/4), -0.25f, 0.25f, deg2rad(-90.0f) },
		{ (int)(steps/4),  0.00f, 0.00f, deg2rad(0.0f) }
	};

	for(int k = 0; k < 4; k++) {
		for(int i = 0; i < tPath[k].steps; i++)
			pose = model.globalKinematics(model.convert2Velocity(0.0, deltaR), 1.0f, pose);
		BOOST_CHECK_SMALL(pose.x() - tPath[k].x, 0.01f);
		BOOST_CHECK_SMALL(pose.y() - tPath[k].y, 0.01f);
		BOOST_CHECK_SMALL(std::abs(SignedAnglef(pose.phi()).smallestDifferenceValue(SignedAnglef(tPath[k].phi))), deg2rad(1.0f));
	}
}


///////////////////////////////////////////////////////////////////////////////

/*
 *         |
 *         \
 *          \____
 *               \
 *                \
 *                |
 *                |
 *                /
 *           ____/
 *          /
 *         /
 *         |
 */
BOOST_AUTO_TEST_CASE(deadReckoning_complex)
{
	Pose2 pose(0.0f, 0.0f, 0.0f);

	DifferentialRobotModel model;
	model.wheelDist = 0.5f;

	int i;

	float innerRad = 0.5f*half_pi<float>();
	float outerRad = (0.5f+model.wheelDist)*half_pi<float>();
	float deltaTrans = 0.1f;

	// rotate 90deg right
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(0.1f*outerRad, 0.1f*innerRad),1.0f,pose);

	// drive one meter forward
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(deltaTrans, deltaTrans),1.0f,pose);

	// rotate 90deg left
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(0.1f*innerRad, 0.1f*outerRad),1.0f,pose);

	// drive one meter forward
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(deltaTrans, deltaTrans),1.0f,pose);

	// rotate 90deg left
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(0.1f*innerRad, 0.1f*outerRad),1.0f,pose);

	// drive one meter forward
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics(model.convert2Velocity(deltaTrans, deltaTrans),1.0f,pose);

	// rotate 90deg right
	for(i = 0; i < 10; i++)
		pose = model.globalKinematics( model.convert2Velocity(0.1f*outerRad, 0.1f*innerRad),1.0f,pose);

	BOOST_CHECK_SMALL(pose.x() - 4.0f, 0.01f);
	BOOST_CHECK_SMALL(pose.y() - 0.0f, 0.01f);	
	BOOST_CHECK_SMALL(std::abs(SignedAnglef(pose.phi()).smallestDifferenceValue(SignedAnglef(0.0f))), deg2rad(1.0f));
}

///////////////////////////////////////////////////////////////////////////////

