/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GeometryVisualization.C
 *    .
 *
 * @author Tim Langner
 * @date   2011/07/22
 */

#include <transform/Pose.h>
#include <visualization/plot/VisualizationPlotCurveJSON.h>

#include <boost/geometry/strategies/strategies.hpp>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

template<typename Pose>
class PosePlotVisualizationBase : public VisualizationPlotCurveJSON<Pose>
{
public:
	PosePlotVisualizationBase() :
		VisualizationPlotCurveJSON<Pose>("Pose")
	{
	}

	virtual void dataChanged(ChannelRead<Pose> data)
	{
		// if channel is no valid node in the transform tree we just plot it
		if (MIRA_FW.getTransformer()->getNode(this->mChannel.getID()) == NULL)
			VisualizationPlotCurveJSON<Pose>::dataChanged(data);
	}

protected:

	virtual void update(Duration dt)
	{
		if (!this->mChannel.isValid())
			return;
		if (MIRA_FW.getTransformer()->getNode(this->mChannel.getID()) == NULL)
			return;
		if (!this->getSite()->isAcceptingUpdates())
			return;

		try
		{
			Time n = Time::now();
			Pose d = this->getAuthority().template getTransform<Pose>(this->mChannel.getID(),
			                                                          this->getSite()->getCameraFrame(),
			                                                          n);
			JSONSerializer s;
			json::Value data = s.serialize(d);
			this->plot(data, n);
			this->ok("Plot");
		}
		catch(Exception& ex)
		{
			this->error("Plot", ex.message());
		}
		catch(std::string& ex)
		{
			this->error("Plot", ex);
		}
	}
};

class Pose2PlotVisualization : public PosePlotVisualizationBase<Pose2>
{
MIRA_META_OBJECT(Pose2PlotVisualization,
                 ("Name", "Pose 2D")
                 ("Description", "Visualization of 2D poses as plot")
                 ("Category", "Pose"))
};

class Pose3PlotVisualization : public PosePlotVisualizationBase<Pose3>
{
MIRA_META_OBJECT(Pose3PlotVisualization,
                 ("Name", "Pose 3D")
                 ("Description", "Visualization of 3D poses as plot")
                 ("Category", "Pose"))
};

class PoseCov2PlotVisualization : public PosePlotVisualizationBase<PoseCov2>
{
MIRA_META_OBJECT(PoseCov2PlotVisualization,
                 ("Name", "Pose 2D")
                 ("Description", "Visualization of 2D poses with covariance as plot")
                 ("Category", "Pose"))
};

class PoseCov3PlotVisualization : public PosePlotVisualizationBase<PoseCov3>
{
MIRA_META_OBJECT(PoseCov3PlotVisualization,
                 ("Name", "Pose 3D")
                 ("Description", "Visualization of 3D poses with covariance as plot")
                 ("Category", "Pose"))
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::Pose2PlotVisualization, mira::VisualizationPlotCurve);
MIRA_CLASS_SERIALIZATION( mira::PoseCov2PlotVisualization, mira::VisualizationPlotCurve);
MIRA_CLASS_SERIALIZATION( mira::Pose3PlotVisualization, mira::VisualizationPlotCurve);
MIRA_CLASS_SERIALIZATION( mira::PoseCov3PlotVisualization, mira::VisualizationPlotCurve);
