/*
 * Copyright (C) 2017 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotGrid.h
 *    A visualization that shows a grid in the plot.
 *
 * @author Christof Schröter
 * @date   2017/12/05
 */

#ifndef _MIRA_VISUALIZATIONPLOTGRID_H_
#define _MIRA_VISUALIZATIONPLOTGRID_H_

#include <qwt_plot_grid.h>
#include <qwt_plot.h>

#include <serialization/SetterNotify.h>
#include <serialization/adapters/Qt/QColor>

#include <visualization/VisualizationPlot.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <class BasePlotType>
class VisualizationPlotGrid : public BasePlotType
{
	static const std::string sStyleHint; // for use in property("LineStyle")

	struct LineProperties {

		LineProperties(VisualizationPlotGrid* parent, const QColor& color = Qt::gray,
		               float lineWidth = 0.f, const Qt::PenStyle& lineStyle = Qt::DotLine)
			: parent(parent), color(color),
			  lineWidth(lineWidth), lineStyle(lineStyle) {}

		template <typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("Color", color,
			           setterNotify(color, &VisualizationPlotGrid::redraw, parent),
			           "The color of the grid", Qt::gray);
			r.property("LineWidth", lineWidth,
			           setterNotify(lineWidth, &VisualizationPlotGrid::redraw, parent),
			           "The line width of the grid", 0.f,
			           PropertyHints::minimum(0.f));
			r.property("LineStyle", lineStyle,
			           setterNotify(lineStyle, &VisualizationPlotGrid::redraw, parent),
			           "The line style of the grid", Qt::DotLine,
			           PropertyHints::enumeration(VisualizationPlotGrid::sStyleHint));
		}

		VisualizationPlotGrid* parent;
		QColor color;
		float lineWidth;
		Qt::PenStyle lineStyle;
	};

public:

	VisualizationPlotGrid() :
		mGrid(NULL),
		mMajorGridProperties(this),
		mMinorGridProperties(this),
		mXMajorEnabled(true),
		mXMinorEnabled(false),
		mYMajorEnabled(true),
		mYMinorEnabled(false)
	{
	}

	virtual ~VisualizationPlotGrid()
	{
		if (mGrid)
		{
			mGrid->detach();
			delete mGrid;
		}
		if (BasePlotType::getSite() && BasePlotType::getSite()->getPlot())
			BasePlotType::getSite()->getPlot()->replot();
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VisualizationPlot);
		int v = r.version(2, this);
		if (v < 2) {
			r.property("Color", mMajorGridProperties.color,
			           setter(&VisualizationPlotGrid::setColor, this),
			           "The color of the grid", Qt::gray);
			r.property("LineWidth", mMajorGridProperties.lineWidth,
			           setter(&VisualizationPlotGrid::setLineWidth, this),
			           "The line width of the grid", 0.f,
			           PropertyHints::minimum(0.f));
			r.property("LineStyle", mMajorGridProperties.lineStyle,
			           setter(&VisualizationPlotGrid::setLineStyle, this),
			           "The line style of the grid", Qt::DotLine,
			           PropertyHints::enumeration(sStyleHint));
		} else {
			r.property("Major Grid", mMajorGridProperties,
			           "Line properties of the major grid", LineProperties(this));
			r.property("Minor Grid", mMinorGridProperties,
			           "Line properties of the minor grid", LineProperties(this));
		}
		r.property("X Major Grid", mXMajorEnabled,
		           setterNotify(mXMajorEnabled, &VisualizationPlotGrid::redraw, this),
		           "Enable major grid in x axis", true);
		r.property("X Minor Grid", mXMinorEnabled,
		            setterNotify(mXMinorEnabled, &VisualizationPlotGrid::redraw, this),
		           "Enable minor grid in x axis", false);
		r.property("Y Major Grid", mYMajorEnabled,
		           setterNotify(mYMajorEnabled, &VisualizationPlotGrid::redraw, this),
		           "Enable major grid in y axis", true);
		r.property("Y Minor Grid", mYMinorEnabled,
		           setterNotify(mYMinorEnabled, &VisualizationPlotGrid::redraw, this),
		           "Enable minor grid in y axis", false);
	}

	virtual void setupScene(IVisualizationPlotSite* site)
	{
		mGrid = createGrid();
	}

	virtual void clear()
	{
	}

	virtual void setEnabled(bool enabled)
	{
		BasePlotType::setEnabled(enabled);
		if (!mGrid)
			return;

		mGrid->setVisible(BasePlotType::isEnabled());
		BasePlotType::getSite()->getPlot()->replot();
	}

	virtual int getYAxis()
	{
		return -1;
	}

	virtual QString getPlotTitle()
	{
		return "Grid";
	}

	void redraw()
	{
		if (!mGrid)
			return;

		mGrid->setMajorPen(QPen(mMajorGridProperties.color,
		                        mMajorGridProperties.lineWidth,
		                        mMajorGridProperties.lineStyle));
		mGrid->setMinorPen(QPen(mMinorGridProperties.color,
		                        mMinorGridProperties.lineWidth,
		                        mMinorGridProperties.lineStyle));
		mGrid->enableX(mXMajorEnabled);
		mGrid->enableXMin(mXMinorEnabled);
		mGrid->enableY(mYMajorEnabled);
		mGrid->enableYMin(mYMinorEnabled);
		BasePlotType::getSite()->getPlot()->replot();
	}

	void setColor(const QColor& color)
	{
		mMajorGridProperties.color = color;
		mMinorGridProperties.color = color;
		redraw();
	}

	void setLineWidth(float width)
	{
		mMajorGridProperties.lineWidth = width;
		mMinorGridProperties.lineWidth = width;
		redraw();
	}

	void setLineStyle(const Qt::PenStyle& style)
	{
		mMajorGridProperties.lineStyle = style;
		mMinorGridProperties.lineStyle = style;
		redraw();
	}

	virtual typename BasePlotType::QwtPlotItems getPlotItems()
	{
		return {mGrid};
	}

	QwtPlotGrid* createGrid()
	{
		mGrid = new QwtPlotGrid();
		mGrid->setMajorPen(QPen(mMajorGridProperties.color,
		                        mMajorGridProperties.lineWidth,
		                        mMajorGridProperties.lineStyle));
		mGrid->setMinorPen(QPen(mMinorGridProperties.color,
		                        mMinorGridProperties.lineWidth,
		                        mMinorGridProperties.lineStyle));
		mGrid->setVisible(BasePlotType::isEnabled());
		mGrid->enableX(mXMajorEnabled);
		mGrid->enableXMin(mXMinorEnabled);
		mGrid->enableY(mYMajorEnabled);
		mGrid->enableYMin(mYMinorEnabled);
		mGrid->attach(BasePlotType::getSite()->getPlot());
		BasePlotType::getSite()->getPlot()->replot();
		mGrid->setZ(20.); // seems to be the default z for QwtPlotCurve (undocumented)
		return mGrid;
	}

protected:
	QwtPlotGrid* mGrid;
	LineProperties mMajorGridProperties;
	LineProperties mMinorGridProperties;
	bool mXMajorEnabled;
	bool mXMinorEnabled;
	bool mYMajorEnabled;
	bool mYMinorEnabled;
};

template <typename BasePlotType>
const std::string VisualizationPlotGrid<BasePlotType>::sStyleHint
	= MakeString() << Qt::SolidLine << "=SolidLine;"
	               << Qt::DashLine << "=DashLine;"
	               << Qt::DotLine << "=DotLine;"
	               << Qt::DashDotLine << "=DashDotLine;"
	               << Qt::DashDotDotLine << "=DashDotDotLine";

///////////////////////////////////////////////////////////////////////////////

}

#endif 
