/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PlaneVisualzation.C
 *
 * @author Christian Vollmer
 * @date   2011/02/09
 */

#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/GridObject.h>
#include <visualization/3d/AxesObject.h>

#include <fw/TransformProperty.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class PlaneVisualization3D :  public Visualization3DBasic<Eigen::Vector4f>
{
	MIRA_META_OBJECT(PlaneVisualization3D,
		("Name", "Plane")
		("Category", "Geometry")
		("Description", "Visualizes a plane specified by an Eigen::Vector4f, "
		                "where the first three elements are the normal and the "
		                "fourth element is the offset."))

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	PlaneVisualization3D() :
		Visualization3DBasic<Eigen::Vector4f>("Plane"),
		mGridObject(NULL),
		mAxesObject(NULL),
		mColor(Ogre::ColourValue::Black),
		mCellCount(20),
		mCellSize(0.2f),
		mShowAxes(false)
	{
		mDataChannel.setOptional();
	}

	/// The destructor.
	virtual ~PlaneVisualization3D() {
		delete mGridObject;
		delete mAxesObject;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3DBasic<Eigen::Vector4f>);
		r.property("CellCount", mCellCount,
		           setter<int>(&PlaneVisualization3D::setCellCount, this),
		           "The number of cells", 20);
		r.property("CellSize", mCellSize,
		           setter<float>(&PlaneVisualization3D::setCellSize, this),
		           "The size of a cell in meter", 0.2f);
		r.property("Color", mColor,
		           setter<Ogre::ColourValue>(&PlaneVisualization3D::setColor, this),
		           "The color of the grid", Ogre::ColourValue::Black);
		r.property("ShowAxes", mShowAxes,
		           setter<bool>(&PlaneVisualization3D::showAxes, this),
		           "Show plane orientation axes", false);
		r.property("Position", mPositionTransform,
		           "The plane center is shown at the projection of this "
		           "position to the plane", TransformProperty());
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mGridObject = new GridObject(mgr, node);
		mGridObject->setCellCount(mCellCount);
		mGridObject->setCellSize(mCellSize);
		mGridObject->setColor(mColor);

		mAxesObject = new AxesObject(mgr, node);
		mAxesObject->setVisible(mShowAxes);
	}

	void dataChanged(ChannelRead<Eigen::Vector4f> read)
	{
		// get normal n and d of the plane with: n*x+d = 0
		Eigen::Vector3f n = read->value().block<3,1>(0,0);
		float nnorm = n.norm();
		n/=nnorm; // normalize normal
		float d = read->value()[3] / nnorm;

		// we compute the orientation here manually, because we want the Z
		// vector of the plane to be oriented upwards
		Ogre::Vector3 normal(n.x(), n.y(), n.z());
		Ogre::Vector3 unitZ(0,1,0);
		Ogre::Vector3 x = unitZ.crossProduct(-normal);
		Ogre::Vector3 y = x.crossProduct(-normal);
		Ogre::Quaternion q;
		q.FromAxes(x,y,normal);


		// compute position of the plane
		Eigen::Vector3f p(0.0f,0.0f,0.0f);
		if(mPositionTransform.isValid())
			p = getTransform<Pose3>(mPositionTransform.getID(), mDataFrameID, read->timestamp).t;

		// compute projection of p to the plane
		Eigen::Vector3f o = -n*d;     // origin of the plane
		Eigen::Vector3f op = p - o;
		float s = op.dot(n);
		Eigen::Vector3f pos = p-(n*s);


		Ogre::Vector3 opos(pos.x(),pos.y(),pos.z());
		mGridObject->setOrientation(q);
		mGridObject->setPosition(opos);

		mAxesObject->setOrientation(q);
		mAxesObject->setPosition(opos);
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3DBasic<Eigen::Vector4f>::setEnabled(enabled);
		if (mGridObject != NULL)
			mGridObject->setVisible(enabled);
		if (mAxesObject != NULL) {
			mAxesObject->setVisible(mShowAxes && enabled);
		}
	}

	//@}

public:
	/** @name Setters */
	//@{

	/// Set the number of cells.
	void setCellCount(int count)
	{
		mCellCount = count;
		if (mGridObject != NULL)
			mGridObject->setCellCount(mCellCount);
	}

	/// Set the cell size in [m].
	void setCellSize(float size)
	{
		mCellSize = size;
		if (mGridObject != NULL)
			mGridObject->setCellSize(mCellSize);
	}

	/// Set the color of the grid.
	void setColor(const Ogre::ColourValue& c)
	{
		mColor = c;
		if (mGridObject != NULL)
			mGridObject->setColor(mColor);
	}

	virtual void showAxes(bool show) {
		mShowAxes = show;
		if (mAxesObject != NULL) {
			mAxesObject->setVisible(mShowAxes && isEnabled());
		}
	}

	//@}

private:
	GridObject* mGridObject;
	AxesObject* mAxesObject;
	Ogre::ColourValue mColor;
	int mCellCount;
	float mCellSize;
	bool mShowAxes;
	TransformProperty mPositionTransform;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::PlaneVisualization3D, mira::Visualization3D);
