/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Rect2DVisualization.C
 *    A visualization for 2D rects.
 *
 * @author Christian Martin
 * @date   2012/02/29
 */

#include <QGraphicsRectItem>
#include <QGraphicsEllipseItem>

#include <serialization/adapters/Qt/QColor>

#include <geometry/Rect.h>

#include <visualization/Visualization2DBasic.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/// A visualization for 2D rects.
template<typename ItemType, typename DataType>
class Rect2DVisualizationBase :
	public Visualization2DBasic<DataType>
{
public:
	typedef Visualization2DBasic<DataType> Base;

	/// Fill style mode.
	enum FillMode
	{
		UNFILLED = 0,   ///< Unfilled mode.
		FILLED_SOLID    ///< Solid fill mode.
	};

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	Rect2DVisualizationBase() :
		mGraphicsItem(NULL),
		mColor(Qt::black),
		mLineWidth(1.0f),
		mFillMode(UNFILLED),
		mFillColor(Qt::yellow)
	{}

	/// The destructor.
	virtual ~Rect2DVisualizationBase() {}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("Color", mColor,
		           setter<QColor>(&Rect2DVisualizationBase::setColor, this),
		           "The color.", Qt::black);
		r.property("LineWidth", mLineWidth,
		           setter<float>(&Rect2DVisualizationBase::setLineWidth, this),
		           "The line width.", 1.0f);
		r.property("FillMode", mFillMode,
		           setter<FillMode>(&Rect2DVisualizationBase::setFillMode, this),
		           "The fill mode.", UNFILLED,
		           mira::PropertyHints::enumeration("Unfilled;Solid"));
		r.property("FillColor", mFillColor,
		           setter<QColor>(&Rect2DVisualizationBase::setFillColor, this),
		           "The fill color.", Qt::yellow);
	}

	//@}

public:
	/** @name Some setters */
	//@{

	/**
	 * @brief Set the color of the rect visualization
	 * @param[in] color The new color.
	 */
	void setColor(const QColor& color)
	{
		mColor = color;
		updatePen();
	}

	/**
	 * @brief Set the line width of the rect visualization
	 * @param[in] lineWidth The new line width.
	 */
	void setLineWidth(float lineWidth)
	{
		mLineWidth = lineWidth;
		updatePen();
	}

	/**
	 * @brief Set the fill style of the visualization.
	 * @param[in] fillMode The fill mode.
	 */
	void setFillMode(FillMode fillMode)
	{
		mFillMode = fillMode;
		updatePen();
	}

	/**
	 * @brief Set the fill color of the rect visualization
	 * @param[in] color The new color.
	 */
	void setFillColor(const QColor& color)
	{
		mFillColor = color;
		updatePen();
	}

	//@}

public:
	/** @name Public implementation of Visualization2DBasic */
	//@{

	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr)
	{
		// Create the graphics item and add it to the scene.
		mGraphicsItem = new ItemType();
		mgr->addItem(mGraphicsItem);
		return mGraphicsItem;
	}

	//@}

protected:
	/** @name Protected implementation of Visualization2DBasic */
	//@{

	virtual void dataChanged(ChannelRead<DataType> data)
	{
		const DataType& value = data->value();
		auto size = value.size();
		mGraphicsItem->setRect((qreal)value.minCorner.x(),
		                       (qreal)value.minCorner.y(),
		                       (qreal)size.width(), (qreal)size.height());
		updatePen();
	}

	//@}

private:
	void updatePen()
	{
		if (mGraphicsItem) {
			mGraphicsItem->setPen(QPen(mColor, mLineWidth));

			if (mFillMode == UNFILLED)
				mGraphicsItem->setBrush(QBrush(mFillColor, Qt::NoBrush));
			else
			if (mFillMode == FILLED_SOLID)
				mGraphicsItem->setBrush(QBrush(mFillColor, Qt::SolidPattern));
		}
	}

protected:
	ItemType* mGraphicsItem;
	QColor mColor;
	float mLineWidth;
	FillMode mFillMode;
	QColor mFillColor;
};

///////////////////////////////////////////////////////////////////////////////

class Rect2iVisualization2D :
	public Rect2DVisualizationBase<QGraphicsRectItem, Rect2i>
{
	MIRA_META_OBJECT(Rect2iVisualization2D,
		("Name", "Rect")
		("Description", "Visualization of a 2D int rectangle")
		("Category", "Geometry"))
};

class Rect2fVisualization2D :
	public Rect2DVisualizationBase<QGraphicsRectItem, Rect2f>
{
	MIRA_META_OBJECT(Rect2fVisualization2D,
		("Name", "Rect")
		("Description", "Visualization of a 2D float rectangle")
		("Category", "Geometry"))
};

class Ellipse2iVisualization2D :
	public Rect2DVisualizationBase<QGraphicsEllipseItem, Rect2i>
{
	MIRA_META_OBJECT(Ellipse2iVisualization2D,
		("Name", "Ellipse")
		("Description", "Visualization of a 2D int rectangle as an ellipse")
		("Category", "Geometry"))
};

class Ellipse2fVisualization2D :
	public Rect2DVisualizationBase<QGraphicsEllipseItem, Rect2f>
{
	MIRA_META_OBJECT(Ellipse2fVisualization2D,
		("Name", "Ellipse")
		("Description", "Visualization of a 2D float rectangle as an ellipse")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}}

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::gui::Rect2iVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2fVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Ellipse2iVisualization2D,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Ellipse2fVisualization2D,
                         mira::Visualization2D);
