/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonVisualization.C
 *    Visualization for polygons in 2D
 *
 * @author Christian Reuther
 * @date   2015/11/05
 */

#include <visualization/Visualization2DBasic.h>

#include <serialization/Serialization.h>
#include <serialization/adapters/Qt/QColor>

#include <geometry/Polygon.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <class Poly>
class PolygonVisualization2DBase : public Visualization2DBasic<Poly>
{
public:
	PolygonVisualization2DBase() : Visualization2DBasic<Poly>(), mPolygon(NULL), mOutlineColor(Qt::black), mFillColor(Qt::transparent) {}

	virtual ~PolygonVisualization2DBase()
	{
		// Don't delete mPolygon as mItem points to it and is deleted already
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization2DBasic<Poly>);
		r.property("OutlineColor", mOutlineColor, setter<QColor>(&PolygonVisualization2DBase::setOutlineColor, this), "", Qt::black);
		r.property("FillColor", mFillColor, setter<QColor>(&PolygonVisualization2DBase::setFillColor, this), "", Qt::transparent);
	}

public:
	void setOutlineColor(const QColor& c)
	{
		mOutlineColor = c;
		if(mPolygon)
		{
			mPolygon->setPen(QPen(QBrush(c), 0.0));
			mPolygon->update();
		}
	}

	void setFillColor(const QColor& c)
	{
		mFillColor = c;
		if(mPolygon)
		{
			mPolygon->setBrush(QBrush(c));
			mPolygon->update();
		}
	}

public:
	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr)
	{
		mPolygon = new QGraphicsPolygonItem(NULL);
		mPolygon->setVisible(this->isEnabled());
		setOutlineColor(mOutlineColor);
		setFillColor(mFillColor);

		mgr->addItem(mPolygon);
		return mPolygon;
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization2DBasic<Poly>::setEnabled(enabled);
	}

	void dataChanged(ChannelRead<Poly> polygon)
	{
		QPolygonF poly;
		foreach(const auto& p, polygon->value())
			poly.append(QPointF(p(0), p(1)));
		mPolygon->setPolygon(poly);
	}

private:
	QGraphicsPolygonItem* mPolygon;
	QColor mOutlineColor;
	QColor mFillColor;
};

///////////////////////////////////////////////////////////////////////////////

class Polygon2fVisualization2D :  public PolygonVisualization2DBase<Polygon2f>
{
	MIRA_META_OBJECT(Polygon2fVisualization2D,
		("Name", "Polygon2f")
		("Description", "Visualizes a 2D float polygon in the xy-plane")
		("Category", "Geometry"))
};

class Polygon2dVisualization2D :  public PolygonVisualization2DBase<Polygon2f>
{
	MIRA_META_OBJECT(Polygon2dVisualization2D,
		("Name", "Polygon2d")
		("Description", "Visualizes a 2D double polygon in the xy-plane")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2fVisualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2dVisualization2D, mira::Visualization2D);
