/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DepthCameraIntrinsic
 *    Intrinsic parameters of a depth camera
 *
 * @author Erik Einhorn
 * @date   2013/08/16
 */

#ifndef _MIRA_CAMERA_DEPTHCAMERAINTRINSIC_H_
#define _MIRA_CAMERA_DEPTHCAMERAINTRINSIC_H_

#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/boost/optional.hpp>

#include <image/Img.h>

#include <cameraparameters/PinholeCameraIntrinsic.h>

namespace mira { namespace camera {

//////////////////////////////////////////////////////////////////////////////

class DepthCameraCorrectionFactors
{
public:

	DepthCameraCorrectionFactors();

	/// Loads the correction factors from file
	DepthCameraCorrectionFactors(const std::string& file);

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("DepthMin", depthMin, "The depth of the first correction factor plane");
		r.member("DepthResolution", depthResolution, "The depth distance between the planes");
		r.member("Factors", factors, "The correction factors");
	}

public:

	/// Stores the correction factors in a binary file
	void saveToFile(const std::string& file);

	/// Stores the correction factors in a binary file
	void loadFromFile(const std::string& file);


public:
	float depthMin;
	float depthResolution;
	std::vector<Img32F1> factors; // correction factors for each depth plane
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Parameters for converting the sensor readings to depth values.
 * To possible models are:
 *
 * 1) z = A * r
 * 2) z = A / (B - r)
 */
class DepthCameraConversionParameters
{
public:

	DepthCameraConversionParameters();

	DepthCameraConversionParameters(float a);

	DepthCameraConversionParameters(float a, float b);

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("A", A, "");
		r.property("B", B, "", -1.0f);
		r.property("BaseDistance", baseDistance, "", 0.075f);
	}

public:

	bool operator==(const DepthCameraConversionParameters& other) const {
		return ((A == other.A) && (B == other.B) && (baseDistance == other.baseDistance));
	}

public:
	float A;
	float B;
	float baseDistance;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Describes the intrinsic parameters of a pinhole depth camera, containing
 * the usual intrinsic parameters of a pinhole camera (focal length, pincushion
 * distortion parameters, etc) as well as additional parameters describing
 * the depth projection.
 * For each physical camera one instance of this class is sufficient since
 * the intrinsic camera parameters remain constant.
 */
class DepthCameraIntrinsicNormalized :
		public PinholeCameraIntrinsicNormalized,
		public DepthCameraConversionParameters

{
public:

	/// Constructor.
	DepthCameraIntrinsicNormalized();

	/**
	 * Sets all intrinsic parameters, such as the focal length Fx and Fy,
	 * the optical center Cx, Cy and the distortion parameters K1, K2, P1, P2.
	 */
	DepthCameraIntrinsicNormalized(const PinholeCameraIntrinsicNormalized& normalizedIntrinsic,
	                               float a, float b=-1.0);

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, PinholeCameraIntrinsicNormalized);
		MIRA_REFLECT_BASE(r, DepthCameraConversionParameters);
		r.member("DepthCorrectionFactors", mDepthCorrectionFactors, "", boost::optional<DepthCameraCorrectionFactors>());
	}


	void reflect(XMLSerializer& r)
	{
		MIRA_REFLECT_BASE(r, PinholeCameraIntrinsicNormalized);
		MIRA_REFLECT_BASE(r, DepthCameraConversionParameters);
		r.member("DepthCorrectionFile", mDepthCorrectionFactorsFile, "", boost::optional<std::string>());
	}

	void reflect(XMLDeserializer& r)
	{
		MIRA_REFLECT_BASE(r, PinholeCameraIntrinsicNormalized);
		MIRA_REFLECT_BASE(r, DepthCameraConversionParameters);
		r.member("DepthCorrectionFile", mDepthCorrectionFactorsFile, "", boost::optional<std::string>());

		if(mDepthCorrectionFactorsFile)
			mDepthCorrectionFactors = DepthCameraCorrectionFactors(*mDepthCorrectionFactorsFile);
	}

public:

	PinholeCameraIntrinsicNormalized& pinholeIntrinsic() { return *this; }
	const PinholeCameraIntrinsicNormalized& pinholeIntrinsic() const { return *this; }

	DepthCameraConversionParameters& depthParameters() { return *this; }
	const DepthCameraConversionParameters& depthParameters() const { return *this; }


public:

	/**
	 * Sets the file name of depth correction file, only. Note: No data
	 * is loaded. This method is just for preparing a config before serialization.
	 * Probably you want to call the loadDepthCorrectionFactors() method to load
	 * a certain correction file.
	 */
	void setDepthCorrectionFactorsFile(const std::string& file);


	void setDepthCorrectionFactors(const DepthCameraCorrectionFactors& factors);


	/**
	 * Loads the depth correction factors from the specified file.
	 */
	void loadDepthCorrectionFactors(const std::string& file);

	bool hasDepthCorrectionFactors() const { return (bool)mDepthCorrectionFactors; }

	/**
	 * Throws XInvalidConfig, if no depth correction factors are associated
	 * with this intrinsic parameter config.
	 */
	DepthCameraCorrectionFactors& depthCorrectionFactors();

	/**
	 * Throws XInvalidConfig, if no depth correction factors are associated
	 * with this intrinsic parameter config.
	 */
	const DepthCameraCorrectionFactors& depthCorrectionFactors() const;


public:

	bool operator==(const DepthCameraIntrinsicNormalized& other) const {
		return (((PinholeCameraIntrinsicNormalized)*this == (PinholeCameraIntrinsicNormalized)other) && ((DepthCameraConversionParameters)*this == (DepthCameraConversionParameters)other));
	}

private:

	boost::optional<DepthCameraCorrectionFactors> mDepthCorrectionFactors;
	boost::optional<std::string> mDepthCorrectionFactorsFile;

};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
