/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditor.C
 *    Implementation of PropertyEditor.h.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2010/12/12
 */

#include <assert.h>
#include <iostream>

#include <widgets/PropertyEditor.h>
#include <widgets/CollapsibleTreeHeader.h>

#include "private/PropertyEditorDelegateRegistry.h"
#include "private/PropertyEditorModel.h"
#include "private/PropertyEditorItemDelegate.h"
#include "private/PropertyEditorTreeView.h"

#include <QAction>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QToolButton>

#include <QThread>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

const QColor PropertyEditor::lightRed   (255,191,191);
const QColor PropertyEditor::lightGreen (191,255,191);
const QColor PropertyEditor::lightBlue  (200,200,255);
const QColor PropertyEditor::lightYellow(255,255,127);
const QColor PropertyEditor::lightPurple(255,191,255);

///////////////////////////////////////////////////////////////////////////////

// register a unique type id
const QEvent::Type PropertyEditor::AddRemoveChildEvent::typeID
	= static_cast<QEvent::Type>(QEvent::registerEventType());

///////////////////////////////////////////////////////////////////////////////

PropertyEditor::PropertyEditor(QWidget *parent) :
		QWidget(parent)
{
	mCurrentEditedProperty = NULL;
	mHideSingleRootNode = false;

	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setContentsMargins(0,0,0,0);
	layout->setMargin(0);
	layout->setSpacing(0);

	mTreeView = new TreeView(this);
	QHeaderView* header = new CollapsibleTreeHeader(Qt::Horizontal,mTreeView);
	mTreeView->setHeader( header );
	mModel = new Model(this, mTreeView);
	mItemDelegate = new ItemDelegate(this);

	mFilter = new TreeViewFilter(mTreeView,this);
	//mFilter->setFilterColumns(0);

	layout->addWidget(mFilter);
	layout->addWidget(mTreeView);

	mTreeView->setModel(mModel);
	mTreeView->setItemDelegate(mItemDelegate);

	// timer for updating our property editor regularly
	mUpdateTimerId = startTimer(500);

	setTabOrder(mTreeView, mFilter);
	mTreeView->setFocus();
}

PropertyEditor::~PropertyEditor()
{
	clear();
}

PropertyNode* PropertyEditor::currentProperty()
{
	QModelIndex index = mTreeView->currentIndex();
	if(!index.isValid())
		return NULL;

	// move up until we reach the top level item
	while(index.parent().isValid())
		index = index.parent();

	PropertyNode* node = mModel->index2node(index);
	return node;
}

void PropertyEditor::clear()
{
	for(auto it=mModel->mProperties.begin(); it!=mModel->mProperties.end(); ++it) {
		RootPropertyNode* root = it->first->getRootNode();
		if (root)
			root->unregisterListener(this);
	}

	mModel->beginResetModel();
	mModel->mProperties.clear();
	mModel->endResetModel();

	mFilter->clearFilter();
}

void PropertyEditor::addProperty(PropertyNode* property, QVariant user)
{
	RootPropertyNode* root = property->getRootNode();
	if (root)
		root->registerListener(this);

	mModel->beginInsertRows(QModelIndex(), mModel->mProperties.size(), mModel->mProperties.size());
	mModel->mProperties.push_back(std::make_pair(property,user));
	mModel->endInsertRows();
	mTreeView->setColumnWidth(0, 120);
}

void PropertyEditor::removeProperty(PropertyNode* property)
{
	RootPropertyNode* root = property->getRootNode();
	if (root)
		root->unregisterListener(this);

	mColorMap.erase(property);

	int row = 0;
	for(auto it=mModel->mProperties.begin(); it!=mModel->mProperties.end(); ++it, ++row)
	{
		if(it->first==property) {

			// make sure to select the next property, if we delete the current one
			int nextRow = -1;
			if(property == currentProperty()) {
				auto nextIt = it; ++nextIt;
				if(nextIt!=mModel->mProperties.end()) {
					nextRow = row;
				} else if(it!=mModel->mProperties.begin()) {
					nextRow = row-1;
				}
			}

			// delete the property
			mModel->beginRemoveRows(QModelIndex(), row, row);
			mModel->mProperties.erase(it);
			mModel->endRemoveRows();

			if(nextRow>=0) {
				QModelIndex nextIndex = mModel->index(nextRow,0);
				mTreeView->setCurrentIndex(nextIndex);
			}

			return;
		}
	}
}

void PropertyEditor::moveUpProperty(PropertyNode* property)
{
	// contains the "General" PropertyNode as first element (and it stays there!)
	if ((mModel->mProperties.size() < 3) || (mModel->mProperties[1].first == property))
		return;

	int row = 2;
	for(auto it=mModel->mProperties.begin()+2; it!=mModel->mProperties.end(); ++it, ++row)
	{
			if(it->first==property) {
				auto pred = it-1;
				mModel->beginMoveRows(QModelIndex(), row, row, QModelIndex(), row-1);
				std::swap(*it, *pred);
				mModel->endMoveRows();
				mTreeView->setCurrentIndex(mModel->index(row-1,0));
				return;
			}
	}
}

void PropertyEditor::moveDownProperty(PropertyNode* property)
{
	// contains the "General" PropertyNode as first element (and it stays there!)
	if ((mModel->mProperties.size() < 3) || (mModel->mProperties.back().first == property))
		return;

	int row = 1;
	for(auto it=mModel->mProperties.begin()+1; it!=mModel->mProperties.end()-1; ++it, ++row)
	{
			if(it->first==property) {
				auto succ = it+1;
				mModel->beginMoveRows(QModelIndex(), row, row, QModelIndex(), row+2);
				std::swap(*it, *succ);
				mModel->endMoveRows();
				mTreeView->setCurrentIndex(mModel->index(row+1,0));
				return;
			}
	}
}

QVariant PropertyEditor::getUser(PropertyNode* property)
{
	for(auto it=mModel->mProperties.begin(); it!=mModel->mProperties.end(); ++it)
	{
		if(it->first==property) {
			return it->second;
		}
	}
	return QVariant();
}

void PropertyEditor::setHideSingleRootNode(bool hide)
{
	mHideSingleRootNode = hide;
}

bool PropertyEditor::getHideSingleRootNode() const
{
	return mHideSingleRootNode;
}

void PropertyEditor::installFilterShortCut(QWidget* widget,
                                           const QString& shortcut)
{
	QAction* showFilterAction = new QAction(widget);
	showFilterAction->setShortcutContext(Qt::WidgetWithChildrenShortcut);
	showFilterAction->setShortcut(shortcut);
	widget->addAction(showFilterAction);
	connect(showFilterAction, SIGNAL(triggered(bool)), this, SLOT(showFilter()));
}

bool PropertyEditor::getAutoHideFilterBar() const
{
	return mFilter->getAutoHide();
}

void PropertyEditor::setAutoHideFilterBar(bool hide)
{
	mFilter->setAutoHide(hide);
}

void PropertyEditor::showFilter()
{
	mFilter->showFilter();
}

void PropertyEditor::hideFilter()
{
	mFilter->hideFilter();
}

/// Sets the background color for the specified property
void PropertyEditor::setColor(const PropertyNode* property, const QColor& color)
{
	mColorMap.insert(std::pair<const PropertyNode*,QColor>(property, color));
}

QColor PropertyEditor::getColor(const PropertyNode* property) const
{
	auto it = mColorMap.find(property);
	if(it!=mColorMap.end())
		return it->second;
	return QColor();
}

void PropertyEditor::timerEvent(QTimerEvent* e)
{
	if(e->timerId()==mUpdateTimerId) {
		// make us dirty, in order to force an update afterwards
		mTreeView->updateAll();
		return;
	}

	QWidget::timerEvent(e);
}

void PropertyEditor::customEvent(QEvent* e)
{
	if (e->type() != AddRemoveChildEvent::typeID)
		return;

	AddRemoveChildEvent* ev = static_cast<AddRemoveChildEvent*>(e);

	QModelIndex index;
	switch (ev->subtype) {
		case AddRemoveChildEvent::BEGIN_ADD:
			index = mModel->node2index(ev->node);
			if (ev->first) {
				mPopulatingIndex << index;
#if QT_VERSION < 0x050000
				mModel->layoutAboutToBeChanged();
#else
				mModel->layoutAboutToBeChanged(mPopulatingIndex);
#endif
			}
			mModel->beginInsertRows(index, ev->index, ev->index + ev->count-1);
			mModel->index(ev->index, 0, index); // create new index
			break;
		case AddRemoveChildEvent::END_ADD:
			mModel->endInsertRows();
			if (!mPopulatingIndex.empty()) {
#if QT_VERSION < 0x050000
				mModel->layoutChanged();
#else
				mModel->layoutChanged(mPopulatingIndex);
#endif
				mPopulatingIndex.clear();
			}
			break;
		case AddRemoveChildEvent::BEGIN_REMOVE:
			mModel->beginRemoveRows(mModel->node2index(ev->node), ev->index, ev->index + ev->count-1);
			break;
		case AddRemoveChildEvent::END_REMOVE:
			mModel->endRemoveRows();
			mPropertyNodeListenerCondition.wakeAll();
			break;
		case AddRemoveChildEvent::BEGIN_MOVE:
			mModel->beginMoveRows(mModel->node2index(ev->node), ev->index, ev->index + ev->count-1,
			                      mModel->node2index(ev->node), ev->destination);
			break;
		case AddRemoveChildEvent::END_MOVE:
			mModel->endMoveRows();
			break;
		default: MIRA_THROW(XRuntime, "Received AddRemoveChildEvent with invalid subtype");
	}
}

std::string PropertyEditor::getText(const PropertyNode* property) const
{
	try {
		DelegatePtr delegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::TEXT);
		if(!delegate)
			return std::string();
		return delegate->getText(property);
	} catch(Exception& ex) {
		return std::string("#exception# ") + ex.message();
	} catch(std::exception& ex) {
		return std::string("#exception# ") + ex.what();
	} catch(...) {
		return std::string("#exception#");
	}
}

QIcon PropertyEditor::getIcon(const PropertyNode* property) const
{
	try {
		DelegatePtr delegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::ICON);
		if(!delegate)
			return QIcon();
		return delegate->getIcon(property);
	} catch(...) {
		return QIcon();
	}
}

QColor PropertyEditor::getBackgroundColor(const PropertyNode* property) const
{
	try {
		DelegatePtr delegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::COLOR);
		if(!delegate)
			return QColor();
		return delegate->getBackgroundColor(property);
	} catch(...) {
		return QColor();
	}
}

QWidget* PropertyEditor::createEditor(PropertyNode* property, QWidget* parent)
{
	try {
		DelegatePtr editorDelegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::EDITOR);
		DelegatePtr dialogDelegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::DIALOG);
		if(!editorDelegate && !dialogDelegate)
			return NULL;

		mCurrentEditedProperty = property;
		PropertyItemEditor* e =
				new PropertyItemEditor(this, parent, property, editorDelegate, dialogDelegate);
		connect(e, SIGNAL(destroyed(QObject*)),
		        this, SLOT(onDestroyedEditorDelegate(QObject*)));
		return e;
	} catch(...) {
		return NULL;
	}
}

std::string PropertyEditor::getName(const PropertyNode* property)
{
	try {
		DelegatePtr delegate =
				DelegateRegistry::instance().getDelegate(property, DelegateRegistry::NAME);
		if(!delegate)
			return property->name();
		return delegate->getName(property);
	} catch(Exception& ex) {
		return std::string("#exception# ") + ex.message();
	} catch(std::exception& ex) {
		return std::string("#exception# ") + ex.what();
	} catch(...) {
		return std::string("#exception#");
	}
}

void PropertyEditor::onDestroyedEditorDelegate(QObject*)
{
	mCurrentEditedProperty = NULL;
}

void PropertyEditor::beginAddChildren(const PropertyNode* node, int index, int count)
{
	// post event, must be processed in Qt main thread
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::BEGIN_ADD, node,
	                                                          index, count, (node->children().size() == 0)));
}

void PropertyEditor::endAddChildren()
{
	// post event, must be processed in Qt main thread
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::END_ADD));
}

void PropertyEditor::beginRemoveChildren(const PropertyNode* node, int index, int count)
{
	// post event, must be processed in Qt main thread
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::BEGIN_REMOVE, node,
	                                                          index, count));
}

void PropertyEditor::endRemoveChildren()
{
	if (QThread::currentThread() == qApp->thread()) {
		// process pending AddRemoveChildEvents
		qApp->sendPostedEvents(this, AddRemoveChildEvent::typeID);
		mModel->endRemoveRows(); 
		return;
	}

	// post event, must be processed in Qt main thread
	// wait for it (must not delete PropertyNode before this is finished!)
	mPropertyNodeListenerMutex.lock();
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::END_REMOVE));
	mPropertyNodeListenerCondition.wait(&mPropertyNodeListenerMutex);
	mPropertyNodeListenerMutex.unlock();
}

void PropertyEditor::beginMoveChildren(const PropertyNode* node, int index, int count, int destination)
{
	// post event, must be processed in Qt main thread
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::BEGIN_MOVE, node,
	                                                          index, count, false, destination));
}

void PropertyEditor::endMoveChildren()
{
	// post event, must be processed in Qt main thread
	QCoreApplication::postEvent(this, new AddRemoveChildEvent(AddRemoveChildEvent::END_MOVE));
}

///////////////////////////////////////////////////////////////////////////////

PropertyItemEditor::PropertyItemEditor(PropertyEditor* editor, QWidget* parent,
                                       PropertyNode* property,
                                       PropertyEditor::DelegatePtr editorDelegate,
                                       PropertyEditor::DelegatePtr dialogDelegate) :
		QWidget(parent),
		mEditor(editor),
		mProperty(property),
		mDialogDelegate(dialogDelegate)
{
	assert(editorDelegate || dialogDelegate);

	QHBoxLayout* layout = new QHBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);

	//...
	QWidget* widget = NULL;
	if(editorDelegate) {
		widget = editorDelegate->createEditor(property, this);
		setFocusProxy(widget);
		setFocusPolicy(widget->focusPolicy());
	} else {
		// add a icon label, if we have an icon
		QIcon icon = mEditor->getIcon(property);
		if(!icon.isNull()) {
			QLabel* pixmapLabel = new QLabel(this);
			//QPixmap p = icon.pixmap(16,16);
			QPixmap p = icon.pixmap(mEditor->mTreeView->iconSize());
			pixmapLabel->setPixmap(p);
			pixmapLabel->setMargin(3);
			layout->addWidget(pixmapLabel);
			layout->addSpacing(3);
		}

		// create a simple label, if we do not have an editor but a dialog
		std::string text = mEditor->getText(property);
		widget = new QLabel(QString::fromUtf8(text.c_str()), this);
	}

	widget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
	layout->addWidget(widget);

	// add a "..." button if we have a dialog delegate
	if(dialogDelegate) {
		mButton = new QToolButton(this);
		mButton->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Ignored);
		mButton->setFixedWidth(20);
		// we we do not have a focus proxy yet, then set it to the button
		if(focusProxy()==NULL) {
			setFocusProxy(mButton);
			setFocusPolicy(mButton->focusPolicy());
		}
		mButton->setText(tr("..."));
		mButton->installEventFilter(this);
		connect(mButton, SIGNAL(clicked()), this, SLOT(buttonClicked()));
		layout->addWidget(mButton,Qt::AlignRight);
	}
}

void PropertyItemEditor::buttonClicked()
{
	assert(mDialogDelegate);
	if(mDialogDelegate->execDialog(mProperty, this))
		mEditor->mTreeView->closeEditor(this, QAbstractItemDelegate::NoHint);
}

bool PropertyItemEditor::eventFilter(QObject *obj, QEvent *e)
{
	if (obj == mButton) {
		switch (e->type()) {
			case QEvent::KeyPress:
			case QEvent::KeyRelease:
			{ // Prevent the QToolButton from handling Enter/Escape meant control the delegate
				switch (static_cast<const QKeyEvent*>(e)->key()) {
					case Qt::Key_Escape:
					case Qt::Key_Enter:
					case Qt::Key_Return:
						e->ignore();
						return true;
					default:
						break;
				}
			}
				break;
			default:
				break;
		}
	}
	return QWidget::eventFilter(obj, e);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::PropertyEditor::Delegate, mira::Object);
