/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicPoints.h
 *    Class for displaying dynamic points using vertex buffer.
 *    Adapted from http://www.ogre3d.org/tikiwiki/DynamicLineDrawing&structure=Cookbook
 *
 *    Original file: DynamicLines.cpp
 */

#include <visualization/3d/DynamicPoints.h>

#include <OGRE/Ogre.h>

#include <utils/Foreach.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

DynamicPoints::DynamicPoints(OperationType opType)
{
	Ogre::VertexDeclaration decl;
	decl.addElement(0, 0, Ogre::VET_FLOAT3, Ogre::VES_POSITION);
	std::size_t offset = Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT3);
	decl.addElement(0, offset, Ogre::VET_COLOUR, Ogre::VES_DIFFUSE);

	initialize(opType, decl, false);
	setMaterial("BaseWhiteNoLighting");
	mDirty = true;
}

DynamicPoints::~DynamicPoints()
{
}

void DynamicPoints::setOperationType(OperationType opType)
{
	mRenderOp.operationType = opType;
}

Ogre::RenderOperation::OperationType DynamicPoints::getOperationType() const
{
	return mRenderOp.operationType;
}

void DynamicPoints::addPoint(const Vector3 &p, const Ogre::ColourValue& c)
{
	PointData point;
	point.point = p;
	point.color = c;
	mPoints.push_back(point);
	mDirty = true;
}

void DynamicPoints::addPoint(Real x, Real y, Real z, const Ogre::ColourValue& c)
{
	PointData point;
	point.point = Vector3(x, y, z);
	point.color = c;
	mPoints.push_back(point);
	mDirty = true;
}

void DynamicPoints::setColor(const Ogre::ColourValue& c)
{
	foreach(auto& p, mPoints)
		p.color = c;
	mDirty = true;
}

const Ogre::Vector3& DynamicPoints::getPoint(unsigned short index) const
{
	assert(index < mPoints.size() && "Point index is out of bounds!!");
	return mPoints[index].point;
}

unsigned short DynamicPoints::getNumPoints(void) const
{
	return (unsigned short) mPoints.size();
}
void DynamicPoints::setPoint(unsigned short index, const Vector3 &value)
{
	assert(index < mPoints.size() && "Point index is out of bounds!!");

	mPoints[index].point = value;
	mDirty = true;
}

void DynamicPoints::clear()
{
	mPoints.clear();
	mDirty = true;
}

void DynamicPoints::update()
{
	if (mDirty)
		fillHardwareBuffers();
}

void DynamicPoints::fillHardwareBuffers()
{
	int size = mPoints.size();

	prepareBuffers(size);

	if (!size)
	{
		mBox.setExtents(Vector3::ZERO, Vector3::ZERO);
		mDirty = false;
		return;
	}

	Vector3 vaabMin = mPoints[0].point;
	Vector3 vaabMax = mPoints[0].point;

	Ogre::HardwareVertexBufferSharedPtr vbuf = getVertexBuffer(0);
	Ogre::RenderSystem* rs = Ogre::Root::getSingleton().getRenderSystem();

	Ogre::Real *prPos = static_cast<Real*> (vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));
	{
		for (int i = 0; i < size; i++)
		{
			*prPos++ = mPoints[i].point.x;
			*prPos++ = mPoints[i].point.y;
			*prPos++ = mPoints[i].point.z;

			Ogre::uint32* colorPtr = (Ogre::uint32*)prPos++;
			rs->convertColourValue(mPoints[i].color, colorPtr);

			if (mPoints[i].point.x < vaabMin.x)
				vaabMin.x = mPoints[i].point.x;
			if (mPoints[i].point.y < vaabMin.y)
				vaabMin.y = mPoints[i].point.y;
			if (mPoints[i].point.z < vaabMin.z)
				vaabMin.z = mPoints[i].point.z;

			if (mPoints[i].point.x > vaabMax.x)
				vaabMax.x = mPoints[i].point.x;
			if (mPoints[i].point.y > vaabMax.y)
				vaabMax.y = mPoints[i].point.y;
			if (mPoints[i].point.z > vaabMax.z)
				vaabMax.z = mPoints[i].point.z;
		}
	}
	vbuf->unlock();

	mBox.setExtents(vaabMin, vaabMax);

	mDirty = false;
}

///////////////////////////////////////////////////////////////////////////////

}
