/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Workbench.C
 *    Implementation of Workbench.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/15
 */

#include <boost/make_shared.hpp>

#include <QDockWidget>
#include <QMdiArea>
#include <QPushButton>
#include <QPainter>

#include <QApplication>

#include <rcp/Workbench.h>

#include <serialization/adapters/std/list>
#include <serialization/adapters/std/map>
#include <serialization/adapters/boost/shared_ptr.hpp>
#include <serialization/adapters/Qt/QByteArray>
#include <serialization/BinarySerializer.h>

#include <xml/XMLDomReflect.h>

#include <geometry/Rect.h>

#include <rcp/EditorPartArea.h>

#include <rcp/PerspectiveTabWidget.h>
#include <rcp/PartListener.h>


#define PARTMEMORY_FILENAME "miracenter.partmemory"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class MinimizedDockWidgetButton : public QPushButton
{
public:
	MinimizedDockWidgetButton(const QString& text, QWidget* parent, Qt::DockWidgetArea area) :
		QPushButton(text,parent), mArea(area) {

		int s=18;
		if(mArea==Qt::TopDockWidgetArea || mArea==Qt::BottomDockWidgetArea) {
			mOrientation = Qt::Horizontal;
			setSizePolicy(QSizePolicy::Preferred,QSizePolicy::Fixed);
			setFixedHeight(s);
		} else {
			mOrientation = Qt::Vertical;
			setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Preferred);
			setFixedWidth(s);
		}
	}


	QSize sizeHint() const
	{
		QSize s = QPushButton::sizeHint();
		if(mOrientation==Qt::Vertical)
			s.transpose();
		return s;
	}

	void paintEvent(QPaintEvent* event)
	{
		QPainter painter(this);
		QRect rct = this->rect();

		if(mOrientation==Qt::Vertical) {
			QSize size = rct.size();
			size.transpose();
			rct.setSize(size);
		}

		if(mArea==Qt::RightDockWidgetArea) {
			painter.rotate(90);
			painter.translate(0, -width());
		} else if(mArea==Qt::LeftDockWidgetArea) {
			painter.rotate(-90);
			painter.translate(-height(), 0);
		}

		QLinearGradient grad(QPointF(0, 0), QPointF(0, rct.height()));
		QPalette pal = palette();

		QColor baseColor;
		QColor textColor;
		bool mIsHighlighted = this->isDown();
		if(mIsHighlighted) {
			baseColor = pal.color(QPalette::Active, QPalette::Highlight);
			textColor = pal.color(QPalette::Active, QPalette::HighlightedText);
		} else {
			baseColor = pal.color(QPalette::Active, QPalette::Mid);
			textColor = pal.color(QPalette::Active, QPalette::Text);
		}

		grad.setColorAt(0,   baseColor.lighter(120));
		grad.setColorAt(0.4, baseColor);
		grad.setColorAt(0.5, baseColor.darker(120));
		grad.setColorAt(1,   baseColor);

		painter.fillRect(rct, grad);

		QFontMetrics fM(painter.font());
		QString text(fM.elidedText(this->text(), Qt::ElideRight,
		                           rct.width()));
		painter.setPen(textColor);
		painter.drawText(rct, Qt::AlignVCenter, text);
	}


private:

	Qt::Orientation mOrientation;
	Qt::DockWidgetArea mArea;

};


///////////////////////////////////////////////////////////////////////////////

Workbench::Workbench(bool setup) :
	mCurrentPerspective(NULL),
	mPerspectiveTabWidget(NULL),
	mActiveEditorPart(NULL)
{
	if(setup)
		setupUi();
}

Workbench::~Workbench()
{
	clearWorkspaceParts();
}

void Workbench::setupUi()
{
	if(menuBar()!=NULL)
		mPerspectiveTabWidget = new PerspectiveTabWidget(menuBar());

	connect(mPerspectiveTabWidget, SIGNAL(changedPerspective(Perspective*)),
	        this, SLOT(onChangedPerspective(Perspective*)));
	connect(mPerspectiveTabWidget, SIGNAL(onAddPerspectiveRequested(std::string)),
	        this, SLOT(addPerspective(std::string)));
	connect(mPerspectiveTabWidget, SIGNAL(onRemovedPerspectiveRequested(Perspective*)),
	        this, SLOT(removePerspective(Perspective*)));
	connect(mPerspectiveTabWidget, SIGNAL(movedPerspective(int,int)),
	        this, SLOT(onMovedPerspective(int,int)));

	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
	if(central==NULL) {
		if(centralWidget()!=NULL)
			MIRA_THROW(XLogical, "There's already a centralWidget set which is"
			           " not of type 'mira::EditorPartArea' or a derived class. "
			           " Please use a central widget of that type.");

		// create central widget of type EditorPartArea and add it, if no such
		// central widget exists yet.
		central = new EditorPartArea(this);
		setCentralWidget(central);
	}

	connect(central, SIGNAL(editorWindowActivated(EditorPartWindow*)),
	        this, SLOT(onEditorWindowActivated(EditorPartWindow*)));

	createMinimizedDockArea(Qt::LeftDockWidgetArea);
	createMinimizedDockArea(Qt::RightDockWidgetArea);
	createMinimizedDockArea(Qt::TopDockWidgetArea);
	createMinimizedDockArea(Qt::BottomDockWidgetArea);

	mDockRestoreMapper = new QSignalMapper(this);
	connect(mDockRestoreMapper, SIGNAL(mapped(QWidget*)),
	        this, SLOT(onViewRestore(QWidget*)));

	loadPartMemory();
}


void Workbench::createMinimizedDockArea(Qt::DockWidgetArea area)
{
	bool horizontal = (int)area >= (int)Qt::TopDockWidgetArea;

	QWidget* widget = new QWidget(this);
	if(horizontal)
		widget->setFixedHeight(22);
	else
		widget->setFixedWidth(22);

	QDockWidget* dock = new QDockWidget(this);
	dock->setWidget(widget);
	dock->hide();
	dock->setFeatures(QDockWidget::NoDockWidgetFeatures);
	dock->setTitleBarWidget(new QWidget(dock)); // no titlebar

	QString id;
	switch(area)
	{
	case Qt::LeftDockWidgetArea:   id="Left"; break;
	case Qt::RightDockWidgetArea:  id="Right"; break;
	case Qt::TopDockWidgetArea:    id="Top"; break;
	case Qt::BottomDockWidgetArea: id="Bottom"; break;
	default: id="Unknown"; break;
	}
	dock->setObjectName("MinimizedDockArea"+id);

	QBoxLayout* layout;

	if(horizontal)
		layout = new QHBoxLayout(widget);
	else
		layout = new QVBoxLayout(widget);

	layout->setSpacing(2);
	layout->setMargin(2);

	mMinimizedDockAreas[area].dock = dock;
	mMinimizedDockAreas[area].widget = widget;
	mMinimizedDockAreas[area].layout = layout;
	mMinimizedDockAreas[area].minimized = false;
}

void Workbench::setupNewViewPart(ViewPart* viewPart)
{
	assert(viewPart!=NULL);

	connect(viewPart, SIGNAL(viewDestroyed(ViewPart*)),
	        this    , SLOT(onViewDestroyed(ViewPart*)));

	connect(viewPart, SIGNAL(viewClosed(ViewPart*)),
	        this    , SLOT(storePartMemory(ViewPart*)));

	connect(viewPart, SIGNAL(viewActivated(ViewPart*)),
	        this    , SLOT(onViewActivated(ViewPart*)));

	connect(viewPart, SIGNAL(minimizedRequested(ViewPart*)),
	        this    , SLOT(onViewMinimizedRequested(ViewPart*)));

	connect(viewPart, SIGNAL(currentDockLocationChanged(Qt::DockWidgetArea)),
	        this    , SLOT(onViewDockLocationChanged(Qt::DockWidgetArea)));



	viewPart->setParent(this);
	viewPart->setWindowTitle(viewPart->getClass().getMetaInfo("Name").c_str());
	viewPart->setObjectName(viewPart->getClass().getMetaInfo("Name").c_str());
	viewPart->init(this);

	addDockWidget(viewPart->getDefaultDockArea(), viewPart);

	mCreatedViews.insert(std::make_pair(viewPart->getClass().getIdentifier(), viewPart));
	mViewParts.push_back(viewPart);
}

void Workbench::setupNewEditorPart(EditorPart* editorPart)
{
	assert(editorPart!=NULL);

	connect(editorPart, SIGNAL(editorDestroyed(EditorPart*)),
	        this      , SLOT(onEditorDestroyed(EditorPart*)));

	connect(editorPart, SIGNAL(editorClosed(EditorPart*)),
	        this      , SLOT(storePartMemory(EditorPart*)));

	editorPart->setParent(this);
	if ( editorPart->windowTitle().isEmpty() )
		editorPart->setWindowTitle(	QString::fromStdString( editorPart->getClass().getMetaInfo("Name") ) );
	editorPart->setAttribute(Qt::WA_DeleteOnClose);
	editorPart->init(this);

	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
	assert(central!=NULL && "The central widget must be of type EditorPartArea!");
	central->addSubWindow(editorPart);
	central->activateSubWindow(editorPart);

	mEditorParts.push_back(editorPart);

	onEditorCreated(editorPart);

	connect(editorPart, SIGNAL(windowTitleChanged(EditorPartWindow*)),
	        this      , SLOT(onWindowTitleChanged(EditorPartWindow*)));
}

QWidget* Workbench::createPart(ClassProxy partClass)
{
	if(partClass.isDerivedFrom(&ViewPart::CLASS())) {
		// if we are trying to create a new view, first check if we have
		// created it already:
		auto it = mCreatedViews.find(partClass.getIdentifier());
		if(it!=mCreatedViews.end()) {
			it->second->show();
			it->second->activate();
			storeCurrentPerspective();
			// use the existing view
			return it->second;
		}
	}

	WorkbenchPart* part = partClass.newInstance<WorkbenchPart>();
	assert(part!=NULL);

	QWidget* partWidget = dynamic_cast<QWidget*>(part);
	if(partWidget==NULL) {
		delete part; // clean up
		MIRA_THROW(XRuntime, "The created WorkbenchPart '" 
		           << partClass.getIdentifier() << "' is not a QWidget");
	}
	partWidget->setParent(this);

	if(partClass.isDerivedFrom(&ViewPart::CLASS())) {
		ViewPart* viewPart = dynamic_cast<ViewPart*>(part);
		setupNewViewPart(viewPart);
	}

	if(partClass.isDerivedFrom(&EditorPart::CLASS())) {
		EditorPart* editorPart = dynamic_cast<EditorPart*>(part);
		setupNewEditorPart(editorPart);
	}

	partWidget->show();
	storeCurrentPerspective();
	return partWidget;
}

///////////////////////////////////////////////////////////////////////////////

struct ViewPartPersistentItem
{
	ViewPartPersistentItem() : part(NULL) {}
	ViewPartPersistentItem(const std::string& iName, ViewPart* iPart) :
		name(iName), part(iPart) {}

	std::string name;
	ViewPart*   part;

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		try {
			r.member("Name", name, "");
			r.member("Part", part, "");
		}
		catch(Exception& ex) {
			MIRA_LOG(ERROR) << "Error loading view part '" << name << "': " << ex.what();
			part = NULL;
		}
	}
};

struct EditorPartPersistentItem
{
	EditorPartPersistentItem() : part(NULL), activated(false),
	                             minimized(false), maximized(false), docked(false) {}
	EditorPartPersistentItem(const std::string& iName, EditorPart* iPart,
	                         const Rect2i& iGeometry,
	                         bool iActivated, bool iMinimized, bool iMaximized,
	                         bool iDocked, bool iOnTop) :
		name(iName), part(iPart), geometry(iGeometry),
		activated(iActivated), minimized(iMinimized), maximized(iMaximized), docked(iDocked), onTop(iOnTop) {}

	std::string name;
	EditorPart* part;
	Rect2i      geometry;  // position and orientation of the editor
	bool        activated;
	bool        minimized;
	bool        maximized;
	bool        docked;
	bool        onTop;

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		try {
			r.member("Name", name, "");
			r.member("Part", part, "");
			r.member("Geometry", geometry, "");
			r.member("Activated", activated, "", false);
			r.member("Minimized", minimized, "", false);
			r.member("Maximized", maximized, "", false);
			r.member("Docked", docked, "", true);
			r.member("OnTop", onTop, "", false);
		}
		catch(Exception& ex) {
			MIRA_LOG(ERROR) << "Error loading editor part '" << name << "': " << ex.what();
			part = NULL;
		}
	}

};

///////////////////////////////////////////////////////////////////////////////

QWidget* Workbench::restorePart(ClassProxy partClass, const XMLDom& xml)
{
	XMLDeserializer s(xml);

	if(partClass.isDerivedFrom(&ViewPart::CLASS())) {
		// if we are trying to create a new view, first check if we have
		// created it already:
		auto it = mCreatedViews.find(partClass.getIdentifier());
		if(it!=mCreatedViews.end()) {
			it->second->show();
			it->second->activate();
			storeCurrentPerspective();
			// use the exisiting view
			return it->second;
		}

		ViewPartPersistentItem viewPart;
		s.deserialize("ViewPart", viewPart);

		if (!viewPart.part)
			MIRA_THROW(XInvalidWorkspace, "Error restoring view part.");

		//viewPart.part->setParent(this);
		setupNewViewPart(viewPart.part);
		viewPart.part->show();

		storeCurrentPerspective();
		return viewPart.part;
	}

	EditorPartPersistentItem editorPart;
	s.deserialize("EditorPart", editorPart);

	if (!editorPart.part)
		MIRA_THROW(XInvalidWorkspace, "Error restoring editor part.");

	//editorPart.part->setParent(this);
	setupNewEditorPart(editorPart.part);
	editorPart.part->setGeometry(editorPart.geometry.x0(), editorPart.geometry.y0(),
	                             editorPart.geometry.width(), editorPart.geometry.height());
	editorPart.part->show();

	// restore docked and maximized state
	editorPart.docked ? editorPart.part->dock() : editorPart.part->undock();
	editorPart.minimized ? editorPart.part->showMinimized() :
	editorPart.maximized ? editorPart.part->showMaximized() : editorPart.part->showNormal();
	editorPart.part->setOnTop(editorPart.onTop);
	
	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
	assert(central!=NULL && "The central widget must be of type EditorPartArea!");
	central->activateSubWindow(editorPart.part);

	storeCurrentPerspective();
	return editorPart.part;
}

///////////////////////////////////////////////////////////////////////////////

void Workbench::clearWorkspaceParts()
{
	while(!mViewParts.empty())
		delete mViewParts.front();
	mViewParts.clear();
	mCreatedViews.clear();

	while(!mEditorParts.empty())
		delete mEditorParts.front();
	mEditorParts.clear();

	mActiveEditorPart=NULL;
}

void Workbench::storeCurrentPerspective()
{
	if(mCurrentPerspective!=NULL)
		storeCurrentPerspective(mCurrentPerspective);
}

void Workbench::storeCurrentPerspective(Perspective* perspective)
{
	perspective->mState.clear();

	XMLSerializer s(perspective->mState);

	// serialize the view parts
	std::list<ViewPartPersistentItem> viewParts;
	foreach(ViewPart* v, mViewParts)
	{
		//if(v->isHidden())
		//	continue;

		viewParts.push_back(ViewPartPersistentItem(v->windowTitle().toLocal8Bit().data(), v));
	}
	s.serialize("Views", viewParts);

	// serialize the editor parts
	std::list<EditorPartPersistentItem> editorParts;
	
	// first add tabbed window in tab order, to make tab order persistent
	std::vector<EditorPartWindow*> tabbedWindows;
	EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
 	if (central) {
 		tabbedWindows = central->getTabbedWindows();
		foreach(EditorPartWindow* w, tabbedWindows)
		{
			EditorPart* e = dynamic_cast<EditorPart*>(w);
			if (!e)
				continue;
			Rect2i geometry(e->geometry().x(), e->geometry().y(),
			                e->geometry().width(), e->geometry().height());
			editorParts.emplace_back(e->windowTitle().toLocal8Bit().data(),
			                         e, geometry,
			                         e->isActivated(), e->isMinimized(), e->isMaximized(),
			                         e->isDocked(), e->isOnTop());
		}
	}
 	
	foreach(EditorPart* e, mEditorParts)
	{
		// tabbed windows already added above
		if (std::find(tabbedWindows.begin(), tabbedWindows.end(), (EditorPartWindow*)e) != tabbedWindows.end())
			continue;

		Rect2i geometry(e->geometry().x(), e->geometry().y(),
		                e->geometry().width(), e->geometry().height());
		editorParts.emplace_back(e->windowTitle().toLocal8Bit().data(),
		                         e, geometry,
		                         e->isActivated(), e->isMinimized(), e->isMaximized(),
		                         e->isDocked(), e->isOnTop());
	}
	s.serialize("Editors", editorParts);

	// store state of tool bars and dock widget positions
	QByteArray stateArray = saveState(0);
	std::string stateArrayStr = QString(stateArray.toBase64()).toStdString();
	s.serialize("State", stateArrayStr);
}

bool Workbench::restorePerspective(Perspective* perspective)
{
	clearWorkspaceParts(); // clean all existing parts

	XMLDeserializer s(perspective->mState);

	// deserialize the view parts
	std::list<ViewPartPersistentItem> viewParts;
	s.deserialize("Views", viewParts);

	blockSignals(true);
	setUpdatesEnabled(false);

	// activating views in subsequent steps may change the window size (if small),
	// we store it to reset it at the end of this method
	QRect geo = geometry();

	bool complete = true;

	// restore each view
	foreach(const ViewPartPersistentItem& i, viewParts)
	{
		if (!i.part) {
			complete = false;
			continue;
		}

		MIRA_LOG(DEBUG) << "Restoring ViewPart " <<
			i.part->getClass().getIdentifier() << ", " << i.name;
		setupNewViewPart(i.part);
		i.part->show();
	}

	std::list<EditorPartPersistentItem> editorParts;
	s.deserialize("Editors", editorParts);

	EditorPart* activeEditor = NULL;
	foreach(const EditorPartPersistentItem& i, editorParts)
	{
		if (!i.part) {
			complete = false;
			continue;
		}

		MIRA_LOG(DEBUG) << "Restoring EditorPart " <<
			i.part->getClass().getIdentifier() << ", " << i.name;
		setupNewEditorPart(i.part);
		i.part->setGeometry(i.geometry.x0(), i.geometry.y0(),
		                    i.geometry.width(), i.geometry.height());
		i.part->show();

		// restore docked and maximized state
		i.docked ? i.part->dock() : i.part->undock();
		i.minimized ? i.part->showMinimized() :
				i.maximized ? i.part->showMaximized() : i.part->showNormal();
		if(i.activated)
			activeEditor = i.part;
		i.part->setOnTop(i.onTop);
	}

	if(activeEditor) {
		EditorPartArea* central = dynamic_cast<EditorPartArea*>(centralWidget());
		assert(central!=NULL && "The central widget must be of type EditorPartArea!");
		central->activateSubWindow(activeEditor);
	}

	// restore state of tool bars and dock widget positions
	std::string stateArrayStr;
	s.deserialize("State", stateArrayStr);
	QByteArray stateArray= QByteArray::fromBase64(QByteArray(stateArrayStr.c_str()));
	restoreState(stateArray,0);

	mMinimizedDockAreas[Qt::LeftDockWidgetArea].dock->hide();
	mMinimizedDockAreas[Qt::RightDockWidgetArea].dock->hide();
	mMinimizedDockAreas[Qt::TopDockWidgetArea].dock->hide();
	mMinimizedDockAreas[Qt::BottomDockWidgetArea].dock->hide();

	setGeometry(geo);

	foreach(const ViewPart* v, mViewParts)
	{
		if(v->isHidden())
			minimizeDockArea(v->getDockArea());
	}

	setUpdatesEnabled(true);
	blockSignals(false);

	return complete;
}

///////////////////////////////////////////////////////////////////////////////

void Workbench::reflect(XMLSerializer& r)
{
	storeCurrentPerspective(); // make sure we have synchronized the current perspective

	// store the perspectives
	r.member("Perspectives", mPerspectives, "");

	// get and store the index of the current active perspective
	int index = 0;
	foreach(PerspectivePtr p, mPerspectives)
	{
		if(p.get()==mCurrentPerspective)
			break;
		++index;
	}
	r.member("CurrentPerspective", index, "");

	// store our windows geometry
	QByteArray geometryArray = saveGeometry();
	std::string geometryArrayStr = QString(geometryArray.toBase64()).toStdString();
	r.member("Geometry", geometryArrayStr, "Binary Qt state, do not edit!");
}

void Workbench::reflect(XMLDeserializer& r)
{
	// delete our current perspectives
	cleanupWorkspace();

	// restore our windows geometry
	std::string geometryArrayStr;
	r.member("Geometry", geometryArrayStr, "Binary Qt state, do not edit!");
	QByteArray geometryArray = QByteArray::fromBase64(QByteArray(geometryArrayStr.c_str()));
	restoreGeometry(geometryArray);

	// QT WORKAROUND:
	// If we want to restore a maximized state, we have to make sure, that
	// our Workbench widget is fully visible and has established its maximized
	// state already before we can continue to restore the workbench elements.
	// If we do not wait here, the sizes of workbench elements and the active
	// editor will not be restored correctly for some reason.
	this->show();
	for(int c=0; c<10000; ++c)
		QApplication::processEvents();

	// restore our perspectives
	r.member("Perspectives", mPerspectives, "");

	// fill the PerspectiveTabWidget
	for(auto it=mPerspectives.begin(); it!=mPerspectives.end(); ++it)
	{
		Perspective* perspective = it->get();
		mPerspectiveTabWidget->addPerspective(perspective);
	}

	// restore the current active perspective
	int index;
	r.member("CurrentPerspective", index, "");

	if(!mPerspectives.empty())
	{
		auto it=mPerspectives.begin();
		for(int i=0; i<index; ++i, ++it) {
			assert(it!=mPerspectives.end());
		}
		MIRA_LOG(DEBUG) << "Activating the perspective: " << index;
		mPerspectiveTabWidget->activatePerspective(it->get());
		if (!onChangedPerspective(it->get()))
			MIRA_THROW(XInvalidWorkspace, "Errors caught while loading workspace.");
	}
}

void Workbench::saveWorkspace(const std::string& file)
{
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("Workbench", *this);
	xml.saveToFile(file);
}

void Workbench::loadWorkspace(const std::string& file)
{
	if(file.empty())
		return;

	try {
		XMLDom xml;
		xml.loadFromFile(file);
		XMLDeserializer s(xml);
		s.deserialize("Workbench", *this);
	} catch(XInvalidWorkspace& ex) {
		MIRA_LOG(WARNING) << "Errors caught while loading workspace";
		throw; // rethrow
	} catch(...) {
		// catch all errors and clean up the dirt that might have been
		// created
		MIRA_LOG(WARNING) << "Failed to load workspace, cleaning up";
		cleanupWorkspace();
		throw; // rethrow
	}

	MIRA_LOG(DEBUG) << "Workspace successfully loaded";
}

void Workbench::cleanupWorkspace()
{
	MIRA_LOG(DEBUG) << "Cleanup workspace" << std::endl;

	while(!mPerspectives.empty())
		removePerspective(mPerspectives.front().get());

	mCurrentPerspective = NULL;

	MIRA_LOG(DEBUG) << "Cleaned up workspace" << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

QMenu* Workbench::createPopupMenu()
{
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////

Workbench::Perspective* Workbench::addPerspective(std::string name)
{
	storeCurrentPerspective();

	// clean up existing parts and prepare an empty perspective
	clearWorkspaceParts();
	mCurrentPerspective=NULL;

	mPerspectives.push_back(PerspectivePtr(new Perspective(name)));
	Perspective* perspective = mPerspectives.back().get();
	storeCurrentPerspective(perspective); // sync the empty perspective
	mPerspectiveTabWidget->addPerspective(perspective);
	mPerspectiveTabWidget->activatePerspective(perspective);
	mCurrentPerspective=perspective;
	return perspective;
}

void Workbench::removePerspective(Workbench::Perspective* perspective)
{
	for(auto it=mPerspectives.begin(); it!=mPerspectives.end(); ++it)
	{
		if(it->get() == perspective) {
			mPerspectives.erase(it);
			break;
		}
	}

	if(perspective==mCurrentPerspective) // we removed the current shown perspective
		mCurrentPerspective = NULL;

	mPerspectiveTabWidget->removePerspective(perspective);
}

void Workbench::activatePerspective(Perspective* perspective)
{
	mPerspectiveTabWidget->activatePerspective(perspective);
}

bool Workbench::onChangedPerspective(Workbench::Perspective* perspective)
{
	// save old perspective
	storeCurrentPerspective();
	bool complete = restorePerspective(perspective);
	mCurrentPerspective = perspective;
	return complete;
}

void Workbench::onMovedPerspective(int from, int to)
{
	assert(from>=0 && to>=0);

	auto fromIt = mPerspectives.begin();
	for(int i=0; i<from; ++i, ++fromIt) {
		assert(fromIt!=mPerspectives.end());
	}

	auto toIt = mPerspectives.begin();
	for(int i=0; i<to; ++i, ++toIt){
		assert(toIt!=mPerspectives.end());
	}

	std::swap(*fromIt, *toIt);
}

void Workbench::onEditorCreated(EditorPart* editor)
{
}

void Workbench::onEditorDestroyed(EditorPart* editor)
{
	// remove the editor from our list
	mEditorParts.remove(editor);

	foreach(IPartListener* l, mPartListeners)
		l->editorClosed(editor);

	MIRA_LOG(DEBUG) << "Editor removed";

	if(editor == mActiveEditorPart)
		mActiveEditorPart = NULL;
}

void Workbench::onViewDestroyed(ViewPart* view)
{
	// remove the editor from our list
	mViewParts.remove(view);

	for(auto it=mCreatedViews.begin(); it!=mCreatedViews.end(); ++it)
	{
		if(it->second == view) {
			mCreatedViews.erase(it);
			break;
		}
	}

	foreach(IPartListener* l, mPartListeners)
		l->viewClosed(view);

	MIRA_LOG(DEBUG) << "View removed";
}

void Workbench::onViewActivated(ViewPart* view)
{
	foreach(EditorPart* p, mEditorParts)
		if(p->isActivated())
			p->deactivate();

	foreach(ViewPart* p, mViewParts)
		if(p->isActivated())
			p->deactivate();

	view->activate();

	foreach(IPartListener* l, mPartListeners)
		l->viewActivated(view);
}

void Workbench::onViewMinimizedRequested(ViewPart* view)
{
	minimizeDockArea(view->getDockArea());
}

///////////////////////////////////////////////////////////////////////////////

void Workbench::addPartListener(IPartListener* listener) {
	mPartListeners.insert(listener);
}

void Workbench::removePartListener(IPartListener* listener) {
	mPartListeners.erase(listener);
}

IWorkbenchPart* Workbench::getActiveEditor()
{
	return mActiveEditorPart;
}

std::list<IWorkbenchPart*> Workbench::getEditors(bool backToFront)
{
	std::list<IWorkbenchPart*> l;

	if (!backToFront) {
		foreach(EditorPart* e, mEditorParts)
			l.push_back(e);
		return l;
	}

	// find the parent of all EditorPartWindows (EditorPartArea -> ScrollArea -> ViewPort)
	QObject* parent = NULL;
	foreach(EditorPart* e, mEditorParts) {
		if (e->parent()) {
			parent = e->parent();
			break;
		}
	}
	if (!parent)
		return l;

	// QObject::children() will return the child widgets in their stack order
	// https://doc.qt.io/qt-5/qobject.html#children
	QObjectList children = parent->children();
	foreach (QObject* object, children) {
		EditorPartWindow* child = qobject_cast<EditorPartWindow*>(object);
		if (!child || !child->isDocked())
			continue;

		EditorPart* editor = dynamic_cast<EditorPart*>(child);
		if (editor)
			l.push_back(editor);
	}

	return l;
}

void Workbench::onWindowTitleChanged(EditorPartWindow* editor)
{
}

void Workbench::onEditorWindowActivated(EditorPartWindow* window)
{
	IWorkbenchPart* part = dynamic_cast<IWorkbenchPart*>(window);
	if(part==NULL)
		return;

	foreach(EditorPart* p, mEditorParts)
		if(p->isActivated())
			p->deactivate();

	foreach(ViewPart* p, mViewParts)
		if(p->isActivated())
			p->deactivate();

	part->activate();

	foreach(IPartListener* l, mPartListeners)
		l->editorActivated(part);

	mActiveEditorPart = part;

}

///////////////////////////////////////////////////////////////////////////////

bool Workbench::isDockAreaMinimized(Qt::DockWidgetArea area)
{
	if(area==Qt::NoDockWidgetArea)
		return false;

	return mMinimizedDockAreas[area].minimized;
}

void Workbench::minimizeDockArea(Qt::DockWidgetArea area)
{
	if(area==Qt::NoDockWidgetArea)
		return;

	QBoxLayout* layout = mMinimizedDockAreas[area].layout;
	QWidget* widget = mMinimizedDockAreas[area].widget;

	// remove all old widgets
	while(layout->count()>0) {
		QWidget* w = layout->itemAt(0)->widget();
		mDockRestoreMapper->removeMappings(w);
		layout->removeItem(layout->itemAt(0));
		delete w;
	}

	// save the states of all dock widgets:
	foreach(ViewPart* p, mViewParts)
	{
		if(p->getDockArea()==area) {
			MinimizedDockWidgetState state;
			state.size = p->geometry().size();
			state.visible = p->isVisible();
			mMinimizedDockWidgetStates[p] = state;
		}
	}

	bool areaIsEmpty = true;
	foreach(ViewPart* p, mViewParts)
	{
		if(p->getDockArea()==area) {
			p->hide();

			QPushButton* btn = new MinimizedDockWidgetButton(p->windowTitle(), widget, area);
			connect(btn, SIGNAL(clicked()), mDockRestoreMapper, SLOT(map()));
			mDockRestoreMapper->setMapping(btn, p);
			layout->addWidget(btn);
			areaIsEmpty = false;
		}
	}

	layout->addStretch();

	if(!areaIsEmpty) {
		addDockWidget(area, mMinimizedDockAreas[area].dock);
		mMinimizedDockAreas[area].dock->show();
	} else {
		removeDockWidget(mMinimizedDockAreas[area].dock);
		mMinimizedDockAreas[area].dock->hide();
	}

	mMinimizedDockAreas[area].minimized = true;
}

void Workbench::onViewDockLocationChanged(Qt::DockWidgetArea area)
{
	if(area==Qt::NoDockWidgetArea)
		return;

	if(mMinimizedDockAreas[area].minimized)
		minimizeDockArea(area);
}

void Workbench::restoreDockArea(Qt::DockWidgetArea area)
{
	removeDockWidget(mMinimizedDockAreas[area].dock);
	mMinimizedDockAreas[area].dock->hide();

	//foreach(ViewPart* p, mViewParts)
	//	if(p->getDockArea()==area)
	//		p->show();

	mMinimizedDockAreas[area].minimized = false;


	// restore states of dockwidgets:

	// Qt resize hack:
	//    see http://stackoverflow.com/questions/2722939/c-resize-a-docked-qt-qdockwidget-programmatically
	foreach(ViewPart* p, mViewParts)
	{
		if(p->getDockArea()!=area)
			continue;

		auto it = mMinimizedDockWidgetStates.find(p);
		if(it==mMinimizedDockWidgetStates.end())
			continue;

		MinimizedDockWidgetState state = it->second;
		QSize oldMinimumSize = p->minimumSize();

		// limit the size only in the direction that is affected by the dock area
		if(area==Qt::LeftDockWidgetArea ||
		   area==Qt::RightDockWidgetArea)
			state.size.setHeight(0);
		if(area==Qt::TopDockWidgetArea ||
		   area==Qt::BottomDockWidgetArea)
			state.size.setWidth(0);
		p->setMinimumSize(state.size);
		p->show();
		qApp->processEvents();
		p->setMinimumSize(oldMinimumSize);
	}

	// restore active tab
	foreach(ViewPart* p, mViewParts)
	{
		if(p->getDockArea()!=area)
			continue;

		auto it = mMinimizedDockWidgetStates.find(p);
		if(it==mMinimizedDockWidgetStates.end())
			continue;

		const MinimizedDockWidgetState& state = it->second;
		if(state.visible) {
			p->setFocus();
			p->raise();
		}
	}
}

void Workbench::onViewRestore(QWidget* view)
{
	ViewPart* v = dynamic_cast<ViewPart*>(view);

	if(v==NULL)
		return;

	restoreDockArea(v->getDockArea());
	v->setFocus();
	v->raise();
}

///////////////////////////////////////////////////////////////////////////////

void Workbench::storePartMemory(ViewPart* part)
{
	XMLDomPtr& entry = mPartMemory[part->getClass().getIdentifier()];
	if (entry)
		entry->clear();
	else
		entry = boost::make_shared<XMLDom>();

	XMLSerializer s(*entry);

	s.serialize("ViewPart", ViewPartPersistentItem(part->windowTitle().toLocal8Bit().data(), part));

	savePartMemory();
}

void Workbench::storePartMemory(EditorPart* part)
{
	XMLDomPtr& entry = mPartMemory[part->getClass().getIdentifier()];
	if (entry)
		entry->clear();
	else
		entry = boost::make_shared<XMLDom>();

	XMLSerializer s(*entry);

	s.serialize("EditorPart",
	            EditorPartPersistentItem(part->windowTitle().toLocal8Bit().data(),
	                                     part,
	                                     Rect2i(part->geometry().x(), part->geometry().y(),
	                                            part->geometry().width(), part->geometry().height()),
	                                     part->isActivated(), part->isMinimized(), part->isMaximized(),
	                                     part->isDocked(), part->isOnTop()));

	savePartMemory();
}

void Workbench::savePartMemory()
{
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("PartMemory", mPartMemory);

	try {
		xml.saveToFile(getAppDataDirectory() / PARTMEMORY_FILENAME);
	} catch(XRuntime&) {
		xml.saveToFile(PARTMEMORY_FILENAME);
	}
}

void Workbench::loadPartMemory()
{
	try {
		XMLDom xml;
		try {
			xml.loadFromFile(getAppDataDirectory() / PARTMEMORY_FILENAME);
		} catch(XRuntime&) {
			xml.loadFromFile(PARTMEMORY_FILENAME);
		}
		XMLDeserializer s(xml);
		s.deserialize("PartMemory", mPartMemory);
	} catch(...) {
		MIRA_LOG(WARNING) << "Failed to load part memory, cannot offer 'Restore Last' view option";
	}
}
///////////////////////////////////////////////////////////////////////////////

}
