/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PerspectiveTabWidget.C
 *    Implementation of PerspectiveTabWidget.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/15
 */

#include <rcp/PerspectiveTabWidget.h>

#include <QMouseEvent>
#include <QMenuBar>
#include <QHBoxLayout>
#include <QTabBar>
#include <QPushButton>

#include <QInputDialog>

#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PerspectiveTabBar::PerspectiveTabBar(QWidget* parent) :
	QTabBar(parent), mCurrentIndexForMenu(-1)
{
	mPopupMenu = new QMenu(this);
	mPopupMenu->addAction(tr("Edit ..."), this, SLOT(menuEdit()));
	mPopupMenu->addSeparator();
	mDeleteAction = mPopupMenu->addAction(tr("Delete"), this, SLOT(menuDelete()));
}

void PerspectiveTabBar::mousePressEvent(QMouseEvent* event)
{
	if(event->button()==Qt::RightButton) {
		int index = tabAt(event->pos());
		if(index>=0) {
			mCurrentIndexForMenu = index;
			mDeleteAction->setEnabled(count() > 1);
			mPopupMenu->popup(event->globalPos());
			return;
		}
	}
	QTabBar::mousePressEvent(event);
}

void PerspectiveTabBar::menuDelete()
{
	assert(mCurrentIndexForMenu>=0);
	emit tabCloseRequested(mCurrentIndexForMenu);
}

void PerspectiveTabBar::menuEdit()
{
	assert(mCurrentIndexForMenu>=0);
	emit tabEditRequested(mCurrentIndexForMenu);
}

///////////////////////////////////////////////////////////////////////////////
#define TAB_HEIGHT 22
#define TAB_HEIGHT_STR_(x) #x
#define TAB_HEIGHT_STR__(x) TAB_HEIGHT_STR_(x)
#define TAB_HEIGHT_STR TAB_HEIGHT_STR__(TAB_HEIGHT)

PerspectiveTabWidget::PerspectiveTabWidget(QMenuBar* menuBar) :
	QWidget(menuBar), mMenuBar(menuBar)
{
	QHBoxLayout* hLayout = new QHBoxLayout(this);
	hLayout->setAlignment ( Qt::AlignRight );
	hLayout->setMargin(0);
	hLayout->setSpacing(0);

	mTabBar = new PerspectiveTabBar(this);
	mTabBar->setShape(QTabBar::RoundedNorth);
	mTabBar->setDocumentMode(true);
	mTabBar->setExpanding(false);
	//mTabBar->setLayoutDirection(Qt::RightToLeft);
    mTabBar->setDrawBase (false);
	mTabBar->setFocusPolicy(Qt::NoFocus);
	mTabBar->setMovable (true);
	mTabBar->setStyleSheet("QTabBar::tab { height: " TAB_HEIGHT_STR "px; }");
	hLayout->addWidget(mTabBar);

	mTabButton = new QPushButton(QIcon(":/icons/Plus.png"), QString(), this);
	mTabButton->setFlat(true);
	mTabButton->setFixedWidth(20);
	mTabButton->setFixedHeight(20);
	mTabButton->setToolTip(tr("Add a new perspective"));
	hLayout->addWidget(mTabButton);

	menuBar->setCornerWidget(this, Qt::TopRightCorner);

	connect(mTabButton, SIGNAL(pressed()), this, SLOT(addPerspective()));
	connect(mTabBar, SIGNAL(currentChanged(int)), this, SLOT(tabCurrentChanged(int)));
	connect(mTabBar, SIGNAL(tabCloseRequested(int)), this, SLOT(tabCloseRequested(int)));
	connect(mTabBar, SIGNAL(tabEditRequested(int)), this, SLOT(tabEditRequested(int)));
	connect(mTabBar, SIGNAL(tabMoved(int,int)), this, SLOT(tabMoved(int,int)));
}

QSize PerspectiveTabWidget::sizeHint() const
{
	static bool zeroSizeHint = false;
	if ( zeroSizeHint )
		return QSize();

	zeroSizeHint = true;
	int available = parentWidget()->width() - parentWidget()->sizeHint().width() - 20;
	zeroSizeHint = false;

	return QSize(available, std::max(mTabBar->sizeHint().height(),TAB_HEIGHT+1));
}

///////////////////////////////////////////////////////////////////////////////

int PerspectiveTabWidget::toIndex(Perspective* perspective)
{
	for(int i=0; i<mTabBar->count(); ++i)
	{
		Perspective* p = toPerspective(i);
		if(p==perspective)
			return i;
	}
	return -1;
}

Workbench::Perspective* PerspectiveTabWidget::toPerspective(int index)
{
	QVariant v = mTabBar->tabData(index);
	Perspective* p = (Perspective*) v.value<void*>();
	return p;
}

void PerspectiveTabWidget::addPerspective(Perspective* perspective)
{
	int index = mTabBar->addTab(perspective->getName().c_str());
	QVariant v = QVariant::fromValue((void*) perspective);
	mTabBar->setTabData(index, v);
}

void PerspectiveTabWidget::removePerspective(Perspective* perspective)
{
	int index = toIndex(perspective);
	assert(index>=0);
	mTabBar->removeTab(index);
}

void PerspectiveTabWidget::activatePerspective(Perspective* perspective)
{
	int index = toIndex(perspective);
	assert(index>=0);
	mTabBar->setCurrentIndex(index);
}

void PerspectiveTabWidget::addPerspective()
{
	QString name = QInputDialog::getText(this, tr("Add Perspective"), tr("Name"),
	                                     QLineEdit::Normal, tr("New Perspective"));
	if(name.isEmpty() || name.isNull())
		return;

	emit onAddPerspectiveRequested(name.toLocal8Bit().data());
}

void PerspectiveTabWidget::tabEditRequested(int index)
{
	QString name = QInputDialog::getText(this, tr("Add Perspective"), tr("Name"),
	                                     QLineEdit::Normal, mTabBar->tabText(index));
	if(name.isEmpty() || name.isNull())
		return;

	mTabBar->setTabText(index, name);

	Perspective* p = toPerspective(index);
	assert(p!=NULL);
	p->setName(name.toLocal8Bit().data());
	emit renamedPerspective(p, name);

}

void PerspectiveTabWidget::tabCurrentChanged(int index)
{
	Perspective* p = toPerspective(index);
	if(p!=NULL) // p is null for newly added tabs (will call changedPerspective in addPerspective)
		emit changedPerspective(p);
}

void PerspectiveTabWidget::tabCloseRequested(int index)
{
	QVariant v = mTabBar->tabData(index);
	Perspective* p = (Perspective*) v.value<void*>();
	assert(p!=NULL);
	emit onRemovedPerspectiveRequested(p);
}

void PerspectiveTabWidget::tabMoved(int from, int to)
{
	emit movedPerspective(from,to);
}

///////////////////////////////////////////////////////////////////////////////

}
