/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ScrollView.h
 *    A abstract base class for scroll views.
 *
 * @author Tim Langner
 * @date   2011/12/30
 */

#ifndef _MIRA_SCROLLVIEW_H_
#define _MIRA_SCROLLVIEW_H_

#include <QAbstractScrollArea>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * ScrollView extends the QAbstractScrollArea by features like automatic
 * scaling of scrollbars, signals when the visible are was moved and so on.
 * It therefore defines a content area (rect) within a scroll viewport
 * and provides several functions to modify this area.
 * For example it allows synchronization of multiple scroll views showing the
 * same data (in a different way) by connecting the contentsMoving signal
 * of one view to a slot  that calls setContentsPos with the respective
 * coordinates in another view.
 */
class ScrollView : public QAbstractScrollArea
{
	Q_OBJECT
public:

	ScrollView(QWidget* parent);

	/// The x coordinate of the content area
	int contentsX()      const { return mContentsRect.x(); }
	/// The y coordinate of the content area
	int contentsY()      const { return mContentsRect.y(); }
	/// The height of the content area
	int contentsHeight() const { return mContentsRect.height(); }
	/// The width of the content area
	int contentsWidth()  const { return mContentsRect.width(); }

	/// Set the position of the content area
	void setContentsPos(int x, int y);
	/// Resize the content area
	void resizeContents(int width, int height);

	/**
	 * Ensures that the position given by x and y with the specified margins
	 * is within the content area by scrolling to the respective position.
	 * It is not guaranteed that the margins will be strictly adhered.
	 * E.g. if x < marginX or y + marginY > viewport.heigth().
	 */
	void ensureVisible(int x, int y, int marginX = 50, int marginY = 50);

	/// Converts viewport coordinates to content area coordinates
	QPoint viewportToContents(const QPoint& pos) const;
	/// Converts content area coordinates to viewport coordinates
	QPoint contentsToViewport(const QPoint& pos) const;

signals:

	/// Signal emitted when the position of the content area changed
	void contentsMoving(int x, int y);

protected:

	void updateScrollBars();
	virtual void scrollContentsBy(int dX, int dY);

	virtual void resizeEvent(QResizeEvent* resizeEvent);
	virtual void paintEvent(QPaintEvent* paintEvent);
	virtual void wheelEvent(QWheelEvent* wheelEvent);

	/**
	 * Overload this method in derived classes to draw the content of the
	 * content area
	 */
	virtual void drawContents(QPainter* painter, const QRect& rect) = 0;

	/**
	 * Call this method to schedule an update of the content area for
	 * the given rect.
	 * Can be overwritten in derived class to do additional update work.
	 */
	virtual void updateContents(const QRect& rect);
	/**
	 * Call this method to schedule an update of the content area.
	 * Can be overwritten in derived class to do additional update work.
	 */
	virtual void updateContents();

	QRect mContentsRect;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
