/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationRegistry.h
 *    Registry that stores mappings from type names to
 *    Visualization classes of Visualizations that are able
 *    to show the data.
 *
 * @author Erik Einhorn
 * @date   2011/01/02
 */

#ifndef _MIRA_VISUALIZATIONREGISTRY_H_
#define _MIRA_VISUALIZATIONREGISTRY_H_

#include <set>
#include <map>

#include <factory/Factory.h>

#include <platform/Typename.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * Stores mappings from type names to Visualization classes of Visualizations
 * that are able to show the data.
 * When a visualization to a yet unknown type is requested, the
 * registry performs an indexing. To do so, it tries to instantiate all
 * available visualization classes that are newly available since the last
 * indexing step. For each instantiated visualization, it queries the
 * important information and destroys the instances afterwards.
 * Note that this indexing usually will be done only once after the process
 * queries a visualization for the first time. Hence, the overhead caused by
 * the instantiation of all available visualizations is small.
 *
 */
class VisualizationRegistry :
	public LazySingleton<VisualizationRegistry>
{
public:

	struct Entry
	{
		Entry(const Typename& iType, ClassProxy iVisClass):
			type(iType), visClass(iVisClass) {}

		Typename    type;     ///< the data type this visualization is made for
		ClassProxy    visClass; ///< the class of this visualization
	};

	std::list<ClassProxy> getVisualizationsFor(const Typename& type, const Class& visualizationType);

private:

	void lookForNewVisualizations();

private:

	std::set<ClassProxy> mKnownVisualizationsClasses;
	std::multimap<Typename, Entry> mKnownVisualizations;

};

//////////////////////////////////////////////////////////////////////////////

}

#endif
