/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeRecorderWidget.h
 *    Widget for recording tapes.
 *
 * @author Tim Langner
 * @date   2012/03/02
 */

#ifndef _MIRA_TAPERECORDERWIDGET_H_
#define _MIRA_TAPERECORDERWIDGET_H_

#include <QTimer>
#include <QDialog>
#include <QDialogButtonBox>
#include <QLabel>
#include <QPushButton>
#include <QToolButton>
#include <QTableWidget>
#include <QTimeEdit>
#include <QWidget>
#include <QCheckBox>

#include <serialization/adapters/std/list>

#include <fw/TapeRecorder.h>

#include <widgets/PropertyEditor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A dialog for adding, removing and editing codecs
 */
class CodecDialog : public QDialog
{
	Q_OBJECT

public:

	CodecDialog(QWidget* parent = NULL);
	~CodecDialog();

	void setCodecs(std::list<BinarySerializerCodecPtr>& codecs);
	std::list<BinarySerializerCodecPtr> getCodecs();

public slots:

	void addRemove();

protected:

	void addCodec(BinarySerializerCodecPtr codec);

	std::map<std::string, std::pair<BinarySerializerCodecPtr, boost::shared_ptr<PropertyNode>>> mCodecs;
	PropertyEditor* mEditor;
	QDialogButtonBox* mButtonBox;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * Widget for displaying used codecs and changing them via a dialog
 */
class CodecWidget : public QWidget
{
	Q_OBJECT
public:

	CodecWidget(QWidget* parent=NULL);

	void setCodecs(const std::list<BinarySerializerCodecPtr>& codecs);
	std::list<BinarySerializerCodecPtr> getCodecs();

public slots:

	void selectCodecs();

protected:

	std::list<BinarySerializerCodecPtr> mCodecs;
	QLabel* mTextLbl;
};

//////////////////////////////////////////////////////////////////////////////

/**
 * A dialog for selecting a duration
 */
class DurationDialog : public QDialog
{
	Q_OBJECT

public:

	DurationDialog(QWidget* parent = NULL);

	void setDuration(const Duration& duration);
	Duration getDuration();

protected:

	QTimeEdit* mTimeEdit;
	QDialogButtonBox* mButtonBox;
};


//////////////////////////////////////////////////////////////////////////////

/**
 * The recorder widget
 */
class TapeRecorderWidget : public QWidget
{
	Q_OBJECT

public:

	struct RecordingConfig {

		template <typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Channels", mChannels,
			         "Channels to record", std::map<std::string, RecordedChannelInfo>());
			r.member("RecordTransforms", mRecordTransforms,
			         "Record transform frames to tape", true);
			r.member("SaveTransformsFile", mSaveTransformsFile,
			         "Save transform tree to extra file", true);
		}

		std::map<std::string, RecordedChannelInfo> mChannels;
		bool mRecordTransforms;
		bool mSaveTransformsFile;
	};

public:

	TapeRecorderWidget(QWidget* parent = NULL);

	void addChannel(const std::string& channelID, const RecordedChannelInfo& codecs=RecordedChannelInfo());
	void recordTransforms(bool recordTransformChannels = true, bool saveTransformsFile = true);
	RecordingConfig getRecordingConfig();

signals:
	void recordingStarted();

public slots:

	void record();
	void recordDuration();
	void stop();
	void check();
	void saveTransformsToggled(bool on);
	void add();
	void remove();
	void removeAll();
	void updateStats();

protected:

	int findButton(int column);
	virtual void dragEnterEvent(QDragEnterEvent *event);
	virtual void dropEvent(QDropEvent *event);
	void saveTransformTree(const Path& file);

protected:

	boost::shared_ptr<TapeRecorder> mRecorder;
	std::set<std::string> mRecordedChannels;

	QTableWidget* mChannels;
	Time mStartTime;
	QLabel* mInfo;
	QPushButton* mBtAdd;
	QPushButton* mBtRemove;
	QToolButton* mBtRecord;
	QToolButton* mBtRecordDuration;
	QToolButton* mBtStop;
	QCheckBox* mChkSvTransformsFile;
	QCheckBox* mChkRecTransformsChans;

	QTimer mCheckTimer;
	QTimer mStatsTimer;
	QTimer mAutoStopTimer;

	Duration mRecordDuration;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
