/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ResourceName.C
 *    Implementation of ResourceName.h.
 *
 * @author Erik Einhorn
 * @date   2010/09/16
 */

#include <fw/ResourceName.h>

#include <locale>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/split.hpp>

#include <utils/Foreach.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ResourceName ResourceName::parent() const
{
	if(isRoot())
		return *this;

	std::string::size_type pos = mName.rfind("/");

	if(pos == std::string::npos)
		return ResourceName();

	if(pos==0)
		return ResourceName("/");

	return ResourceName(mName.substr(0,pos));
}

std::string ResourceName::leaf() const
{
	static std::string empty;
	if(isRoot())
		return empty;
	std::string::size_type pos = mName.rfind("/");

	if(pos == std::string::npos)
		return mName; // mName is a leaf itself

	assert(pos < mName.size());
	return mName.substr(pos+1);
}

bool ResourceName::isAncestorOf(const ResourceName& other) const
{
	if(other.mName.size()<this->mName.size())
		return false;

	if(other.isRoot()) // if other is root it has no ancestor
		return true;

	if(this->isRoot()) // if we are root, we are an ancestor of everyone
		return true;

	std::size_t p = this->mName.size();

	// we are ancestor of "other", if
	// this:   prefix
	// other:  prefix/xyz

	// check if prefix matches.
	if(other.mName.substr(0,p)!=this->mName)
		return false;

	if(other.mName[p]=='/') // look for the / (which comes before xyz)
		return true;

	return false; // otherwise the namespace is different
}

bool ResourceName::isParentOf(const ResourceName& other) const
{
	return other.parent() == *this;
}

bool ResourceName::isSuccessorOf(const ResourceName& other) const
{
	return other.isAncestorOf(*this);
}

bool ResourceName::isChildOf(const ResourceName& other) const
{
	return other.isParentOf(*this);
}

ResourceName ResourceName::makeFullyQualified(const ResourceName& name,
                                              const ResourceName& ns)
{
	if(name.isFullyQualified()) // already is fully qualified
		return name;

	if(!ns.isFullyQualified())
		MIRA_THROW(XInvalidParameter,
		           "Trying to create fully qualified name "
		           "using a non fully qualified namespace '"<< ns.str() << "'");

	std::vector<std::string> nameElements;
	boost::split(nameElements, name.str(), boost::is_from_range('/','/'));

	int upwards=0;
	int k = 0;
	for(; k<(int)nameElements.size(); ++k)
	{
		if (nameElements[k]=="..")
			 ++upwards;
		else if (nameElements[k]==".")
		{
			// ignore
		}
		else
			break;
	}

	if(k==0)
		return ns / name;

	std::vector<std::string> nsElements;
	boost::split(nsElements,ns.str(), boost::is_from_range('/','/'));

	ResourceName res("/");

	for(int i=1; i< (int)nsElements.size() - upwards; ++i)
	{
		if (!nsElements[i].empty())
			res /= nsElements[i];
	}

	for(int i=k; i< (int) nameElements.size(); ++i)
	{
		if (!nameElements[i].empty())
			res /= nameElements[i];
	}

	return res;
}

std::string ResourceName::normalize(const std::string& name)
{
	std::string r = name;
	boost::trim(r);

	// remove double //
	for(std::string::size_type pos = r.find("//"); pos!=std::string::npos; pos = r.find("//", pos))
		r.erase(pos,1);

	// remove trailing / (if not global namespace "/")
	if(r.size()>1 && *r.rbegin() == '/')
		r.erase(r.size()-1,1);

	return r;
}

// static
bool ResourceName::validate(const std::string& name)
{
	if(name.empty())
		return false;

	return true;
}

///////////////////////////////////////////////////////////////////////////////

}
