/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractChannel.C
 *    Implementation of AbstractChannel.h.
 *
 * @author Erik Einhorn
 * @date   2010/09/08
 */

#include <fw/Framework.h>

#include <fw/AbstractChannel.h>
#include <fw/ChannelManager.h>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

AbstractChannel::AbstractChannel(const std::string& id, ChannelBufferBase* buffer) :
	mID(id), mBuffer(buffer), mUntypedBuffer(NULL),
	mNrOfSubscribersWithoutChannelSubscriber(0),
	mLastSlotTime(Time::invalid()),
	mNrOfDataChanges(0)
{
	assert(mBuffer!=NULL);
}

AbstractChannel::~AbstractChannel()
{
	// TODO: do we have to make sure that no one is holding any slot, similar
	// to promote(), by write-locking each slot first?
	//  - Gah, just delete the thing ;)
	delete mBuffer; // whohooo
	delete mUntypedBuffer;
}

///////////////////////////////////////////////////////////////////////////////

bool AbstractChannel::hasPublisher() const
{
	return MIRA_FW.getChannelManager().hasPublisher(getID(), true);
}

void AbstractChannel::removeSubscriber()
{
	boost::mutex::scoped_lock lock(mSubscribersMutex);
	if (mNrOfSubscribersWithoutChannelSubscriber>0)
		--mNrOfSubscribersWithoutChannelSubscriber;
}

void AbstractChannel::removeSubscriber(AbstractChannelSubscriberPtr subscriber)
{
	boost::mutex::scoped_lock lock(mSubscribersMutex);
	mSubscribers.remove(subscriber);

	// tell the subscriber that it has been detached
	if(subscriber->getChannel()==this)
		subscriber->detachChannel();
}

void AbstractChannel::finishWrite(Slot* s)
{
	bool dropped;
	try {
		if(!mBuffer->finishWriteSlot(s, &dropped))
		{
			if(dropped) {
				MIRA_LOG(WARNING) << "Dropping data (slot) in channel '" << mID
				                  << "' since slot with timestamp " << mLastSlotTime << " already exists";
			}
			else {
				MIRA_LOG(WARNING) << "Timestamp of added slot (" << s->timestamp() << ") was older "
				                  << "than newest data (" << mLastSlotTime << ") in channel '" << mID
				                  << "', channel subscribers will not be notified. Check your publishers.";
			}
			return; // is not newest value, so do not notify listeners
		}
	} catch(Exception& ex) {
		MIRA_RETHROW(ex, "Failed to write to channel '" << mID << "'.");
	}

	++mNrOfDataChanges;
	mLastSlotTime = s->timestamp();

	// otherwise inform all subscribers about new data
	boost::mutex::scoped_lock lock(mSubscribersMutex);
	foreach(AbstractChannelSubscriberPtr subscriber, mSubscribers) {
		if(subscriber.get() != NULL)
			subscriber->signal();
	};
}

void AbstractChannel::discardWrite(Slot* s)
{
	mBuffer->discardWriteSlot(s);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
