/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OneSlotReadWriteTest.C
 *    Base functionality for testing concurrent read/write access on
 *    a single ChannelBuffer slot
 *
 * @author Christof Schröter
 * @date   2024/12/09
 */

#ifndef _MIRA_TESTS_ONESLOTREADWRITETEST_H_
#define _MIRA_TESTS_ONESLOTREADWRITETEST_H_

#include <fw/Framework.h>

using namespace mira;

template <typename T>
void writeChannel(Channel<T>& channel)
{
	BOOST_CHECK_EQUAL(channel.getMaxSlots(), 1);

	ChannelWrite<T> w = channel.write();
	w->value() = T{};
}

template <typename ChannelType>
void testReadWriteLocking(bool expect = true)
{
	auto tname = typeName<ChannelType>();
	Authority authority("/"+tname, "Authority");

	auto channel = authority.publishAndSubscribe<ChannelType>("Channel");

	// set the slot limit to 1, this is what this test is all about
	MIRA_FW.getChannelManager().setMaxSlots("/"+tname+"/Channel", 1);

	// ensure there is data on the channel so we can obtain a read
	channel.post(ChannelType{});

	{
		// obtain a channel read
		ChannelRead<ChannelType> r = channel.read();

		// while holding the read, try to obtain a channel write
		boost::thread writeThread(boost::bind(&writeChannel<ChannelType>, channel));

		// if write succeed, the thread will be able to join, otherwise it will be blocked
		BOOST_CHECK_EQUAL(writeThread.timed_join(Duration::seconds(1)), expect);

		// after finishing (releasing) the read, write should always succeed
		r.finish();
		if (writeThread.joinable()) {
			BOOST_CHECK(writeThread.timed_join(Duration::seconds(1)));
		}
	}

	// a void read lock will always lock the slot (block writing)
	auto channelv = authority.subscribe<void>("Channel");
	ChannelRead<void> r = channelv.read();

	boost::thread writeThread(boost::bind(&writeChannel<ChannelType>, channel));
	BOOST_CHECK(!writeThread.timed_join(Duration::seconds(1)));

	r.finish();
	BOOST_CHECK(writeThread.timed_join(Duration::seconds(1)));
}

#endif
