/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UnitManager.h
 *    Managing class for all (micro) units.
 *
 * @author Tim Langner
 * @date   2011/02/25
 */

#ifndef _MIRA_UNITMANAGER_H_
#define _MIRA_UNITMANAGER_H_

#include <tuple>

#include <fw/MicroUnit.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * Class managing micro units and units.
 */
class UnitManager
{
public:

	~UnitManager();

	/**
	 * Adds a thread group with namespace and name.
	 * A thread group forms a parent authority of all enclosed units.
	 * These units will share the dispatcher thread of the thread groups authority
	 */
	std::string addThreadGroup(const ResourceName& ns, const std::string& name);

	/**
	 * Adds a unit to list of known ones. After all units are added
	 * they can be initialized, resumed and paused.
	 * @param[in] unit The pointer to the micro unit
	 * @param[in] ns The namespace this unit resides in
	 * @param[in] name The name of the unit
	 * @param[in] threadGroup The thread group that contains this unit or empty
	 * @param[in] startAtInit Flag if unit shall be started after initialization (default=true)
	 */
	void addUnit(MicroUnitPtr unit, const std::string& ns,
	             const std::string& name, const std::string& threadGroup = "",
	             bool startAtInit=true);

	/**
	 * Remove a known unit by its full id
	 */
	void removeUnit(const std::string& id);

	/**
	 * Get a unit from the manager by its full id.
	 * @return pointer to unit or null pointer if not found
	 */
	MicroUnitPtr getUnit(const std::string& id);

	/**
	 * Initializes (checks in) and starts all known units that are not yet
	 * initialized.
	 */
	void initializeUnits();

	/**
	 * Starts/Resumes all known units that are not yet started.
	 */
	void startUnits();

	/**
	 * Stops/Pauses all known units
	 */
	void stopUnits(bool destruct=false);

protected:
	// each tuple contains 1. the shared pointer to the micro unit
	//                     2. the namespace of the unit as string
	//                     3. the name of the unit as string
	//                     4. the name of the thread group if any
	//                     5. flag if unit shall be started after initialization
	typedef std::tuple<MicroUnitPtr, std::string, std::string, std::string, bool> UnitEntry;
	typedef std::list<UnitEntry> UnitList;
	UnitList mUnits;
	std::map<std::string, boost::shared_ptr<Authority>> mThreadGroups;
	mutable boost::mutex mUnitsMutex;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
