/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceLevel.h
 *    QoS management informations.
 *
 * @author Tim Langner
 * @date   2011/10/06
 */

#ifndef _MIRA_SERVICELEVEL_H_
#define _MIRA_SERVICELEVEL_H_

#ifndef Q_MOC_RUN
#include <boost/algorithm/string/replace.hpp>
#endif

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * QoS management information for channels
 */
class ServiceLevelBase
{
public:

	enum ConnectionType
	{
		UDP,
		TCP
	};

	ServiceLevelBase() : connectionType(TCP), interval(Duration::seconds(-1)) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType version
			= r.version(2, serialization::AcceptDesiredVersion(), this);
		r.member("Codecs", codecs,"", std::list<BinarySerializerCodecPtr>());
		r.member("ConnectionType", connectionType, "", TCP);

		if (version >= 2)
			r.member("Interval", interval, "", Duration::seconds(-1));
	}

	// Specialization for XMLDeserializer
	// for convenience: add the Interval member without requiring a version
	// (with proper default if not present)
	void reflect(XMLDeserializer& r)
	{
		r.member("Codecs", codecs,"", std::list<BinarySerializerCodecPtr>());
		r.member("ConnectionType", connectionType, "", TCP);
		r.member("Interval", interval, "", Duration::seconds(-1));
	}

	std::list<BinarySerializerCodecPtr> codecs;
	ConnectionType connectionType;
	Duration interval; // negative by default, as invalid Duration cannot be serialized properly :(
};

/// ServiceLevel by channel name
class ServiceLevel : public ServiceLevelBase
{
public:

	ServiceLevel() {}

	ServiceLevel(const std::string& channel) :
		channelID(channel) {}

	ServiceLevel(const std::string& channel, const ServiceLevelBase& other) :
		ServiceLevelBase(other), channelID(channel) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Channel", channelID, "", "");

		// As an exception, do not use reflectBase here, in order to remain compatible
		// with older serialization. (Cannot use 'version' in this case either, as the
		// ServiceLevel must be understood by older code, and that will understand
		// there is a version, but not accept a newer version :( )
		ServiceLevelBase::reflect(r);
	}

	std::string channelID;
};

/// ServiceLevel by channel type
class TypeServiceLevel : public ServiceLevelBase
{
public:

	TypeServiceLevel() {}

	TypeServiceLevel(const Typename& type) :
		channelType(type) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ChannelType", channelType,
		         setter(&TypeServiceLevel::setChannelType, this),
		         "", "");
		MIRA_REFLECT_BASE(r, ServiceLevelBase);
	}

	void setChannelType(Typename type) {
		boost::replace_all(type, "(", "<");
		boost::replace_all(type, "[", "<");
		boost::replace_all(type, "{", "<");

		boost::replace_all(type, ")", ">");
		boost::replace_all(type, "]", ">");
		boost::replace_all(type, "}", ">");

		channelType = type;
	}

	Typename channelType;
};
///////////////////////////////////////////////////////////////////////////////

}

#endif 
