/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TimeHumanReadableTest.C
 *    Test cases for TimeHumanReadable.h.
 *
 * @author Christof Schröter
 * @date   2024/09/19
 */

#include <boost/test/unit_test.hpp>

#include <utils/TimeHumanReadable.h>
#include <serialization/JSONSerializer.h>
#include <serialization/XMLSerializer.h>

using namespace mira;

BOOST_AUTO_TEST_CASE( testConversion )
{
	DurationSerializedAsIsoString ds = Duration(0, 0, 0);
	BOOST_CHECK_EQUAL(Duration(ds), ds);

	DurationSerializedAsHMS dhms = Duration(0, 0, 1);
	BOOST_CHECK_EQUAL(Duration(dhms), dhms);

	const DurationSerializedAsIsoString dsconst = Duration(0, 1, 0);
	BOOST_CHECK_EQUAL(Duration(dsconst), dsconst);

	const DurationSerializedAsHMS dhmsconst = Duration(1, 0, 0);
	BOOST_CHECK_EQUAL(Duration(dhmsconst), dhmsconst);

	TimeSerializedAsIsoString ts = Time::now();
	BOOST_CHECK_EQUAL(Time(ts), ts);

	TimeSerializedAsYMDHMS tymd = Time::now() + Duration(0, 0, 1);
	BOOST_CHECK_EQUAL(Time(tymd), tymd);

	const TimeSerializedAsIsoString tsconst = Time::now() + Duration(0, 1, 0);
	BOOST_CHECK_EQUAL(Time(tsconst), tsconst);

	const TimeSerializedAsYMDHMS tymdconst = Time::now() + Duration(1, 0, 0);
	BOOST_CHECK_EQUAL(Time(tymdconst), tymdconst);
}

template <typename TimeType>
void testSymmetry()
{
	TimeType time = Time::now();
//		std::cout << time << std::endl;

	XMLDom xml;
	XMLSerializer xs(xml);

	xs.serialize("time", time, "");
	std::cout << xml.saveToString() << std::endl;

	TimeType time2;
	XMLDeserializer xds(xml);
	xds.deserialize("time", time2);
	BOOST_CHECK_EQUAL(time, time2);


	JSONSerializer js(true);
	json::Value v = js.serialize(time);
	std::cout << json::write(v) << std::endl;

	TimeType time3;
	JSONDeserializer jds(v);
	jds.deserialize(time3);
	BOOST_CHECK_EQUAL(time, time3);
}

BOOST_AUTO_TEST_CASE( testTimeSymmetry )
{
	testSymmetry<TimeSerializedAsIsoString>();
	testSymmetry<TimeSerializedAsYMDHMS>();
}

struct MyClass
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("DateRaw", date, "");
		auto accDa = humanReadableAccessor(dateString);
		r.property("DateString", accDa, "");
		r.property("DateYMD", dateYMD, "");

		r.property("TimeRaw", time, "");
		auto accT = humanReadableAccessor(timeHR);
		r.property("TimeString", accT, "");
		auto accTV = humanReadableAccessor(timeHR, TimeTypesSerializedAsElements());
		r.property("TimeVerbose", accTV, "");

		r.property("DurationRaw", duration, "");
		auto accDu = humanReadableAccessor(durationHR);
		r.property("DurationString", accDu, "");
	}

	Date date                   = Date(2024, 10, 23);
	Date dateString             = Date(2000, 1, 23);
	DateSerializedAsYMD dateYMD = Date(2000, 1, 31);
	Time time                   = Time::now();
	Time timeHR                 = Time::now() + Duration::seconds(10);
	Duration duration           = Duration(0, 12, 34);
	Duration durationHR         = Duration(1, 23, 45);
};

BOOST_AUTO_TEST_CASE( testAccessors )
{
	MyClass obj;
	XMLDom xml;
	XMLSerializer xs(xml);

	xs.serialize("obj", obj, "");
	std::cout << xml.saveToString() << std::endl;

	MyClass obj2;

	XMLDeserializer xds(xml);
	xds.deserialize("obj", obj2);
	BOOST_CHECK_EQUAL(obj.time, obj2.time);
}

BOOST_AUTO_TEST_CASE( testProperties )
{
	MyClass obj;
	PropertySerializer s;
	PropertyNode* node = s.reflectProperties("Object", obj);
}

TimeSerializedAsIsoString timeFromXmlString(const std::string& s)
{
	std::cout << "'" << s << "'";
	for (int n = s.size(); n < 32; ++n)
		std::cout << " ";
	std::cout << "--> ";
	std::cout.flush();

	TimeSerializedAsIsoString time;
	XMLDom xml;
	xml.loadFromString("<root><time>" + s + "</time></root>");
	XMLDeserializer xds(xml);
	try {
		xds.deserialize("time", time);
	}
	catch(std::exception& ex) {
		std::cout << typeName(ex) << std::endl;
		throw;
	}
	catch(...) {
		std::cout << "unknown exception" << std::endl;
		throw;
	}
	std::cout << time << std::endl;
	return time;
}

DurationSerializedAsIsoString durationFromXmlString(const std::string& s)
{
	std::cout << "'" << s << "'";
	for (int n = s.size(); n < 32; ++n)
		std::cout << " ";
	std::cout << "--> ";
	std::cout.flush();

	DurationSerializedAsIsoString duration;
	XMLDom xml;
	xml.loadFromString("<root><duration>" + s + "</duration></root>");
	XMLDeserializer xds(xml);
	try {
		xds.deserialize("duration", duration);
	}
	catch(std::exception& ex) {
		std::cout << typeName(ex) << std::endl;
		throw;
	}
	catch(...) {
		std::cout << "unknown exception" << std::endl;
		throw;
	}
	std::cout << duration << std::endl;
	return duration;
}


BOOST_AUTO_TEST_CASE( testFromStringEdgeCases )
{
	// check deserialization for different variants of
	// correct and ill-formed time/duration string

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-02-01T01:23:45"),
	                  Time(Date(2024,2,1),
	                       Duration(1, 23, 45)));

	BOOST_CHECK_THROW(timeFromXmlString("123456789"), XIO);

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-02-01T01:2:5"),
	                  Time(Date(2024,2,1),
	                       Duration(1, 2, 5)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:66:11"),
	                  Time(Date(2024,9,19),
	                       Duration::hours(10)+Duration::minutes(66)+Duration::seconds(11)));

	TimeSerializedAsIsoString time = timeFromXmlString("2024-09-19T25:61:23");
	BOOST_CHECK_EQUAL(time, Time(Date(2024,9,19),
	                             Duration::hours(25)+Duration::minutes(61)+Duration::seconds(23)));
	BOOST_CHECK_EQUAL(time, Time(Date(2024,9,20),
	                             Duration(2, 1, 23)));

	BOOST_CHECK_THROW(timeFromXmlString("2024-02-30T10:23:45"),
	                  boost::gregorian::bad_day_of_month);

	BOOST_CHECK_EQUAL(timeFromXmlString("3024-09-19T10:66:11"),
	                  Time(Date(3024,9,19),
	                       Duration::hours(10)+Duration::minutes(66)+Duration::seconds(11)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10"),
	                  Time(Date(2024,9,19),
	                       Duration::hours(10))); // minutes = seconds = 0

	BOOST_CHECK_THROW(timeFromXmlString("2024-09-19T10:abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(timeFromXmlString("2024-09-19T10:23ab"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(timeFromXmlString("2024-09-19T10:01:23.abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(timeFromXmlString("2024-09-19T10:01:23.012abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.23"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 230000)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.234"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 234000)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.098138ab"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 98138)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.098138+2:00"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 98138)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.098138Z"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 98138)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.098138CEST"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 98138)));

	BOOST_CHECK_EQUAL(timeFromXmlString("2024-09-19T10:23:41.098138 CEST"),
	                  Time(Date(2024,9,19),
	                       Duration(10, 23, 41, 98138)));

	BOOST_CHECK_EQUAL(durationFromXmlString("01:23:45"),
	                  Duration(1, 23, 45));

	BOOST_CHECK_THROW(durationFromXmlString("123456789"), XIO);

	BOOST_CHECK_EQUAL(durationFromXmlString("1:2:5"),
	                  Duration(1, 2, 5));

	BOOST_CHECK_EQUAL(durationFromXmlString("10:66:11"),
	                  Duration::hours(10)+Duration::minutes(66)+Duration::seconds(11));

	BOOST_CHECK_EQUAL(durationFromXmlString("25:61:23"),
	                  Duration::hours(25)+Duration::minutes(61)+Duration::seconds(23));

	BOOST_CHECK_EQUAL(durationFromXmlString("10:00"),
	                       Duration::hours(10)); // minutes = seconds = 0

	BOOST_CHECK_THROW(durationFromXmlString("10:abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(durationFromXmlString("10:23ab"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(durationFromXmlString("10:01:23.abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_THROW(durationFromXmlString("10:01:23.012abc"),
	                  boost::bad_lexical_cast);

	BOOST_CHECK_EQUAL(durationFromXmlString("10:23:41.23"),
	                  Duration(10, 23, 41, 230000));

	BOOST_CHECK_EQUAL(durationFromXmlString("10:23:41.234"),
	                  Duration(10, 23, 41, 234000));

	BOOST_CHECK_EQUAL(durationFromXmlString("10:23:41.098138ab"),
	                  Duration(10, 23, 41, 98138));

	BOOST_CHECK_EQUAL(durationFromXmlString("0:00:00.0505"),
	                  Duration(0, 0, 0, 50500));
	BOOST_CHECK_EQUAL(durationFromXmlString("0:0.0505"), /* this is interpreted as 505 seconds! */
	                  Duration(0, 8, 25));

	BOOST_CHECK_THROW(durationFromXmlString("0.0505"), XIO);
}
