/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StampedDataQueueTest.C
 *    Test cases for StampedDataQueue.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/05
 */

#include <boost/test/unit_test.hpp>

#include <utils/StampedDataQueue.h>
#include <utils/Foreach.h>

using namespace std;
using namespace mira;

BOOST_AUTO_TEST_CASE( test )
{

	typedef StampedDataQueue<int> Queue;

	Queue q;
	Time t0=Time::now();

	q.insert(makeStamped(-3, t0-Duration::seconds(3)));
	q.insert(makeStamped(-2, t0-Duration::seconds(2)));
	q.insert(makeStamped(-1, t0-Duration::seconds(1)));
	q.insert(makeStamped(0,  t0+Duration::seconds(0)));
	q.insert(makeStamped(1, t0+Duration::seconds(1)));
	q.insert(makeStamped(2, t0+Duration::seconds(2)));
	q.insert(makeStamped(3, t0+Duration::seconds(3)));

	Stamped<int> s1 = q.getData(t0+Duration::milliseconds(2499));
	Stamped<int> s2 = q.getData(t0+Duration::milliseconds(2500));
	Stamped<int> s3 = q.getData(t0+Duration::milliseconds(2501));
	BOOST_CHECK_EQUAL(s1.value(), 2);
	BOOST_CHECK_EQUAL(s2.value(), 2);
	BOOST_CHECK_EQUAL(s3.value(), 3);

	auto its = q.getInterval(t0, 0, 4, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 0);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);

	its = q.getInterval(t0, 5, 4, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 1);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);

	its = q.getInterval(t0, 0, 1, 5);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), -1);

	its = q.getInterval(t0, 6, 1, 5);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), -2);

	its = q.getInterval(t0, 0, 5, 5);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);

	its = q.getInterval(t0, 2, 1, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 0);
	BOOST_CHECK_EQUAL((--its.second)->value(), -1);

	// requesting 0 items both before and after the designated timestamp
	// should return an empty interval
	its = q.getInterval(t0, 0, 0, 0);
	BOOST_CHECK_EQUAL(std::distance(its.first, its.second), 0);

	its = q.getInterval(t0+Duration::milliseconds(500), 0, 4, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 1);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);

	its = q.getInterval(t0+Duration::milliseconds(500), 5, 4, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 1);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);

	its = q.getInterval(t0+Duration::milliseconds(500), 0, 1, 4);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), 0);

	its = q.getInterval(t0+Duration::milliseconds(500), 5, 1, 4);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), -1);

	its = q.getInterval(t0+Duration::milliseconds(500), 2, 1, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 1);
	BOOST_CHECK_EQUAL((--its.second)->value(), 0);

	its = q.getInterval(t0+Duration::milliseconds(500), 0, 0, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 1);
	BOOST_CHECK_EQUAL((--its.second)->value(), 1);

	its = q.getInterval(t0+Duration::milliseconds(500), 0, 1, 0);
	BOOST_CHECK_EQUAL(its.first->value(), 0);
	BOOST_CHECK_EQUAL((--its.second)->value(), 0);

	its = q.getInterval(t0+Duration::seconds(4), 0, 1, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), 3);

	its = q.getInterval(t0+Duration::seconds(4), 2, 1, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), 2);

	its = q.getInterval(t0+Duration::seconds(4), 0, 0, 1);
	BOOST_CHECK_EQUAL(std::distance(its.first, its.second), 0);

	its = q.getInterval(t0+Duration::seconds(4), 1, 0, 1);
	BOOST_CHECK_EQUAL(its.first->value(), 3);
	BOOST_CHECK_EQUAL((--its.second)->value(), 3);

	its = q.getInterval(t0-Duration::seconds(4), 0, 1, 0);
	BOOST_CHECK_EQUAL(std::distance(its.first, its.second), 0);

	its = q.getInterval(t0-Duration::seconds(4), 1, 1, 0);
	BOOST_CHECK_EQUAL(its.first->value(), -3);
	BOOST_CHECK_EQUAL((--its.second)->value(), -3);
}
