/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoolAllocatorTest.C
 * 		Benchmark and manual test for PoolAllocator (this is not a unit test!)
 *
 * @author Erik Einhorn
 * @date   2011/11/17
 */

#include <boost/pool/object_pool.hpp>
#include <boost/pool/pool_alloc.hpp>

#include <platform/Memory.h>
#include <utils/PoolAllocator.h>
#include <utils/Time.h>

using namespace mira;

#define SIZE  48
//#define COUNT 40000000ULL
#define COUNT 10000000ULL


class DummyClass
{
public:
	char buf[SIZE];
};

uint64 mphys0;

template <typename Allocator>
Duration benchmark(Allocator& alloc)
{
	DummyClass* p;
	uint64 real = 0;

	Time start = Time::now();

	//
	for(uint64 i=0; i<COUNT; ++i) {
		//p = new DummyClass;
		p = alloc.allocate(1);
		real += sizeof(DummyClass);

		if(i%2 == 0) {
			alloc.deallocate(p,1);
			real -= sizeof(DummyClass);
		}

		if(i%1000000 == 0) {
			uint64 mreal = real/1024;
			uint64 mphys = getMemoryUsage().totalHeap-mphys0;
			double ratio = (double)mphys / (double)mreal;
			std::cout << "    " << mreal << ", " << mphys << ": " << ratio << std::endl;
		}
	}
	return Time::now() - start;
}

///////////////////////////////////////////////////////////////////////////////

int main()
{
	mphys0 = getMemoryUsage().totalHeap;

	// new/delete 3.8 s
	// boost pool: 5.6 s
	// own PoolAllocator: 3.0s

	{
		PoolAllocator<DummyClass> pool(1024, false);
		std::cout << "PoolAllocator"<<std::endl;
		Duration d = benchmark(pool);
		std::cout << "Time: " << d << std::endl << std::endl;
	}

	{
		boost::fast_pool_allocator<DummyClass> pool;
		std::cout << "boost::fast_pool_allocator"<<std::endl;
		Duration d = benchmark(pool);
		std::cout << "Time: " << d << std::endl << std::endl;
	}

	{
		boost::pool_allocator<DummyClass> pool;
		std::cout << "boost::pool_allocator"<<std::endl;
		Duration d = benchmark(pool);
		std::cout << "Time: " << d << std::endl << std::endl;
	}

	{
		std::allocator<DummyClass> alloc;
		std::cout << "std::allocator"<<std::endl;
		Duration d = benchmark(alloc);
		std::cout << "Time: " << d << std::endl << std::endl;
	}

	return 0;
}

