/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryStreamTest.C
 *    Test case for BinaryStream.
 *
 * @author Erik Einhorn
 * @date 2010/06/28
 */

#include <boost/test/unit_test.hpp>

#include <iostream>
#include <fstream>
#include <stream/BinaryStream.h>
#include <math/Math.h>


using namespace std;
using namespace mira;


#define checkIOStream(T, x, prepareos, prepareis) \
{								\
	prepareos                   \
	os << (T)x;					\
	prepareis                   \
	T r;						\
	is >> r;					\
	BOOST_CHECK_EQUAL(x,  r); 	\
}


#define checkIOStreamLimits(T, prepareos, prepareis)  \
{                               \
	prepareos                   \
	os << numeric_limits<T>::min();      \
	os << numeric_limits<T>::max();      \
	os << numeric_limits<T>::infinity(); \
	prepareis                   \
	T min, max, infty;			\
	is >> min >> max >> infty;	\
	BOOST_CHECK_EQUAL(min,   numeric_limits<T>::min()); 	\
	BOOST_CHECK_EQUAL(max,   numeric_limits<T>::max()); 	\
	BOOST_CHECK_EQUAL(infty, numeric_limits<T>::infinity()); \
}

#define BasicTest(prepareos, prepareis)                       \
	checkIOStream(bool, true,	prepareos, prepareis);        \
	checkIOStream(char, 'F',	prepareos, prepareis);        \
	checkIOStream(int,  1234,	prepareos, prepareis);        \
	checkIOStream(uint8, 65,	prepareos, prepareis);        \
	checkIOStream(uint8, 255,	prepareos, prepareis);        \
	checkIOStream(uint16, 10,	prepareos, prepareis);        \
	checkIOStream(uint32, 10,	prepareos, prepareis);        \
	checkIOStream(uint64, 10,	prepareos, prepareis);        \
	checkIOStream(int8, 65,		prepareos, prepareis);        \
	checkIOStream(int8, -128,	prepareos, prepareis);        \
	checkIOStream(int8, -127,	prepareos, prepareis);        \
	checkIOStream(int16, -10,	prepareos, prepareis);        \
	checkIOStream(int32, -10,	prepareos, prepareis);        \
	checkIOStream(int64, -10,	prepareos, prepareis);        \
	checkIOStream(float, -10.10f,	prepareos, prepareis);    \
	checkIOStream(double, -10.10f,	prepareos, prepareis);

BOOST_AUTO_TEST_CASE( BasicTest_BinaryStlStream )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss);
	#define prepareis BinaryStlIstream is(ss);
	BasicTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( BasicTest_BinaryStlStream_Net )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss); os << net;
	#define prepareis BinaryStlIstream is(ss); is >> net;
	BasicTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( BasicTest_BinaryBufferStream )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data);
	#define prepareis BinaryBufferIstream is(&data);
	BasicTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( BasicTest_BinaryBufferStream_Net )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data); os << net;
	#define prepareis BinaryBufferIstream is(&data); is >> net;
	BasicTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}



#define LimitTest(prepareos, prepareis)                       \
	checkIOStreamLimits(bool,	prepareos, prepareis);        \
	checkIOStreamLimits(char,	prepareos, prepareis);        \
	checkIOStreamLimits(int,	prepareos, prepareis);        \
	checkIOStreamLimits(uint8,	prepareos, prepareis);        \
	checkIOStreamLimits(uint16,	prepareos, prepareis);        \
	checkIOStreamLimits(uint32,	prepareos, prepareis);        \
	checkIOStreamLimits(uint64,	prepareos, prepareis);        \
	checkIOStreamLimits(int8,	prepareos, prepareis);        \
	checkIOStreamLimits(int16,	prepareos, prepareis);        \
	checkIOStreamLimits(int32,	prepareos, prepareis);        \
	checkIOStreamLimits(int64,	prepareos, prepareis);        \
	checkIOStreamLimits(float,	prepareos, prepareis);        \
	checkIOStreamLimits(double,	prepareos, prepareis);

BOOST_AUTO_TEST_CASE( LimitTest_BinaryStlStream )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss);
	#define prepareis BinaryStlIstream is(ss);
	LimitTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( LimitTest_BinaryStlStream_Net )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss); os << net;
	#define prepareis BinaryStlIstream is(ss); is >> net;
	LimitTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( LimitTest_BinaryBufferStream )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data);
	#define prepareis BinaryBufferIstream is(&data);
	LimitTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( LimitTest_BinaryBufferStream_Net )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data); os << net;
	#define prepareis BinaryBufferIstream is(&data); is >> net;
	LimitTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}


#define NaNTest(prepareos, prepareis)                       \
{                                                           \
	prepareos                                               \
	os << numeric_limits<float>::quiet_NaN();               \
	prepareis                                               \
	float r;                                                \
	is >> r;                                                \
	BOOST_CHECK(boost::math::isnan(r));                     \
}

BOOST_AUTO_TEST_CASE( NaNTest_BinaryStlStream )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss);
	#define prepareis BinaryStlIstream is(ss);
	NaNTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( NaNTest_BinaryStlStream_Net )
{
	#define prepareos stringstream ss; BinaryStlOstream os(ss); os << net;
	#define prepareis BinaryStlIstream is(ss); is >> net;
	NaNTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( NaNTest_BinaryBufferStream )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data);
	#define prepareis BinaryBufferIstream is(&data);
	NaNTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE( NaNTest_BinaryBufferStream_Net )
{
	#define prepareos Buffer<uint8> data; BinaryBufferOstream os(&data); os << net;
	#define prepareis BinaryBufferIstream is(&data); is >> net;
	NaNTest(prepareos, prepareis)
	#undef prepareos
	#undef prepareis
}

BOOST_AUTO_TEST_CASE(ComplexTest_BinaryBufferStream)
{
	Buffer<uint8> v1;
	{
		BinaryBufferOstream os(&v1);
		int a,b;
		char c = 123;;
		a=123;
		b=456;
		os << c << a << b << "Hallo";
	}

	{
		int a,b;
		char c;
		std::string str;
		BinaryBufferIstream is(&v1);
		is >> c >> a >> b >> str;
		BOOST_CHECK_EQUAL(a,123);
		BOOST_CHECK_EQUAL(b,456);
		BOOST_CHECK_EQUAL(c,123);
		BOOST_CHECK_EQUAL(str,"Hallo");
	}
}

