/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSAKeyTest.C
 *    Test case for RSAKey.
 *
 * @author Christian Martin
 * @date   2011/04/08
 */

#include <boost/test/unit_test.hpp>

#include <security/RSAKey.h>

#include <serialization/BinarySerializer.h>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////
// a 2048bit RSA key pair for some tests

const std::string sTestKeyPublic2048 =
	"PUBLIC:270:3082010a0282010100bbe3255f8ef0420b1e266eb9e3fa397e0dfc3b888"
	"68bc8e9dfc0677b75a38778fa2ffdb5f43990dee95195802869512c6fc63a53fd9aefe"
	"d655dec5d82a94be079906580ca896737340fd50c26413febc1bdca1e74f6c97c9b569"
	"bd70e93e025ef1634563e382f6a73c83e20feb5b7752761bfc38b3b0c494ea339e2267"
	"38100dd94f4e0634e0bdf27854acd54d6e544a70a1c68778c2109b7f5eb3c9966e7098"
	"8125b4a8e4562b6dac5f04c7bb03e56c9ebbca83f046b7b4ea71e818333ae6cf001b56"
	"cf6fdf01fbd640ac951085273be93d9549c5b810617566fbaadf063d43d2c1d14bc8ea"
	"0ac1f7afd30fa1935331cf17474b02b39926cfe5fc86f08135b0203010001;";

const std::string sTestKeyPrivate2048 =
	"PRIVATE:1192:308204a40201000282010100bbe3255f8ef0420b1e266eb9e3fa397e0"
	"dfc3b88868bc8e9dfc0677b75a38778fa2ffdb5f43990dee95195802869512c6fc63a5"
	"3fd9aefed655dec5d82a94be079906580ca896737340fd50c26413febc1bdca1e74f6c"
	"97c9b569bd70e93e025ef1634563e382f6a73c83e20feb5b7752761bfc38b3b0c494ea"
	"339e226738100dd94f4e0634e0bdf27854acd54d6e544a70a1c68778c2109b7f5eb3c9"
	"966e70988125b4a8e4562b6dac5f04c7bb03e56c9ebbca83f046b7b4ea71e818333ae6"
	"cf001b56cf6fdf01fbd640ac951085273be93d9549c5b810617566fbaadf063d43d2c1"
	"d14bc8ea0ac1f7afd30fa1935331cf17474b02b39926cfe5fc86f08135b02030100010"
	"2820101008dfd07c662683110e3d5198283c61ae2a74101e09805556d60f9af7a7e997"
	"98cb44dc647566c968dcef727aaf55f39c8022464fa0a42a7a4d92114ba4a3f14a76c5"
	"6f24e5541676e317af17f3ffdb987ba1f6a7a7fdbecc2ad45d379bf7b833e47a39923b"
	"fdb8801c3c40527bc35a991f0940f6e806ebc473e155e6e08aeeb42674ea0792213a7f"
	"bd0edae8755788c785b1a560f5afa01c2a36635042329743263cd6d50c1ef55106404c"
	"338c3b594ee980596371dc22b87a3bf80cb004104059a9e921d607b6a5dd7141d5d6bc"
	"aa81a56cc8d217da2bf61805905d7359b8bc1bb4f2396da3cd896dbdd3b2e73845594a"
	"cf949431d9f13553c8bb33c7f5ca05902818100e66e07360c7d68a5895f05e376274b7"
	"e88c3112bb9b2dab61de76151fb47f8243de3cd4d9ed125f84202df2bbaf16450bb34b"
	"68ba4bc99536a176f1f67f950568fcd9bfda69abe5830943354b8b57b9cd6ebb7937fe"
	"30f053cd362f61ddf91002d8cfcfeb8fabb7b795e27cb494a3ee9cff245ffb9cce0f1d"
	"0cf9c29dd610e9d02818100d0bc95a9359f154396caa4984dfe5c06d86d372215dce39"
	"e32d064140f9b291e728bf61015b80c4d940eb7b802bed13c522073e14631025365e2e"
	"81e9ef907ff079ce51c3f2a70d4111ad9b22bd9446ce0357dc15d9e383414286d834c4"
	"58f5780262f1a07f6665cc0b37329f9d745ec69ade0efd53585c162fe8b6c10d3cc570"
	"2818100c585f7a47b1c5d41d8f99d6097fda3173a408c4d810731b7fef642543624412"
	"f3fe6302ca341015d47a8928937dc887333a641fc7e2035066de0dc9e7d50176fbc213"
	"1e0429a4f23e5b08437d185b00a87a8d35a1abc163a1523d6a6c680c818f0f2f4439d6"
	"940a0dd2ebac9e56f4172c7901434db9e260ce343b3c7a73ef579028180204476d2408"
	"2c613bbb4b9bafd9b8b78576e92ae4738ed4ae04a0c9e794720151fd9f900c58abd042"
	"5161d5a69ba3bb81a6dffdbdc30e94e211af18d024036f73aa30b5bd17beb132d4dbc4"
	"769b5dd790333e1abc9a012ca222f99b6494d1b7a8849ded02458f7bbf0e20cb47975c"
	"8ce620bca10e921abfa982607417e5c6f63028180021fb5be49ce0a57c2f35d1a57a75"
	"308f03c97eb6ffda7ba1bd462eb62892efa41cc816e86eb9b83e7346121795bea6dafd"
	"ec831e14268c8b24c148fd62338913c25e291df884bdd7cf148537bfddcd5c53c8a966"
	"f91fe0907af7e28ce66d938140b7b14eccee120ba5bd5dbc938120b7299cd36e014400"
	"81e5c3ae306494819;";
;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSAKeyTest_KnownStringTest )
{
	// parse the string to a key-pair
	RSAKey publicKey = fromString<RSAKey>(sTestKeyPublic2048);
	RSAKey privateKey = fromString<RSAKey>(sTestKeyPrivate2048);

	// check if the keys are valid
	BOOST_CHECK(publicKey.isValid());
	BOOST_CHECK(publicKey.isPublicKey());
	BOOST_CHECK(privateKey.isValid());
	BOOST_CHECK(privateKey.isPrivateKey());

	BOOST_CHECK_EQUAL(publicKey.getNStr(), privateKey.getNStr());
	BOOST_CHECK_EQUAL(publicKey.getEStr(), privateKey.getEStr());
	BOOST_CHECK_EQUAL(publicKey.getDStr(), "");
	BOOST_CHECK_NE(privateKey.getDStr(), "");

	// convert back to string and compare with the original string
	std::string strPublic  = toString(publicKey);
	std::string strPrivate = toString(privateKey);
	BOOST_CHECK_EQUAL(strPublic, sTestKeyPublic2048);
	BOOST_CHECK_EQUAL(strPrivate, sTestKeyPrivate2048);
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSAKeyTest_CopyCtor )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	RSAKey tPublicKeyCopy(tPublicKey);
	RSAKey tPrivateKeyCopy(tPrivateKey);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_AssignOp )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	RSAKey tPublicKeyCopy;
	tPublicKeyCopy = tPublicKey;
	RSAKey tPrivateKeyCopy;
	tPrivateKeyCopy = tPrivateKey;

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_ToFromString )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	string tPublicStr = toString(tPublicKey);
	string tPrivateStr = toString(tPrivateKey);

	RSAKey tPublicKeyCopy = fromString<RSAKey>(tPublicStr);
	RSAKey tPrivateKeyCopy = fromString<RSAKey>(tPrivateStr);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_N_E_D )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	string tPublicStrN = tPublicKey.getNStr();
	string tPublicStrE = tPublicKey.getEStr();
	string tPublicStrD = tPublicKey.getDStr();

	string tPrivateStrN = tPrivateKey.getNStr();
	string tPrivateStrE = tPrivateKey.getEStr();
	string tPrivateStrD = tPrivateKey.getDStr();

	RSAKey tPublicKeyCopy(tPublicStrN, tPublicStrE, tPublicStrD);
	RSAKey tPrivateKeyCopy(tPrivateStrN, tPrivateStrE, tPrivateStrD);

	BOOST_CHECK(tPublicKeyCopy.isValid());
	BOOST_CHECK(tPrivateKeyCopy.isValid());
	BOOST_CHECK(tPublicKeyCopy.isPublicKey());
	BOOST_CHECK(tPrivateKeyCopy.isPrivateKey());

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_clear )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(!tPublicKey.isPrivateKey());

	tPublicKey.clear();
	BOOST_CHECK(!tPublicKey.isValid());
	BOOST_CHECK(!tPublicKey.isPublicKey());
	BOOST_CHECK(!tPublicKey.isPrivateKey());

	tPrivateKey.clear();
	BOOST_CHECK(!tPrivateKey.isValid());
	BOOST_CHECK(!tPrivateKey.isPublicKey());
	BOOST_CHECK(!tPrivateKey.isPrivateKey());
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_stream )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	stringstream tPubStream;
	tPubStream << tPublicKey;
	RSAKey tPublicKeyCopy;
	tPubStream >> tPublicKeyCopy;

	stringstream tPrivStream;
	tPrivStream << tPrivateKey;
	RSAKey tPrivateKeyCopy;
	tPrivStream >> tPrivateKeyCopy;

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_reflect )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	Buffer<uint8> binaryBuffer;
	BinaryBufferSerializer binSerializer(&binaryBuffer);
	binSerializer.serialize(tPublicKey);
	binSerializer.serialize(tPrivateKey);

	RSAKey tPublicKeyCopy;
	RSAKey tPrivateKeyCopy;
	BinaryBufferDeserializer binDeserializer(&binaryBuffer);
	binDeserializer.deserialize(tPublicKeyCopy);
	binDeserializer.deserialize(tPrivateKeyCopy);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}
