/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSAFilterTest.C
 *    Test case for RSAFilter.
 *
 * @author Christian Martin
 * @date   2010/11/09
 */

#include <boost/test/unit_test.hpp>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/device/file_descriptor.hpp>
#include <boost/iostreams/device/array.hpp>
#include <boost/iostreams/filtering_stream.hpp>

#include <boost/filesystem/operations.hpp>

#include <security/RSAFilter.h>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSAFilter_ShortDataTest )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	string tInputString = "Hello_world!";
	{
		boost::iostreams::filtering_ostream tOut;
		tOut.push(RSAPublicEncryptionFilter(tPublicKey));
		tOut.push(boost::iostreams::file_sink("rsa.dat", std::ios::out | std::ios::binary));
		tOut << tInputString << endl;
	}

	string tOutputString;
	{
		boost::iostreams::filtering_istream tIn;
		tIn.push(RSAPrivateDecryptionFilter(tPrivateKey));
		tIn.push(boost::iostreams::file_source("rsa.dat", std::ios::in | std::ios::binary));
		tIn >> tOutputString;
	}

	boost::filesystem::remove("rsa.dat");

	BOOST_CHECK_EQUAL(tInputString, tOutputString);
}

BOOST_AUTO_TEST_CASE( RSAFilter_LongDataTest )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	string tInputString1;
	for(int i = 0; i < 2000; i++)
		tInputString1 += "0123456789";
	string tInputString2;
	for(int i = 0; i < 1000; i++)
		tInputString2 += "0123456789";
	{
		boost::iostreams::filtering_ostream tOut;
		tOut.push(RSAPublicEncryptionFilter(tPublicKey));
		tOut.push(boost::iostreams::file_sink("rsa.dat", std::ios::out | std::ios::binary));
		tOut << tInputString1 << endl;
		tOut << tInputString2 << endl;
	}

	string tOutputString1;
	string tOutputString2;
	{
		boost::iostreams::filtering_istream tIn;
		tIn.push(RSAPrivateDecryptionFilter(tPrivateKey));
		tIn.push(boost::iostreams::file_source("rsa.dat", std::ios::in | std::ios::binary));
		tIn >> tOutputString1;
		tIn >> tOutputString2;
	}

	boost::filesystem::remove("rsa.dat");

	BOOST_CHECK_EQUAL(tInputString1, tOutputString1);
	BOOST_CHECK_EQUAL(tInputString2, tOutputString2);
}

//////////////////////////////////////////////////////////////////////////////

const std::string sTestKeyPublic2048 =
	"PUBLIC:270:3082010a0282010100b73de45cccbc752827dd30125217ae0fe597f5991"
	"aa7a23ec1c6ae29dfe02f3cf9d777d663ff1d8e8f57b043d853e5cd142fbc3f5219358"
	"f8e443d6ac0eacb2c29172de2bbcccb06a8d7cc71008a036530295e5a64d8c10548480"
	"c4167b2df3459a2c4f4e7ceee84f38c2365258cefd8c6dbdd5622380e5853d364265ee"
	"8e5ed9768ac34b365c83c27b49656ab133f75ef106effa5edb232321e4dab3a1ccead8"
	"742bcc024d187a2b002bb7679071939080d76304436d45ac15dbdf5e2dbd758a523c0d"
	"2cfe75ef029f79250310fb9cbb4839e715daa3c1db230c5c67cb9d909aebdf5759ac92"
	"933bd12d798d339152aedea55c0443dfb2385b19ece783f99f50203010001;";

const std::string sTestKeyPrivate2048 =
	"PRIVATE:1192:308204a40201000282010100b73de45cccbc752827dd30125217ae0fe"
	"597f5991aa7a23ec1c6ae29dfe02f3cf9d777d663ff1d8e8f57b043d853e5cd142fbc3"
	"f5219358f8e443d6ac0eacb2c29172de2bbcccb06a8d7cc71008a036530295e5a64d8c"
	"10548480c4167b2df3459a2c4f4e7ceee84f38c2365258cefd8c6dbdd5622380e5853d"
	"364265ee8e5ed9768ac34b365c83c27b49656ab133f75ef106effa5edb232321e4dab3"
	"a1ccead8742bcc024d187a2b002bb7679071939080d76304436d45ac15dbdf5e2dbd75"
	"8a523c0d2cfe75ef029f79250310fb9cbb4839e715daa3c1db230c5c67cb9d909aebdf"
	"5759ac92933bd12d798d339152aedea55c0443dfb2385b19ece783f99f502030100010"
	"2820100366572283642efc886e4fb8344c970cb994d9e03e04cc936361203dbbc70f1b"
	"4f93a048229572f70058f736d00880f5cecd42adee541448c24208fd6e4040f6241a47"
	"6e3aafa657ec99a9e3f2178d6ff2fe7e7f18fa464d06b9fafbbaecc03c4a7a351329ea"
	"9e856a1720583de3cd36a97c5e713cdbeb44b763ef7a9085dbcfb6493bc60740520e3b"
	"2771098f532e496954aaa06dd0c72335118f6649e1d60c91bd655cf4e0e43b64d5045d"
	"1fb486ffb35857bac1769880ad0a6c1d76ac35eda827e98810c9f816587c6336badd67"
	"ae3c3e98aeab600efedce2e2f25f2a83b93d78882a5525cd09e6181bf1eb9e9c37308a"
	"dc3c8bb9ae4dde5fe2b315ea9b44102818100f34bdbe3dbfe1b870a7997eebbd4aef51"
	"fc69bdc1b60316bab11dd2304a6091c8cff10a4b0c7619afe239e7eb4f57f47301f77c"
	"27c4389d7381ea6e5cda35d0c973afaeafdae999ffefd306cd377617c8a9dfb546fdb0"
	"3ece4fe580f11a74fda4cc4e0095c6873ea6d62a9f93d1d7925f68baf5745b70789ab7"
	"50a7ef6f39bf102818100c0cf48b6d7691573b61f7dd5530ddaaf570ac7219b1d85332"
	"c86f07dd54d1990716c2a75e8358c6f8e19568be09e0a0f74f118c3c56dca9843d075a"
	"1951931123b2246aab60abde78757734c6b949fea3a13f8824040a5b31c0457cae53f5"
	"316d72bf72c694c9a233dec25906a01ccfc3015f81aeb58cc5a090712fb25b9b245028"
	"1810083af3b85493d7088d71299398e50bfa1cd61225b30361089c63e15032f30ef1d9"
	"4f3397eea70005473d9b39fc1325f81987a6fc7c49a89b1039f1396ae870d861314c95"
	"b0793c972e0010a84dcb0710387c39a954935b21a3697b3f1f40c2b1c6d10e2722ea21"
	"a9b78393ee6b83d8ba5109cea7ac2311ed3550a3474241617010281801257e62817085"
	"55523df8d1e2d7fa4ed878f6bf0838c646ec0eeb9948f257da82352521bd392bca0a45"
	"93ec5a202071cff582cadfb6b7d99cce1d93f1d2b3f62e7a3542bcb93ec0b49f204f32"
	"14d0b85c22ec236ca1c1c6794e3c8e271bafb91b1b8ea141c948019983b8e3b560a83d"
	"0fe5628bb8ee66cd2a8073744138bbe2d02818100d0b8bba20fa398e3be86f0960915e"
	"f009415b3e8e940ed6b7b842f7acec6cec9605bf354a0cb55de9db19f87806a43e4415"
	"75397d70e74fe1387e91082ab73235de478f5d362c649fe5980b48763c910331444e1e"
	"f98a542063c0b4ac50916c9c134ed345b09c8cb3a3832fa9c3921c2638d0e25e034523"
	"ac52d937139dda542;";

const string sTestDataShort = "abcdefghijklmnopqrstuvwxyz0123456789";

const uint8 sTestDataEncryptedShort[] = {
	0x9f,0x00,0xd1,0xeb,0xd5,0x4b,0x13,0xc6,0x4d,0xee,0x65,0x5e,0x1f,0xc5,
	0x76,0x39,0xc0,0xe1,0x4b,0x47,0x13,0x3a,0x8f,0xc5,0xb9,0x79,0x54,0xa3,
	0xff,0x99,0xb6,0x29,0xb3,0xb5,0x89,0xab,0x4f,0x34,0x0a,0x7c,0x58,0x7a,
	0xb3,0xd4,0x4d,0x6e,0x97,0xd4,0x8e,0x83,0xe1,0x17,0x29,0x5b,0x44,0xeb,
	0x2b,0x64,0x98,0x33,0xe4,0xcd,0xfc,0x95,0xcf,0x6a,0x69,0x69,0x9c,0xa6,
	0x69,0x6c,0xa3,0x38,0x35,0x57,0x56,0xb2,0xcf,0x88,0xda,0x25,0x23,0x17,
	0xf2,0x68,0xd0,0xa9,0x61,0x2e,0x37,0x55,0xb8,0x9d,0x0b,0xf6,0x4f,0xa3,
	0xec,0xa1,0x84,0x9e,0x3d,0x17,0x33,0x5b,0x39,0x07,0x00,0xbe,0x7d,0x34,
	0x59,0x45,0x47,0x23,0xfa,0x91,0xe7,0x67,0x9c,0x79,0xc8,0x6f,0x61,0x02,
	0xa7,0x58,0x5f,0x08,0x58,0x63,0x42,0x66,0x26,0x4b,0x36,0xc3,0xb7,0x64,
	0x3e,0xc4,0x47,0x7d,0xd1,0x76,0xc3,0x7f,0xd6,0xfd,0xda,0xbf,0x77,0x4c,
	0xa6,0x71,0x8d,0xbc,0x0d,0x84,0x91,0xfc,0x9a,0xcd,0x88,0xbc,0xfd,0xc7,
	0x58,0xf7,0x64,0xa5,0x2c,0x98,0xd1,0x43,0x75,0x93,0xf6,0xf0,0xfc,0x54,
	0xed,0x04,0xec,0x5c,0x8f,0x42,0x07,0xf4,0x44,0x5e,0x1a,0x65,0xdd,0x23,
	0x42,0x54,0x49,0x7d,0x01,0x46,0x6d,0x38,0x9f,0x7c,0xc8,0xb8,0x3c,0x2f,
	0xe9,0xf6,0xc1,0xbb,0xee,0x99,0xb3,0xbe,0xe3,0xa1,0xf1,0x81,0x5f,0xf0,
	0x1d,0xb3,0xd0,0x22,0x48,0xf1,0xd0,0x80,0xf6,0x60,0x1c,0xfb,0x70,0x72,
	0x81,0x8f,0x56,0xe4,0x42,0x18,0x69,0x84,0x46,0xb1,0x29,0xf3,0x60,0xc8,
	0x48,0x0d,0x3a,0x4d
};

const string sTestDataLong =
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ+"
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-"
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ#"
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ<"
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ>"
	"abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ!";

const uint8 sTestDataEncryptedLong[] = {
	0x56,0x27,0xeb,0x13,0xe4,0x8a,0xf9,0x25,0x0b,0x27,0x65,0x3f,0x19,0x48,
	0x35,0x13,0x02,0xfa,0x1f,0xb4,0x49,0xee,0x13,0xca,0x6f,0x7a,0x05,0xd3,
	0x96,0x01,0xa3,0x0e,0x2b,0x16,0x4b,0x3c,0x16,0x63,0x2d,0x42,0x2e,0x70,
	0x86,0xea,0xc9,0x99,0x38,0x58,0x5a,0xbd,0xef,0xf7,0x25,0xe5,0xe7,0x77,
	0x7c,0xf8,0xe2,0x01,0x8b,0xb0,0xdf,0x84,0xeb,0x40,0x12,0x40,0xb8,0x80,
	0xe3,0x61,0xba,0x15,0x4e,0x7e,0x36,0x1a,0x67,0x82,0x34,0x94,0xac,0xac,
	0xf6,0x04,0x5f,0x58,0x2e,0x0e,0xc5,0x16,0x90,0x07,0x8c,0x02,0xcb,0x2b,
	0xcb,0x81,0xa2,0xf6,0xeb,0xda,0x45,0x51,0x56,0x3b,0x88,0xb1,0x7e,0x80,
	0x3b,0x49,0x7b,0xbd,0xb7,0x3e,0xaf,0x38,0x16,0xce,0xeb,0x19,0x33,0x0a,
	0xaf,0xf6,0x7c,0xb6,0x32,0xb2,0xf9,0x2e,0x97,0x41,0xf8,0xf8,0x2c,0xdf,
	0xed,0xef,0x50,0x6c,0x35,0x40,0xe3,0xa3,0xfb,0x34,0xd2,0xd4,0xc2,0x15,
	0x3b,0x0e,0xc9,0x35,0x0a,0xa4,0xac,0x6f,0x61,0x54,0xec,0x8f,0x95,0x31,
	0x33,0x73,0xa9,0xd6,0x8c,0x7d,0x88,0x1f,0x28,0x6f,0x29,0x05,0xbe,0x5f,
	0x44,0x07,0x3b,0x9a,0x35,0x54,0x13,0xca,0x13,0xcb,0xbc,0x16,0x64,0x5c,
	0xfb,0x87,0x49,0x73,0x37,0x1d,0x7b,0x12,0x4e,0x06,0xa1,0x6f,0xac,0x88,
	0xb7,0xce,0x94,0xd0,0x72,0xb7,0x82,0x64,0xbd,0x9e,0x9c,0x90,0xe6,0x61,
	0x02,0x8d,0xde,0x54,0xd6,0xb1,0x5d,0x78,0x11,0xc0,0x28,0x6c,0x17,0x93,
	0x1e,0x9f,0xf3,0xbd,0x96,0x01,0x2d,0xe7,0xd8,0x70,0xa2,0x1d,0xe9,0x92,
	0x91,0xd8,0xd7,0x78,0x93,0xd6,0xf5,0x8c,0xfc,0xa5,0x4a,0x3c,0x53,0xf7,
	0xa4,0x90,0x19,0x08,0x11,0x26,0x32,0x6d,0xe1,0xc9,0xfe,0x13,0xb2,0x9c,
	0x48,0xbe,0xcc,0xe7,0x9a,0x1c,0xfc,0xed,0x6e,0x16,0x03,0x3a,0xe0,0x49,
	0x34,0xe2,0xff,0xc7,0x0f,0x23,0x3b,0xb0,0xd7,0xed,0xe9,0x51,0x81,0x05,
	0x8a,0xa6,0xf0,0xfa,0xda,0x3e,0xda,0xee,0xbd,0x4f,0x55,0xce,0xff,0xce,
	0xed,0x4c,0x16,0x61,0x31,0x1c,0x9c,0x04,0x54,0x1e,0x69,0x03,0xa9,0x25,
	0xcf,0xed,0xc2,0x7c,0xe7,0x40,0x2e,0x1c,0x1b,0xcc,0x2a,0xb6,0xa5,0x27,
	0xb8,0xe5,0x9b,0x9d,0xd0,0xeb,0x22,0x3b,0xef,0x87,0x1a,0x07,0x6b,0xa8,
	0x7a,0x80,0x3e,0xc9,0x44,0xd4,0x11,0x8e,0x32,0xff,0x8d,0xec,0xff,0x33,
	0x7e,0x2a,0x6a,0x41,0xdf,0x70,0x06,0xf8,0xc7,0xd2,0x4b,0x1e,0xbc,0x59,
	0xdc,0x23,0x7f,0x3e,0x64,0x2a,0xc5,0xed,0x02,0x71,0xb4,0xfc,0xe8,0xc1,
	0xee,0x7d,0x2e,0x45,0x58,0xa7,0x09,0x13,0xdd,0x96,0x65,0xb0,0xd0,0x4a,
	0xe3,0x3f,0xad,0x38,0x9e,0x8d,0x87,0xdb,0x3e,0xda,0xb8,0x7e,0xdb,0x85,
	0xe5,0x8a,0x40,0x52,0x59,0x4c,0x1d,0x3b,0x16,0x1f,0xbc,0x87,0xd2,0x1e,
	0x85,0xe5,0xe6,0x88,0x66,0x6f,0xd6,0xa4,0x72,0x53,0x5b,0x01,0x4b,0xd8,
	0x3e,0x8c,0xc1,0x08,0x5f,0x63,0xfa,0xee,0x7a,0xd5,0x6b,0x4a,0x40,0x09,
	0x4a,0x80,0x4d,0x49,0x77,0xf3,0x5c,0xc2,0xf3,0xc5,0xc1,0x45,0xfa,0x33,
	0x59,0x30,0x13,0xa6,0x66,0xa6,0xaf,0x00,0x51,0xc3,0xcb,0x08,0xbf,0x8b,
	0x8b,0xa3,0xf6,0x98,0x36,0x49,0x5f,0xf0
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSAFilter_EncryptDecryptTestShort )
{
	// parse the string to a key-pair
	RSAKey publicKey = fromString<RSAKey>(sTestKeyPublic2048);
	RSAKey privateKey = fromString<RSAKey>(sTestKeyPrivate2048);

	// decrypt the data
	boost::iostreams::array_source buffer{(char*)sTestDataEncryptedShort,
	                                      sizeof(sTestDataEncryptedShort)};
	boost::iostreams::filtering_istream fs;
	fs.push(RSAPrivateDecryptionFilter(privateKey));
	fs.push(buffer);

	string outStr;
	fs >> outStr;

	BOOST_CHECK_EQUAL(outStr, sTestDataShort);
}

BOOST_AUTO_TEST_CASE( RSAFilter_EncryptDecryptTestLong )
{
	// parse the string to a key-pair
	RSAKey publicKey = fromString<RSAKey>(sTestKeyPublic2048);
	RSAKey privateKey = fromString<RSAKey>(sTestKeyPrivate2048);

	// decrypt the data
	boost::iostreams::array_source buffer{(char*)sTestDataEncryptedLong,
	                                      sizeof(sTestDataEncryptedLong)};
	boost::iostreams::filtering_istream fs;
	fs.push(RSAPrivateDecryptionFilter(privateKey));
	fs.push(buffer);

	string outStr;
	fs >> outStr;

	BOOST_CHECK_EQUAL(outStr, sTestDataLong);
}
