/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggingAux.C
 *    Implementation of LoggingAux.h.
 *
 * @author Erik Einhorn
 * @date   2011/02/27
 */

#include <error/LoggingAux.h>

#include <error/Exception.h>
#include <error/LoggingCore.h>
#include <error/SignalHandler.h>

#include <utils/MakeString.h>
#include <platform/Typename.h>


namespace mira {

//////////////////////////////////////////////////////////////////////////////

namespace Private {

void logAuxLogger(const std::exception& info,
                  LogRecord& record, std::ostringstream& os)
{
	record.function = "";
	os.flush();
	record.message = os.str();
	record.message += ("[" + typeName(info, false) + "] ") + info.what();

	try {
		// try to cast to mira::Exception
		const Exception& e = dynamic_cast<const Exception&>(info);
		if (MIRA_LOGGER.getSeverityLevel() >= DEBUG)
		{
			std::string cs = MakeString() << e.callStack();
			record.details = "[CALLSTACK]:\n" + cs;
		}

		record.file = e.getInfo().file;
		record.line = e.getInfo().line;
		record.threadID = e.getThreadID();
	} catch(...) {
		// just a std::exception without additional information
		record.file = "";
		record.line = 0;
		record.threadID = ThreadID(0);
	}

	MIRA_LOGGER.log(record);
}

void logAuxLogger(const IntSignal& info,
                  LogRecord& record, std::ostringstream& os)
{
	record.file = "";
	record.line = 0;
	record.function = "";
	record.threadID = getCurrentThreadID();

	os.flush();
	record.message = os.str();

	switch(info.sig) {
		case SIGSEGV: record.message += "Segmentation fault!"; break;
		case SIGFPE:  record.message += "Floating point exception!"; break;
		case SIGILL:  record.message += "Illegal instruction!"; break;
		case SIGABRT: record.message += "Aborted!"; break;
		default:      record.message += "Abnormal program termination!"; break;
	}

	std::string cs = MakeString() << info.callStack;
	record.details = "[CALLSTACK]:\n" + cs;
	MIRA_LOGGER.log(record);
}

} // Private

//////////////////////////////////////////////////////////////////////////////

} // namespace
