/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLMacroProcessor.h
 *    Processor for XML macros.
 *
 * @author Adrian Kriegel
 * @date   Fri Jan 17 2025
 */

#ifndef _MIRA_XML_MACROS_XML_MACRO_PROCESSOR_H_
#define _MIRA_XML_MACROS_XML_MACRO_PROCESSOR_H_

#include <deque>
#include <stack>
#include <unordered_map>

#include <xml/macros/IMacro.h>
#include <xml/macros/Scope.h>
#include <xml/macros/Types.h>

namespace mira {

class XMLDomPreprocessor;

class XMLMacroProcessor
{
public:
	// The `macro` namespace used for bultin functions.
	static const XMLDom::NameSpace NAMESPACE;

public:
	///////////////////////////////////////////////////////////////////////////////
	// API for XMLDomPreprocessor
	///////////////////////////////////////////////////////////////////////////////

	XMLMacroProcessor(XMLDomPreprocessor&);

	XMLDomPreprocessor& getPreprocessor()
	{
		return *mPreprocessor;
	}

	/**
	 * Expand a macro.
	 * @param ioNode Node to expand in-place.
	 * @returns an iterator to the next node to process.
	 */
	XMLNode expand(XMLNode ioNode);

	bool isMacro(const XMLDom::const_iterator& node);

	/**
	 * To be called before the contents of a document are processed.
	 * @param root The original root node of the included file.
	 * @param ioDocument The root node to modify.
	 */
	void onDocumentEntered(XMLDom::const_iterator root, XMLDom::iterator ioDocument);

	/**
	 * To be called after an included document has been processed.
	 * @param targetNamespace Make macros from the included document available under the specified namespace.
	 * Empty to not import macros.
	 */
	void onDocumentLeft(const std::string& targetNamespace = "");

public:
	///////////////////////////////////////////////////////////////////////////////
	// Internal API for macro processing
	///////////////////////////////////////////////////////////////////////////////

	XMLNode processInPlace(XMLNode ioNode);

	XMLNode invokeMacro(const std::string& name, XMLNode node);

	xmlmacros::ScopePtr swapScope(xmlmacros::ScopePtr scope)
	{
		std::swap(mCurrentScope, scope);
		return scope;
	}

	[[nodiscard]] xmlmacros::Scope& getCurrentScope()
	{
		assert(mCurrentScope != nullptr);
		return *mCurrentScope;
	}

	[[nodiscard]] const xmlmacros::Scope& getCurrentScope() const
	{
		assert(mCurrentScope != nullptr);
		return *mCurrentScope;
	}

	[[nodiscard]] xmlmacros::ScopePtr getCurrentScopePtr() const
	{
		return mCurrentScope;
	}

	[[nodiscard]] xmlmacros::ScopePtr getGlobalScopePtr() const
	{
		return mGlobalScope;
	}

	/**
	 * @returns the content of the given macro as string.
	 */
	std::string coerceToString(const std::string& name);

	void pushInvocation(xmlmacros::SourceInfo&&);

	void popInvocation();

public:
	static std::tuple<xmlmacros::Identifier, std::string> getIdentifier(XMLDom::const_iterator node)
	{
		std::string nodeName = *node;
		auto dotPos = nodeName.find('.');

		std::string name = dotPos == std::string::npos ? 
			std::move(nodeName) : 
			nodeName.substr(0, dotPos);

		std::string selector = dotPos == std::string::npos ? 
			std::string() : 
			nodeName.substr(dotPos + 1);

		return {
			{node.nameSpace().prefix, std::move(name)}, 
			std::move(selector)
		};
	}

	static xmlmacros::Identifier makeIdentifier(const std::string& name)
	{
		return {NAMESPACE.prefix, name};
	}

protected:
	/**
	 * Global scope of the macro processor.
	 */
	xmlmacros::ScopePtr mGlobalScope;

	/**
	 * Current active definition scope.
	 */
	xmlmacros::ScopePtr mCurrentScope;

	/** Scopes of higher-up documents. */
	std::stack<xmlmacros::ScopePtr> mImportStack;

protected:
	/** Used to prevent infinite recursion. */
	const static uint MAX_INVOKE_DEPTH = 1000;

	/** Keeps track of the invoke-path of macros for error messages. */
	std::vector<xmlmacros::SourceInfo> mInvokeStack;

protected:
	XMLDomPreprocessor* mPreprocessor;

protected:
	xmlmacros::MacroPtr mDefineHandler;

	xmlmacros::MacroPtr mInlineImportHandler;

}; // class XMLMacroProcessor

} // namespace mira

#endif // _MIRA_XML_MACROS_XML_MACRO_PROCESSOR_H_
