/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UserDefinedMacro.h
 *    IMacro implementation for user defined macros.
 *
 * @author Adrian Kriegel
 * @date   Fri Jan 17 2025
 */

#ifndef _MIRA_XML_MACROS_USER_DEFINED_MACRO_H_
#define _MIRA_XML_MACROS_USER_DEFINED_MACRO_H_

#include <xml/macros/IMacro.h>
#include <xml/macros/Scope.h>
#include <xml/macros/Utils.h>

namespace mira::xmlmacros {

/**
 * A single parameter inside <Parameters> of a macro definition.
 */
struct Parameter
{
	std::string name;
	std::optional<CopyableXMLDom> defaultValue;

	void reflect(XMLDeserializer& r);
}; // struct Parameter

/**
 * <Parameters> of a macro definition.
 */
struct Parameters
{
	std::vector<Parameter> parameters;

	void reflect(XMLDeserializer& r);
}; // struct Parameters

/**
 * Macro definition. @see DefineMacro
 */
struct Definition
{
	std::string name;

	Parameters parameters;

	CopyableXMLDom body;

	SourceInfo source;

	void reflect(XMLDeserializer& r);

}; // struct Definition

/**
 * Invocation of a user defined macro:
 *
 * <::MyMacro Param1="Value1" Param2="Value2">
 *     <Param3>...</Param3>
 * </::MyMacro>
 */
struct Invocation
{
	const Definition* const definition;

	struct Argument
	{
		CopyableXMLDom value;
		bool isDefault = false;
		SourceInfo source;

		void reflect(XMLDeserializer& r)
		{
			source.file = r.getNode().uri();
			source.line = r.getNode().line();
			r.delegate(value);
		}
	};

	std::vector<Argument> arguments;

	Invocation(const Definition& definition) : definition(&definition) {}

	void reflect(XMLDeserializer& r);
}; // struct Invocation

/**
 * Dynamically added to the processor by the DefineMacro macro.
 */
class UserDefinedMacro : public IMacro
{
public:
	UserDefinedMacro(Definition definition, std::shared_ptr<Scope> scope)
	: mDefinition(std::move(definition)), mScope(std::move(scope))
	{}

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

	[[nodiscard]] const Definition& getDefinition() const
	{
		return mDefinition;
	}

	[[nodiscard]] ScopePtr getScope()
	{
		return mScope;
	}

	[[nodiscard]] std::optional<SourceInfo> getSource() const override
	{
		return mDefinition.source;
	}

	[[nodiscard]] bool canCoerceToString() const final
	{
		for (const auto& param : mDefinition.parameters.parameters) {
			if (!param.defaultValue.has_value()) {
				return false;
			}
		}

		return true;
	}

	void coerceToString(XMLMacroProcessor& processor, std::ostream& ss) final;

protected:
	Definition mDefinition;

	// Scope the macro was defined in.
	ScopePtr mScope;

}; // class UserDefinedMacro

/**
 * Replaces the input node with a different node.
 */
class XMLValueMacro : public IMacro
{
public:
	XMLValueMacro(ScopePtr scope, CopyableXMLDom value, SourceInfo source = SourceInfo())
	: mScope(std::move(scope)), mValue(std::move(value)), mSource(std::move(source))
	{}

	XMLNode expand(XMLMacroProcessor& processor, XMLNode node) final;

	[[nodiscard]] const XMLDom& getValue() const
	{
		return mValue;
	}

	[[nodiscard]] bool canCoerceToString() const final
	{
		return true;
	}

	void coerceToString(XMLMacroProcessor&, std::ostream& ss) final;

	[[nodiscard]] CopyableXMLDom evaluate(XMLMacroProcessor&) const;

protected:
	ScopePtr mScope;
	CopyableXMLDom mValue;
	SourceInfo mSource;
}; // class XMLValueMacro

} // namespace mira::xmlmacros

#endif // _MIRA_XML_MACROS_USER_DEFINED_MACRO_H_
