/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HumanReadableEnum.h
 *    Macros for easy creation of enumerations with conversion functions between enumeration values and human readable strings
 *
 * @author Tim van der Grinten, Ronny Stricker
 * @date   2018/03/06
 */

#ifndef _MIRA_HUMANREADABLEENUM_H_
#define _MIRA_HUMANREADABLEENUM_H_

#include <boost/preprocessor/tuple/to_seq.hpp>
#include <boost/preprocessor/seq/for_each.hpp>

#ifndef Q_MOC_RUN
#include <boost/bimap.hpp>
#include <boost/assign.hpp>
#endif

#include <utils/VariadicMacro.h>
#include <utils/ToString.h>

///////////////////////////////////////////////////////////////////////////////

/*
 * Usage:
 *
 * desired enumeration:
 * enum class Gender {
 *   UNKNOWN,
 *   FEMALE,
 *   MALE
 * }
 *
 * use in header file:
 * MIRA_HUMANREADABLE_ENUM_DECLARE(Gender, UNKNOWN, FEMALE, MALE);
 *
 * use in code file:
 * MIRA_HUMANREADABLE_ENUM_DEFINE(Gender, UNKNOWN, FEMALE, MALE);
 *
 * now you can use following functions:
 * std::string toString(const Gender& v); -> converts Gender::UNKNOWN to "UNKNOWN"
 * Gender toGender(const std::string& v); -> converts "FEMALE" to Gender::FEMALE
 */

///////////////////////////////////////////////////////////////////////////////

#define MIRA_HUMANREADABLE_ENUM_VALUE_EXPAND(VALUE) #VALUE

#define MIRA_HUMANREADABLE_ENUM_MAP_ITEM(R, NAME, VALUE) \
    (NAME::VALUE, MIRA_HUMANREADABLE_ENUM_VALUE_EXPAND(VALUE))

#define MIRA_HUMANREADABLE_ENUM_DECLARE(NAME, VALUES...) \
enum class NAME { VALUES };                              \
std::string toString(const NAME& v);                     \
NAME to##NAME(const std::string& v);

#define MIRA_HUMANREADABLE_ENUM_DEFINE(NAME, VALUES...)                 \
const boost::bimap<NAME, std::string> NAME##Map                         \
    = boost::assign::list_of<boost::bimap<NAME, std::string>::relation> \
BOOST_PP_SEQ_FOR_EACH(                                                  \
    MIRA_HUMANREADABLE_ENUM_MAP_ITEM,                                   \
    NAME,                                                               \
    BOOST_PP_TUPLE_TO_SEQ(MIRA_VARIADIC_SIZE(VALUES), (VALUES)));       \
std::string toString(const NAME& v) {                                   \
    auto it = NAME##Map.left.find(v);                                   \
    if ( it == NAME##Map.left.end() )                                   \
        MIRA_THROW(mira::XInvalidParameter, mira::toString(int(v)) +    \
                   " is not a valid value for enumeration "#NAME)       \
    return it->second;                                                  \
}                                                                       \
NAME to##NAME(const std::string& v) {                                   \
    auto it = NAME##Map.right.find(v);                                  \
    if ( it == NAME##Map.right.end() )                                  \
        MIRA_THROW(mira::XInvalidParameter, v +                         \
                   " is not a valid value for enumeration "#NAME)       \
    return it->second;                                                  \
}

///////////////////////////////////////////////////////////////////////////////

#endif
