/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Atomic.h
 *    Contains methods for simple atomic operations (such as increment, etc.)
 *    that are thread-safe without the need to lock a mutex, etc.
 *    This is basically a facade for boost/interprocess/detail/atomic.hpp
 *    which may change in future releases.
 *
 * @author Erik Einhorn
 * @date   2012/10/01
 */

#ifndef _MIRA_ATOMIC_H_
#define _MIRA_ATOMIC_H_

#ifndef Q_MOC_RUN
#include <boost/version.hpp>
#include <boost/interprocess/detail/atomic.hpp>
#endif

#include <platform/Types.h>


#if BOOST_VERSION >= 104800
	#define MIRA_BOOST_DETAIL_NAMESPACE ipcdetail
#else
	#define MIRA_BOOST_DETAIL_NAMESPACE detail
#endif

namespace mira {

namespace atomic {
///////////////////////////////////////////////////////////////////////////////

/**
 * Increments the value of the variable pointed to by var and returns its old
 * value.
 * This is the atomic equivalent for:
 * \code
 * uint32 tmp = *var;
 * *var++;
 * return tmp;
 * \endcode
 */
inline uint32 inc(volatile uint32* var) {
	return boost::interprocess::MIRA_BOOST_DETAIL_NAMESPACE::atomic_inc32(var);
}

/**
 * Decrements the value of the variable pointed to by var and returns its old
 * value.
 * This is the atomic equivalent for:
 * \code
 * uint32 tmp = *var;
 * *var--;
 * return tmp;
 * \endcode
 */
inline uint32 dec(volatile uint32* var) {
	return boost::interprocess::MIRA_BOOST_DETAIL_NAMESPACE::atomic_dec32(var);
}

/**
 * Sets the value of the variable pointed to by var in an atomic operation.
 */
inline void write(volatile uint32* var, uint32 value) {
	boost::interprocess::MIRA_BOOST_DETAIL_NAMESPACE::atomic_write32(var, value);
}

/**
 * Compare And Swap.
 * Compare the variable '*var' with 'cmp'. If both values are the same, *var is
 * replaced with 'value'. The old value of *var will be returned.
 *
 * This is the atomic equivalent for:
 * \code
 * uint32 tmp = *var;
 * if(*var == cmp)
 *     *var = value;
 * return tmp;
 * \endcode
 */
inline uint32 cas(volatile uint32* var, uint32 value, uint32 cmp) {
	return boost::interprocess::MIRA_BOOST_DETAIL_NAMESPACE::atomic_cas32(var, value, cmp);
}


///////////////////////////////////////////////////////////////////////////////
}
} // namespace

#endif
