/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryStream.h
 *  Contains the BinaryIStream and BinaryOStream classes for fast and
 *  efficient streaming of binary data.
 *
 *  Modified on: 28.06.2010 for MIRA base
 *      - adapted to coding conventions
 *      - added net() and host() manipulator
 *      - BinaryStream can now also be used with faster BinaryBufferStreamBase
 *        that completely bypasses the slow ostream and istream implementation
 *        of the STL (performance speedup: 10x)
 *
 * @author Erik Einhorn
 * @date   2009/04/04
 */

#ifndef _MIRA_BINARY_STREAM_H_
#define _MIRA_BINARY_STREAM_H_

#include <iostream>
#include <string.h> // for strlen

#include <platform/Endian.h> // for conversion to network byte order

#include <utils/Buffer.h>
#include <utils/EnumToFlags.h>

#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Helper class that is a base for binaryostream and binaryistream
 * in the same way as ios_base is a base of ostream and istream.
 * It defines additional format flags for the binary mode and
 * provides the setf() and unsetf() method for setting and clearing
 * these flags.
 *
 * @note This class must use the same naming conventions as the original
 *       std::ios_base of the STL (e.g. setf, unsetf) !!
 *
 * @ingroup StreamModule
 */
class BinaryIosBase
{
public:

	typedef enum
	{
		none = 0,
		net  = 1L << 0
	} fmtflags;
	MIRA_ENUM_TO_FLAGS_INCLASS(fmtflags)

	BinaryIosBase() : mFmtFlags(none) {}


	/**
	 * @brief  Setting new format flags.
	 * @param  flags  Additional flags to set.
	 * @return  The previous format control flags.
	 *
	 * This function sets additional flags in format control. Flags that
	 * were previously set remain set.
	 *
	 * @note This method has the same behavior as ios_base::setf() but
	 *       works for our additional binary format flags only!
	 */
	fmtflags setf(fmtflags flags)
	{
		fmtflags old = flags;
		mFmtFlags |= flags;
		return old;
	}

	/**
	 *  @brief  Clearing format flags.
	 *  @param  mask  The flags to unset.
	 *
	 *  This function clears @a mask in the format flags.
	 */
	void unsetf(fmtflags mask)
	{
		mFmtFlags &= ~mask;
	}


protected:

	// our format flags that were set by the manipulators or the setf() unsetf() methods
	fmtflags mFmtFlags;

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Although this class is not a template we must keep its methods inline for
 * performance reasons! write() and read() are called very often!
 *  @ingroup StreamModule
 */
class BinaryBufferStreamBase
{
public:

	// a couple of stl conform typedefs
	typedef Buffer<uint8> buffer_type;
	typedef buffer_type::value_type char_type;
	typedef std::size_t pos_type;

public:
	BinaryBufferStreamBase(buffer_type* buffer) :
		mBuffer(buffer), mGetPos(0), mPutPos(0), mFailBit(false)
	{
		if(mBuffer->size()==0)
			mBuffer->resize(1);
	}

public:

	// access to internal data buffer
	buffer_type* rdbuf()
	{
		return mBuffer;
	}

public:

	// keep this inline for performance reasons!!!
	void write(const char* data, std::size_t size)
	{
		std::size_t npos = mPutPos+size;

		// resize the vector buffer if necessary
		if(npos>mBuffer->size())
			mBuffer->resize(npos);

		memcpy(mBuffer->data()+mPutPos, data, size);
		mPutPos = npos;
	}

	// keep this inline for performance reasons!!!
	void read(char* data, std::size_t size)
	{
		std::size_t npos = mGetPos+size;

		// make sure that we do not read above end of buffer
		// (see istream::read)
		if(npos>mBuffer->size()) {
			size = mBuffer->size() - mGetPos;
			npos = mBuffer->size();
			mFailBit = true;
		}

		memcpy(data, mBuffer->data()+mGetPos, size);
		mGetPos = npos;
	}

public:

	std::size_t tellg() const {
		return mGetPos;
	}

	BinaryBufferStreamBase& seekg(std::size_t pos) {
		mGetPos = pos;
		return *this;
	}

	std::size_t tellp() const {
		return mPutPos;
	}

	BinaryBufferStreamBase& seekp(std::size_t pos) {
		mPutPos = pos;
		return *this;
	}

public:

	// keep this inline for performance reasons!!!
	bool eof() const {
		return mPutPos == mBuffer->size();
	}

	bool fail() const {
		return mFailBit;
	}

private:

	buffer_type* mBuffer;
	std::size_t  mGetPos;
	std::size_t  mPutPos;
	bool mFailBit;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Output stream adapter that can be assigned to any output stream
 *        and allows binary output using the << stream operators.
 *
 * This class can be used with different stream "underlays". The underlay
 * provides the actual stream functionality and stores or transmits the data.
 * Currently the binary stream comes in two flavors:
 * - \ref mira::BinaryStlOstream "BinaryStlOstream", which uses stl::ostream
 *   as underlay
 * - \ref mira::BinaryBufferOstream "BinaryBufferOstream", which uses
 *    BinaryBufferStreamBase as underlay
 *
 * BinaryStlOstream can be used as adapter to any existing STL conform stream
 * (like file streams, string streams, zip-streams, etc). The resulting binary
 * stream object can be used to put data into the underlying stream in binary
 * format using the <<-operator. Additionally, there exist special manipulators
 * to control the byte-order: 'net' will set the binary stream into network
 * byte order (big endian) mode, while 'host' sets it into host byte order
 * mode (the default).
 *
 * Example:
 * \code
 * ofstream ofs("myfile.bin");
 * BinaryStlOstream os(ofs); // create binary stream and attach to file stream
 *
 * int a = 4;
 * os << a;
 *
 * char c = 'A';
 * os << 1.234f << c;
 *
 * os << "This is a string";
 *
 * uint32 x = 0x12345F;
 * os << net << x; // write x in network byte order (Big Endian)
 * os << host;     // set stream back to host byte order
 * \endcode
 *
 * BinaryBufferOstream is similar to BinaryStlOstream but stores the data
 * in a Buffer.
 *
 * Example:
 * \code
 * Buffer<uint8> buf;
 * BinaryBufferOstream bos(&buf);
 *
 * os << "This is a string";
 * \endcode
 *
 * @see BinaryIstream
 * @ingroup StreamModule
 */
template <typename StreamUnderlay = BinaryBufferStreamBase>
class BinaryOstream : public StreamUnderlay,
                      public BinaryIosBase // used as mixin
{
	typedef StreamUnderlay Base;

public:

	// a couple of stl conform typedefs
	typedef typename Base::char_type char_type;
	typedef decltype(std::declval<Base>().rdbuf()) streambuffer_pointer;
	typedef typename StreamUnderlay::pos_type pos_type;

public:

	/**
	 * Constructor.
	 * Constructs an BinaryOstream based on the specified stream buffer
	 * object. The stream buffer object must be specified as
	 * pointer. If you use BinaryOstream based on STL streams, then the
	 * stream buffer object must be of the type std::basic_streambuf<_CharT, _Traits>
	 * (see corresponding documentation of STL streams). If you use
	 * BinaryOstream based on BinaryBufferStreamBase, the stream buffer object
	 * must be of the type std::vector<uint8>.
	 */
	BinaryOstream(streambuffer_pointer buffer) : Base(buffer) {}

	/**
	 * Constructor.
	 * The underlying output stream must be specified as parameter.
	 */
	BinaryOstream(Base& s) : Base(s.rdbuf()) {}

	/**
	 * Constructor specifying the stream and the current write position.
	 */
	BinaryOstream(Base& s, pos_type pos) : Base(s.rdbuf()) { this->seekp(pos); }

	/// stream operator for the built-in C++ datatypes
	BinaryOstream& operator<<(const char* v)          {return writeString(v);}
	BinaryOstream& operator<<(const std::string& v)   {return writeString(v);}
	BinaryOstream& operator<<(const bool& v)          {return toBinary<bool>(v);}
	BinaryOstream& operator<<(const char& v)          {return toBinary<char>(v);}
	BinaryOstream& operator<<(const uint8& v)         {return toBinary<uint8>(v);}
	BinaryOstream& operator<<(const uint16& v)        {return toBinary<uint16>(v);}
	BinaryOstream& operator<<(const uint32& v)        {return toBinary<uint32>(v);}
	BinaryOstream& operator<<(const uint64& v)        {return toBinary<uint64>(v);}
	BinaryOstream& operator<<(const int8& v)          {return toBinary<int8>(v);}
	BinaryOstream& operator<<(const int16& v)         {return toBinary<int16>(v);}
	BinaryOstream& operator<<(const int32& v)         {return toBinary<int32>(v);}
	BinaryOstream& operator<<(const int64& v)         {return toBinary<int64>(v);}
	BinaryOstream& operator<<(const float& v)         {return toBinary<float>(v);}
	BinaryOstream& operator<<(const double& v)        {return toBinary<double>(v);}
	template<typename T>
	BinaryOstream& operator<<(const Buffer<T>& v) {
		uint64 size = v.size();
		*this << size;
		write(v.data(), v.size());
		return *this;
	}

	// Interface for manipulators
	/**
	 * This stream operator is for internal use only. It provides an interface for
	 * stream manipulators like net and host. The manipulators are passed as
	 * function pointers. The passed manipulator functions are then called with
	 * the binary stream object as parameter (same mechanism as ios_base with
	 * manipulators hex, oct, etc).
	 */
	BinaryOstream& operator<<(BinaryIosBase& (*manipFn) (BinaryIosBase&))
	{
		manipFn(*this); // call the manipulator
		return *this;
	}

public:

	template <typename T>
	void write(const T* data, std::size_t count)
	{
		static_assert(sizeof(char_type)==1, "binaryostream must have char_type of size 1");
		Base::write(reinterpret_cast<const char*>(data), count*sizeof(T));
	}

	/**
	 * Method for explicitly writing a C-string to the stream.
	 * <pre>
	 *     stream.writeString(string)
	 * has the same effect as
	 *     stream << string;
	 * </pre>
	 */
	BinaryOstream& writeString(const char* value)
	{
		writeString(value, (uint32)strlen(value));
		return *this;
	}

	/**
	 * Method for explicitly writing a STL string to the stream.
	 * <pre>
	 *     stream.writeString(string)
	 * has the same effect as
	 *     stream << string;
	 * </pre>
	 */
	BinaryOstream& writeString(const std::string& value)
	{
		writeString(value, (uint32)value.length());
		return *this;
	}

	/**
	 * Method for explicitly writing the first 'length' characters of
	 * the C-string 'value'.
	 */
	BinaryOstream& writeString(const char* value, uint32 length)
	{
		this->operator<<(length); // write the length
		write(value, length);
		return *this;
	}

	/**
	 * Method for explicitly writing the first 'length' characters of
	 * the STL-string 'value'.
	 */
	BinaryOstream& writeString(const std::string& value, uint32 length)
	{
		this->operator<<(length); // write the length
		write(value.c_str(), length);
		return *this;
	}

protected:

	/**
	 * Writes every type T to this binary stream. Can be used for
	 * native C++ types or POD.
	 */
	template<typename T>
	BinaryOstream& toBinary(const T& value)
	{
		if(mFmtFlags & net) { // write in network byte order
			// convert first
			typename NetworkTypeTrait<T>::Type v = hostToNetwork<T>(value);
			write(&v, 1);
		} else // write in host byte order
			write(&value, 1);
		return *this;
	}
};

/**
 * Typedef for binary output streams based on STL streams. See BinaryOstream
 * for details.
 * @ingroup StreamModule
 */

using BinaryStlOstream = BinaryOstream<std::basic_ostream<char, std::char_traits<char>>>;

/**
 * Typedef for binary output streams based on a Buffer. This is up to
 * one magnitude faster than when using STL streams. See BinaryOstream for
 * details.
 * @ingroup StreamModule
 */

using BinaryBufferOstream = BinaryOstream<BinaryBufferStreamBase>;

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Input stream adapter that can be assigned to any input stream
 *        and allows binary input using the >> stream operators.
 *
 * This class can be used with different stream "underlays". The underlay
 * provides the actual stream functionality and stores or transmits the data.
 * Currently the binary stream comes in two flavors:
 * - \ref mira::BinaryStlIstream "BinaryStlIstream", which uses stl::istream
 *   as underlay
 * - \ref mira::BinaryBufferIstream "BinaryBufferIstream", which uses
 *    BinaryBufferStreamBase as underlay
 *
 * BinaryStlIstream can be used as adapter to any existing STL conform stream
 * This class can be used as adapter to any existing STL conform stream (like
 * file streams, string streams, zip-streams, etc). The resulting binary
 * stream object can be used to read data from the underlying stream in binary
 * format using the >>-operator. Additionally, there exist special manipulators
 * to control the byte-order: 'net' will set the binary stream into network
 * byte order (big endian) mode. Afterwards all read values are interpreted as
 * Big Endian values. 'host' sets it into host byte order mode (the default).
 *
 * Example:
 * \code
 * // reads the file that was written in the binaryostream example:
 *
 * ifstream ifs("myfile.bin");
 * BinaryStlIstream is(ifs); // create binary stream and attach to file stream
 *
 * int a;
 * is >> a;       // a=4
 *
 * float f;
 * char c;
 * is >> f >> c;  // f=1.234f, c='A'
 *
 * string s;
 * is >> s;       // s="This is a string"
 *
 * uint32 x
 * is >> net >> x; // x = 0x12345F;
 *
 * is >> host;     // set stream back to host byte order
 * \endcode
 *
 * BinaryBufferIstream is similar to BinaryStlIstream but reads the data
 * from a Buffer.
 *
 * Example:
 * \code
 * Buffer<uint8> buf;
 * ... // fill the buffer with data
 * BinaryBufferIstream is(&buf);
 *
 * std::string s;
 * os >> s; // s="This is a string"
 * \endcode
 *
 * @see BinaryOstream
 * @ingroup StreamModule
 */
template <typename StreamUnderlay = BinaryBufferStreamBase>
class BinaryIstream : public StreamUnderlay,
                      public BinaryIosBase // used as mixin
{
	typedef StreamUnderlay Base;

public:

	typedef typename Base::char_type char_type;
	typedef decltype(std::declval<Base>().rdbuf()) streambuffer_pointer;
	typedef typename StreamUnderlay::pos_type pos_type;

public:

	/**
	 * Constructor.
	 * Constructs a BinaryIstream based on the specified stream buffer
	 * object. The stream buffer object must be specified as
	 * pointer. If you use BinaryIstream based on STL streams, then the
	 * stream buffer object must be of the type std::basic_streambuf<_CharT, _Traits>
	 * (see corresponding documentation of STL streams). If you use
	 * BinaryIstream based on BinaryBufferStreamBase, the stream buffer object
	 * must be of the type std::vector<uint8>.
	 */
	BinaryIstream(streambuffer_pointer buffer) : Base(buffer) {}

	/**
	 * Constructor.
	 * The underlying input stream must be specified as parameter.
	 */
	BinaryIstream(Base& s) : Base(s.rdbuf()) {}

	/**
	 * Constructor specifying the stream and the current read position.
	 */
	BinaryIstream(Base& s, pos_type pos) : Base(s.rdbuf()) { this->seekg(pos); }

	/// stream operator for the built-in C++ datatypes
	BinaryIstream& operator>>(char* v)          { return stringFromBinary(v); }
	BinaryIstream& operator>>(std::string& v)   { return stringFromBinary(v); }
	BinaryIstream& operator>>(bool& v)          { return fromBinary<bool>(v);}
	BinaryIstream& operator>>(char& v)          { return fromBinary<char>(v);}
	BinaryIstream& operator>>(uint8& v)         { return fromBinary<uint8>(v);}
	BinaryIstream& operator>>(uint16& v)        { return fromBinary<uint16>(v);}
	BinaryIstream& operator>>(uint32& v)        { return fromBinary<uint32>(v);}
	BinaryIstream& operator>>(uint64& v)        { return fromBinary<uint64>(v);}
	BinaryIstream& operator>>(int8& v)          { return fromBinary<int8>(v);}
	BinaryIstream& operator>>(int16& v)         { return fromBinary<int16>(v);}
	BinaryIstream& operator>>(int32& v)         { return fromBinary<int32>(v);}
	BinaryIstream& operator>>(int64& v)         { return fromBinary<int64>(v);}
	BinaryIstream& operator>>(float& v)         { return fromBinary<float>(v);}
	BinaryIstream& operator>>(double& v)        { return fromBinary<double>(v);}
	template<typename T>
	BinaryIstream& operator>>(Buffer<T>& v) {
		uint64 size = 0;
		*this >> size;
		if(size!=0 && this->eof())
			MIRA_THROW(XIO, "Failed to read buffer from binary stream, premature end of stream");
		v.resize((std::size_t)size);
		read(v.data(), v.size());
		return *this;
	}

	/**
	 * This stream operator is for internal use only. It provides an interface for
	 * stream manipulators like net and host. The manipulators are passed as
	 * function pointers. The passed manipulator functions are then called with
	 * the binary stream object as parameter (same mechanism as ios_base with
	 * manipulators hex, oct, etc).
	 */
	BinaryIstream&
	operator>>(BinaryIosBase& (*manipFn) (BinaryIosBase&))
	{
		manipFn(*this); // call the manipulator
		return *this;
	}

public:

	template <typename T>
	void read(T* data, std::size_t count)
	{
		static_assert(sizeof(char_type)==1, "binaryistream must have char_type of size 1");
		Base::read(reinterpret_cast<char*>(data), count*sizeof(T));
	}


protected:

	/**
	 * Reads every type T from this binary stream. Can be used for
	 * native C++ types or POD only !
	 */
	template<typename T>
	BinaryIstream& fromBinary(T& value)
	{
		if(mFmtFlags & net) { // read in network byte order
			typename NetworkTypeTrait<T>::Type v;
			read(&v, 1);
			value = networkToHost<T>(v); // convert first
		} else // read in host byte order
			read(&value, 1);
		return *this;
	}

	/**
	 * Special method for reading C-strings (pointer value must point to
	 * memory address with enough memory for storing the read string.
	 */
	BinaryIstream& stringFromBinary(char* value)
	{
		uint32 l;
		this->operator>>(l);  // read the length
		if(this->eof())
			MIRA_THROW(XIO, "Failed to read string from binary stream, premature end of stream");
		read(value, l);
		value[l]='\0';
		return *this;
	}

	/**
	 * Special method for reading STL-strings.
	 */
	BinaryIstream& stringFromBinary(std::string& value)
	{
		uint32 l;
		this->operator>>(l); // read the length

		if(this->eof())
			MIRA_THROW(XIO, "Failed to read string from binary stream, premature end of stream");

		value.resize(l);
		read((char*)value.c_str(), l);
		((char*)value.c_str())[l]='\0';
		return *this;
	}
};

/**
 * Typedef for binary input streams based on STL streams. See BinaryIstream
 * for details.
 * @ingroup StreamModule
 */

using BinaryStlIstream = BinaryIstream<std::basic_istream<char, std::char_traits<char>>>;

/**
 * Typedef for binary input streams based on a Buffer. This is up to
 * one magnitude faster than when using STL streams. See BinaryIstream
 * for details.
 * @ingroup StreamModule
 */

using BinaryBufferIstream = BinaryIstream<BinaryBufferStreamBase>;

///////////////////////////////////////////////////////////////////////////////
/// Manipulators:
///////////////////////////////////////////////////////////////////////////////

/**
 * Manipulator that sets a binary input/output stream into network byte order
 * mode.
 *
 * Example:
 * \code
 *     BinaryBufferOstream s(...);
 *
 *     s << net << 12345678;
 * \endcode
 *
 *  @ingroup StreamModule
 */
inline BinaryIosBase& net(BinaryIosBase& stream)
{
	stream.setf(BinaryIosBase::net);
	return stream;
}

/**
 * Manipulator that sets a binary input/output stream into host byte order
 * mode and can be used to reset a previous set network byte order mode.
 * The host byte order mode is the default.
 *
 * Example:
 * \code
 *     BinaryBufferOstream s(...);
 *
 *     s << host << 12345678;
 * \endcode
 *
 *  @ingroup StreamModule
 */
inline BinaryIosBase& host(BinaryIosBase& stream)
{
	stream.unsetf(BinaryIosBase::net); // remove net-format flag
	return stream;
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
