/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SetterNotify.h
 *    Provides a special setter, that simply sets the value of the member
 *    to the given value AND calls a user defined notification method
 *    afterwards.
 *
 * @author Erik Einhorn
 * @date   2012/05/08
 */

#ifndef _MIRA_SETTERNOTIFY_H_
#define _MIRA_SETTERNOTIFY_H_

#include <serialization/GetterSetter.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

template<typename T>
void setterNotifyHelper(const T& value, T* member, boost::function<void ()> notifyFn)
{
	*member = value;
	notifyFn();
}

}
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Creates a setter that simply sets the value of the specified member to the
 * given value AND calls a user defined notification method afterwards.
 *
 * Example:
 * \code
 * r.property("Foo", mFoo, setterNotify(mFoo, boost::bind(&MyClass::valueChanged, this)),"");
 * \endcode
 *
 * @see @ref SerializationPage
 */
template<typename T>
Setter<T> setterNotify(T& member, boost::function<void ()> notifyFn)
{
	boost::function<void(const T&)> fn =
			boost::bind(&Private::setterNotifyHelper<T>, _1, &member, notifyFn);
	return setter<T>(fn);
}

/**
 * @ingroup SerializationModule
 * Creates a setter that simply sets the value of the specified member to the
 * given value AND calls a user defined notification method afterwards.
 *
 * Same as above method but with easier binding to member functions.
 *
 * Example:
 * \code
 * r.property("Foo", mFoo, setterNotify(mFoo, &MyClass::valueChanged, this),"");
 * \endcode
 *
 * @see @ref SerializationPage
 */
template<typename T, typename Class>
Setter<T> setterNotify(T& member, void (Class::*f)(), Class* obj)
{
	boost::function<void ()> notifyFn = boost::bind( f, boost::ref(*obj));
	return setterNotify(member, notifyFn);
}

/**
 * @ingroup SerializationModule
 * Creates a setter that simply sets the value of the specified member to the
 * given value AND calls a user defined notification method afterwards.
 *
 * Same as above method but with easier binding to member functions.
 *
 * Example:
 * \code
 * r.property("Foo", mFoo, setterNotify(mFoo, &MyClass::valueChanged, this),"");
 * \endcode
 *
 * @see @ref SerializationPage
 */
template<typename T, typename Class>
Setter<T> setterNotify(T& member, void (Class::*f)() const, Class* obj)
{
	boost::function<void ()> notifyFn = boost::bind( f, boost::ref(*obj));
	return setterNotify(member, notifyFn);
}

///////////////////////////////////////////////////////////////////////////////

}
#endif
