/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyNode.h
 *    Declaration and implementation of the property node hierarchy.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2011/04/13
 */

#ifndef _MIRA_PROPERTYNODE_H_
#define _MIRA_PROPERTYNODE_H_

#include <serialization/PropertyHint.h>
#include <serialization/adapters/std/pair>
#include <serialization/SplitReflect.h>
#include <serialization/JSONSerializer.h>
#include <platform/Typename.h>
#include <thread/ScopedAccess.h>
#include <utils/IsCopyAssignable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declarations
class PropertySerializer;
class RootPropertyNode;

class PropertyNodeListener;
typedef std::set<PropertyNodeListener*> PropertyNodeListenerList;

template <typename T>
class TypedPropertyNode;

class PropertyNodeInfo
{
public:
	PropertyNodeInfo() {}
	PropertyNodeInfo(const std::string& id, const std::string& name,
	                 const std::string& comment, const Typename& type,
	                 bool isReadOnly, bool isVolatile) :
	                 	mID(id), mName(name), mComment(comment),
	                 	mType(type), mIsReadOnly(isReadOnly),
	                 	mIsVolatile(isVolatile) {}

	// special copy constructor, that clones the property hint
	PropertyNodeInfo(const PropertyNodeInfo& info) :
		mID(info.mID), mName(info.mName), mComment(info.mComment),
		mHint(info.mHint.clone()), mType(info.mType),
		mIsReadOnly(info.mIsReadOnly), mIsVolatile(info.mIsVolatile) {}

public:
	PropertyNodeInfo& operator=(const PropertyNodeInfo& info)
	{
		mID         = info.mID;
		mName       = info.mName;
		mComment    = info.mComment;
		mHint       = info.mHint.clone();
		mType       = info.mType;
		mIsReadOnly = info.mIsReadOnly;
		mIsVolatile = info.mIsVolatile;

		return *this;
	}

public:

	template<typename Reflector>
	void reflectCommon(Reflector& r)
	{
		serialization::VersionType v = r.version(2, this);
		r.member("ID", mID, "");
		r.member("Name", mName, "");
		r.member("Comment", mComment, "");
		r.member("Type", mType, "The typename");
		r.member("IsReadOnly", mIsReadOnly, "");
		if (v > 1)
			r.member("IsVolatile", mIsVolatile, "");
	}

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		reflectCommon(r);
		PropertyHint::AttributeValueList hints = mHint.toList();
		r.member("Hint", hints, "The property hints", REFLECT_CTRLFLAG_TEMP_TRACKING);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		reflectCommon(r);
		PropertyHint::AttributeValueList hints;
		r.member("Hint", hints, "The property hints", REFLECT_CTRLFLAG_TEMP_TRACKING);
		mHint.fromList(hints);
	}

	MIRA_SPLIT_REFLECT_MEMBER

public:

	/// Returns the unique id of this property.
	const std::string& id() const { return mID; }

	/// Returns the name of this property as specified in the reflect() method.
	const std::string& name() const { return mName; }

	/// Returns the comment that is associated with this property.
	const std::string& comment() const { return mComment; }

	/// Returns the type of this property as Typename.
	const Typename& type() const { return mType; }

	/**
	 * Returns the specified value for the given property hint attribute.
	 * If no such attribute is set in this hint, then the defaultValue is
	 * returned.
	 */
	template <typename T>
	T getHint(const std::string& attribute, const T& defaultValue = T()) const {
		return mHint.get(attribute, defaultValue);
	}

	/// Returns true if a hint with the specified attribute exists
	bool hasHint(const std::string& attribute) const {
		return mHint.has(attribute);
	}

	/// Returns true, if this property is read-only and hence, can not be modified.
	bool isReadOnly() const {return mIsReadOnly;}

	/// Returns true, if this property is volatile and hence, must be locked for access.
	bool isVolatile() const {return mIsVolatile;}

	/// For internal use by PropertySerializer only: Overrides the name of the property.
	void setName(const std::string& name) {
		mName = name;
	}

protected:
	std::string  mID;      ///< the unique id of the property
	std::string  mName;    ///< the name of the property (mostly equal to mID)
	std::string  mComment; ///< the comment that is associated to the property
	PropertyHint mHint;    ///< the specified property hints
	Typename     mType;    ///< the type of the property
	bool         mIsReadOnly; ///< indicates whether the property is read-only
	bool         mIsVolatile; ///< indicates whether the property is volatile
};

/**
 * @ingroup SerializationModule
 * Abstract base class for all derived property node classes.
 * A property node represents a property, which is a special member of an
 * object that is reflected via the "property" method to indicate that the
 * member can be changed dynamically at runtime.
 * A property node usually is part of a hierarchy of property nodes. If a
 * property has a member that itself is property, its corresponding
 * property node is added as children to the parent node.
 *
 * The property nodes are created by the PropertySerializer that is used to
 * visit all properties that are specified in the reflect() methods of the
 * objects. The PropertySerializer will create TypedPropertyNode instances
 * that implement the abstract PropertyNode class for each certain type.
 */
class PropertyNode : public PropertyNodeInfo
{
	friend class PropertySerializer; // needs to call addChild, etc

public:
	typedef std::vector<PropertyNode*> NodeList;

public:
	PropertyNode(const PropertyNodeInfo& info) :
		PropertyNodeInfo(info), mParent(NULL), mUpdated(true) {}

	PropertyNode(const std::string& id, const std::string& name,
	             const std::string& comment, const Typename& type,
	             bool isReadOnly, bool isVolatile) :
		PropertyNodeInfo(id, name, comment, type, isReadOnly, isVolatile),
		mParent(NULL), mUpdated(true) {}

	virtual ~PropertyNode();

public:

	/// Returns the parent property node (or NULL, if this is a root node)
	virtual PropertyNode* parent() { return mParent; }

	/// Returns the parent property node (or NULL, if this is a root node)
	virtual const PropertyNode* parent() const { return mParent; }

	/// Returns a vector with all child property nodes.
	virtual NodeList& children() { return mChildren; }

	/// Returns a vector with all child property nodes.
	virtual const NodeList& children() const { return mChildren; }

	/**
	 * Searches for a child property node (which may also be a child of a
	 * child of a child, etc).
	 * The "path" to the child node is specified via the provided vector,
	 * where the first item corresponds to the id of the child property, the
	 * second item corresponds to the child property of the child property,
	 * etc. The parameter 'level' specifies the item where to start within
	 * the specified vector, this parameter usually will be 0 in the first
	 * call.
	 */
	const PropertyNode* findChildNode(const std::vector<std::string>& ids,
	                                  std::size_t level = 0) const;

	/**
	 * @copydoc findChildNode(const std::vector<std::string>& ids,
	                          std::size_t level = 0) const
	 */
	PropertyNode* findChildNode(const std::vector<std::string>& ids,
	                            std::size_t level = 0);

	/**
	 * Searches for a child property node (which may also be a child of a
	 * child of a child, etc).
	 * The "path" to the property is addressed via its full name. The following
	 * example
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * addresses the child property "value" of the property "myMember"
	 * of the child property "myObject".
	 */
	const PropertyNode* findChildNode(const std::string& id) const;

	/**
	 * @copydoc findChildNode(const std::string& id) const
	 */
	PropertyNode* findChildNode(const std::string& id);

public:

	/**
	 * Returns the full qualified ID of this property, including the names
	 * of the parent properties separated by '.'
	 *
	 * If no optional parameter is specified, the name is generated up to
	 * the root of the property hierarchy. If another parent node is
	 * specified as the name is generated relative to that node.
	 */
	std::string fullID(PropertyNode* p = NULL) const
	{
		std::string s = id();
		boost::algorithm::replace_all(s, ".", "\\");
		for(const PropertyNode* n = parent(); n!=NULL && n!=p; n=n->parent())
			s = boost::algorithm::replace_all_copy(n->id(), ".", "\\") + "." + s;
		return s;
	}

public:

	/**
	 * Sets the value of the property, where the value is described as
	 * JSON value.
	 */
	virtual void setFromJSON(const json::Value& value) = 0;

	/**
	 * Returns the value of the property as JSON value.
	 */
	virtual json::Value getAsJSON() const = 0;

public:

	/**
	 * Sets the value of the property, where the value is given as string.
	 */
	virtual void setFromString(const std::string& value);

	/**
	 * Returns the value of the property as string.
	 */
	virtual std::string getAsString() const;

public:

	/**
	 * Casts this property node to a typed property node, or returns NULL
	 * if the types do not match
	 */
	template <typename T>
	TypedPropertyNode<T>* toTyped();

	/**
	 * Casts this property node to a typed property node, or returns NULL
	 * if the types do not match
	 */
	template <typename T>
	const TypedPropertyNode<T>* toTyped() const;

public:

	/**
	 * Synchronize with the reflected object (update PropertyNode when reflected content changes)
	 */
	virtual void synchronize() = 0;

public:

	virtual RootPropertyNode* getRootNode();
	virtual const RootPropertyNode* getRootNode() const;

protected:

	/**
	 * Adds the specified property node as child node.
	 * Inserts before the element at position index (or at end, if index is not a valid position).
	 * Notifies all registered listeners.
	 */
	void addChild(PropertyNode* child, int index = -1);

	/**
	 * Removes child at specific index (without deleting it).
	 * Notifies all registered listeners.
	 */
	void removeChild(int index) { removeChild(index, std::next(mChildren.begin(), index)); }

	/**
	 * Removes contiguous children, starting at index (without deleting them).
	 * Notifies all registered listeners.
	 */
	void removeChildren(int index, int count) { removeChildren(index, std::next(mChildren.begin(), index), count); }

	/**
	 * Removes child at specific index (without deleting it).
	 * Notifies all registered listeners.
	 * Use this for efficiency if iterator is already known, otherwise use removeChild(index)
	 */
	void removeChild(int index, NodeList::iterator it);

	/**
	 * Removes contiguous children, starting at index (without deleting them).
	 * Notifies all registered listeners.
	 * Use this for efficiency if iterator is already known, otherwise use removeChild(index)
	 * Iterator it must be a forward iterator to the first of the removed children (at index).
	 */
	void removeChildren(int index, NodeList::iterator it, int count);

	/**
	 * Removes all child nodes (without deleting them).
	 * Notifies all registered listeners.
	 */
	void removeAllChildren();

	/**
	 * Moves a child node from index to before element at destination.
	 * Notifies all registered listeners.
	 */
	void moveChild(int index, int destination) {
		moveChild(index, std::next(mChildren.begin(), index),
		          destination, std::next(mChildren.begin(), destination)); }

	/**
	 * Moves a child node from index to before element at destination.
	 * Notifies all registered listeners.
	 * Use this for efficiency if iterators are already known, otherwise use moveChild(index, destination)
	 */
	void moveChild(int index, NodeList::iterator it, int destination, NodeList::iterator destIt);

protected:

	/// Caller must ensure to keep listeners locked between beginAddChildren and endAddChildren!
	virtual void beginAddChildren(PropertyNodeListenerList& listeners, int index, int count);
	virtual void endAddChildren(PropertyNodeListenerList& listeners);

	/// Caller must ensure to keep listeners locked between beginRemoveChildren and endRemoveChildren!
	virtual void beginRemoveChildren(PropertyNodeListenerList& listeners, int index, NodeList::iterator it, int count);
	virtual void endRemoveChildren(PropertyNodeListenerList& listeners);

	/// Caller must ensure to keep listeners locked between beginMoveChildren and endMoveChildren!
	virtual void beginMoveChildren(PropertyNodeListenerList& listeners, int index, NodeList::iterator it, int count, int destination);
	virtual void endMoveChildren(PropertyNodeListenerList& listeners);

private:

	/// Pointer to the parent property node
	PropertyNode* mParent;

	/// Pointers to all child property nodes
	NodeList mChildren;

	/// During synchronization, mark updated nodes so the non-updated can be removed
	bool mUpdated;
};

///////////////////////////////////////////////////////////////////////////////

class PropertyNodeListener
{
public:
	virtual ~PropertyNodeListener() {}

	virtual void beginAddChildren(const PropertyNode* node, int index, int count) = 0;
	virtual void endAddChildren() = 0;

	virtual void beginRemoveChildren(const PropertyNode* node, int index, int count) = 0;
	virtual void endRemoveChildren() = 0;

	virtual void beginMoveChildren(const PropertyNode* node, int index, int count, int destination) = 0;
	virtual void endMoveChildren() = 0;
};

///////////////////////////////////////////////////////////////////////////////

/// A special node that acts only as (empty) root node for a property tree.
class RootPropertyNode : public PropertyNode
// Deriving from PropertyNode allows to use PropertyNode's mParent to refer
// to an instance of RootPropertyNode, instead of adding another pointer in
// PropertyNode (in each instance, even though it is only required in the root).
{
public:

	RootPropertyNode();

public:

	void addChild(PropertyNode* child, int index = -1);

	void removeChild(int index, NodeList::iterator it);
	void removeChild(int index);
	void removeAllChildren();

private:

	// These are not meant to ever be called
	virtual void setFromJSON(const json::Value& value);
	virtual json::Value getAsJSON() const;
	virtual void synchronize();

protected:

	virtual RootPropertyNode* getRootNode();
	virtual const RootPropertyNode* getRootNode() const;

public:

	void registerListener(PropertyNodeListener* listener);
	void unregisterListener(PropertyNodeListener* listener);

public:

	void lock();
	bool tryLock();
	void unlock();

protected:
	friend class PropertyNode;
	typedef ProtecteeMixin<PropertyNodeListenerList> ProtectedListenerList;

	ScopedAccess<ProtectedListenerList> getListeners(bool alreadyLocked = false);

private:

	ProtectedListenerList mListeners;

	boost::timed_mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Abstract base class for all typed property nodes.
 * A typed property node allows to get and set the value of the property
 * directly using the correct type of the property.
 * An untyped PropertyNode can be cast easily into a TypedPropertyNode
 * using the PropertyNode::toTyped() method.
 *
 * \copydetails mira::PropertyNode
 *
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNode : public PropertyNode
{
public:

	/// The type of the property
	typedef T value_type; // STL conform typedef

public:

	TypedPropertyNode(const PropertyNodeInfo& info) :
		PropertyNode(info) {
		assert(info.type()==typeName<value_type>());
	}

	TypedPropertyNode(const std::string& id, const std::string& name,
	                  const std::string& comment, bool isReadOnly, bool isVolatile) :
		PropertyNode(id, name, comment, typeName<value_type>(), isReadOnly, isVolatile) {}

public:

	/**
	 * Sets the property to the specified value.
	 */
	virtual void set(const value_type& value) = 0;

	/**
	 * Returns the value of the property.
	 */
	virtual value_type get() const = 0;

};

///////////////////////////////////////////////////////////////////////////////

// LockedPropertyNodeAccess encapsulates locking all access to a TypedPropertyNode's
// internal reference/pointer/accessor.
// Optionally, when (const!) access is attempted but not possible (because the node is locked
// by another thread), it can return a backup of the previous value. The backup value is
// automatically updated with each successful read (const access).

template <typename NodeType, typename ValueType>
class LockedPropertyNodeAccessCommon : boost::noncopyable {
protected:
	LockedPropertyNodeAccessCommon(NodeType* parent, const ValueType& ref)
		: p(parent), r(ref),
		  root(p->isVolatile() ? parent->getRootNode() : NULL),
		  lockFailed(false) {
		if (root && !root->tryLock()) {
			root = NULL;
			lockFailed = true;
		}
	}

	LockedPropertyNodeAccessCommon(LockedPropertyNodeAccessCommon&& other) noexcept : r(other.r), root(NULL) {
		std::swap(p, other.p);
		std::swap(root, other.root);
		std::swap(lockFailed, other.lockFailed);
	}

	~LockedPropertyNodeAccessCommon() {
		if (root)
			root->unlock();
	}

public:
	LockedPropertyNodeAccessCommon& operator=(LockedPropertyNodeAccessCommon&& other) noexcept {
		std::swap(p, other.p);
		r = other.r;
		std::swap(root, other.root);
		std::swap(lockFailed, other.lockFailed);
		return *this;
	}

protected:
	NodeType* p;
	const ValueType& r;
	RootPropertyNode* root;
	bool lockFailed;
};

template <typename NodeType, typename ValueType,
          bool ValueTypePointer = false, bool UseBackup = false>
class LockedPropertyNodeAccess : public LockedPropertyNodeAccessCommon<NodeType, ValueType> {

public:
	LockedPropertyNodeAccess(NodeType* parent, const ValueType& ref)
		: LockedPropertyNodeAccessCommon<NodeType, ValueType>(parent, ref) {}

public:
	const ValueType& operator*() const {
		if (this->lockFailed) {
			MIRA_THROW(XIO, "Property is not available (yet?)")
		}

		return this->r;
	}

	ValueType& operator*() {
		if (this->lockFailed)
			MIRA_THROW(XIO, "PropertyNode is locked");

		return const_cast<ValueType&>(this->r);
	}
};

// specialize for ValueTypePointer=false, UseBackup=true --> invalid
template <typename NodeType, typename ValueType>
class LockedPropertyNodeAccess<NodeType, ValueType, false, true> {
	static_assert(sizeof(ValueType)==0, // assertion must depend on template parameter!
	              "LockedPropertyNodeAccess: combination ValueTypePointer=false, UseBackup=true not implemented");
};

// specialize for ValueTypePointer=true, UseBackup=false
template <typename NodeType, typename ValueType>
class LockedPropertyNodeAccess<NodeType, ValueType, true, false>
	: public LockedPropertyNodeAccessCommon<NodeType, ValueType*> {

protected:
	typedef ValueType* PointerType;

public:
	LockedPropertyNodeAccess(NodeType* parent, const PointerType& ref)
		: LockedPropertyNodeAccessCommon<NodeType, PointerType>(parent, ref) {}

public:
	const ValueType& operator*() const {
		if (this->lockFailed) {
			MIRA_THROW(XIO, "Property is not available (yet?)")
		}

		return *this->r;
	}

	ValueType& operator*() {
		if (this->lockFailed)
			MIRA_THROW(XIO, "PropertyNode is locked");

		return const_cast<ValueType&>(*this->r);
	}
};

// specialize for ValueTypePointer=true, UseBackup=true
template <typename NodeType, typename ValueType>
class LockedPropertyNodeAccess<NodeType, ValueType, true, true>
	: public LockedPropertyNodeAccessCommon<NodeType, ValueType*> {

protected:
	typedef ValueType* PointerType;

public:
	LockedPropertyNodeAccess(NodeType* parent, const PointerType& ref)
		: LockedPropertyNodeAccessCommon<NodeType, PointerType>(parent, ref) {
		if (this->root)
			this->p->mBackupValue.reset(new ValueType(*this->r));
	}

public:
	const ValueType& operator*() const {
		if (this->lockFailed) {

			if (this->p->mBackupValue)
				return *this->p->mBackupValue;

			MIRA_THROW(XIO, "Property is not available (yet?)")
		}

		return *this->r;
	}

	ValueType& operator*() {
		if (this->lockFailed)
			MIRA_THROW(XIO, "PropertyNode is locked");

		return const_cast<ValueType&>(*this->r);
	}
};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
class TypedPropertyNodeImplGetSetMixinBase : public TypedPropertyNode<T>
{
	typedef TypedPropertyNode<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImplGetSetMixinBase(const std::string& id, const std::string& name,
	                                     const std::string& comment, value_type& value,
	                                     bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, isReadOnly, isVolatile), mValue(&value) {}

protected:
	// all access to mValue (except update()) goes through LockedAccess,
	// ensures locking the PropertyNode if required
	typedef LockedPropertyNodeAccess<TypedPropertyNodeImplGetSetMixinBase<T>, value_type, true,
	                                 IsCopyAssignable<value_type>::value> LockedAccess;
	friend LockedAccess;

	const LockedAccess value() const {
		return LockedAccess(const_cast<TypedPropertyNodeImplGetSetMixinBase*>(this), mValue);
	}

	LockedAccess value() {
		return LockedAccess(this, mValue);
	}

protected:

	void setValue(const value_type& value) {
		*this->value() = value;
	}

protected:
	friend class PropertySerializer;

	/**
	 * Is called by PropertySerializer to update the internal representation
	 * of the value of the property. Not meant to be called anywhere else!
	 */
	void update(T& value) {
		mValue = &value;
	}

private:
	value_type* mValue;
	std::unique_ptr<value_type> mBackupValue; // value_type may not be default constructible
};

/**
 * Partial Implementations of the get/set of TypedPropertyNode specialized
 * for normal classes / for classes that are not copyable.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T, bool /* = true */>
class TypedPropertyNodeImplGetSetMixin : public TypedPropertyNodeImplGetSetMixinBase<T>
{
	typedef TypedPropertyNodeImplGetSetMixinBase<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImplGetSetMixin(const std::string& id, const std::string& name,
	                                 const std::string& comment, value_type& value,
	                                 bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, value, isReadOnly, isVolatile) {}

protected:

	void setValue(const value_type& value) {
		/**
		 * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		 * IF YOU GET A COMPILER ERROR HERE, your class with type ValueType
		 * does not support the assignment operator. In this case, you
		 * must specialize mira::IsCopyAssignable for your type to indicate
		 * that. This will resolve this issue.
		 */
		Base::setValue(value);
	}

public:

	virtual value_type get() const {
		return *Base::value();
	}
};

template <typename T>
class TypedPropertyNodeImplGetSetMixin<T,false> : public TypedPropertyNodeImplGetSetMixinBase<T>
{
	typedef TypedPropertyNodeImplGetSetMixinBase<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImplGetSetMixin(const std::string& id, const std::string& name,
	                                 const std::string& comment, value_type& value,
	                                 bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, value, isReadOnly, isVolatile) {}

protected:

	void setValue(const value_type& value) {
		MIRA_THROW(XLogical, "Cannot set value of property, since the underlying class does not provide an assignment operator");
	}

public:

	virtual value_type get() const {
		MIRA_THROW(XLogical, "Cannot get value of property, since the underlying class does not provide an assignment operator");
	}
};

/**
 * Implementation of TypedPropertyNode.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNodeImpl : public TypedPropertyNodeImplGetSetMixin<T,IsCopyAssignable<T>::value>
{
	typedef TypedPropertyNodeImplGetSetMixin<T,IsCopyAssignable<T>::value> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, value_type& value,
	                      bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, value, isReadOnly, isVolatile) {}
public:

	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(*this->value());
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}

	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(*this->value());
	}

public:

	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		Base::setValue(value);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}

	// implemented in GetSetMixin
	//virtual value_type get() const;

private:

	/**
	 * Synchronizes this node and its children with the
	 * the actual underlying data representation.
	 */
	virtual void synchronize();
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Implementation of TypedPropertyNode for pointers.
 * @note This class is for internal use by the property serialization framework
 */
template <typename T>
class TypedPropertyNodeImpl<T*> : public TypedPropertyNode<T*>
{
	typedef TypedPropertyNode<T*> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, const value_type& value,
	                      bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, isReadOnly, isVolatile), mPointer(value) {}

protected:
	// all access to mPointer (except update()) goes through LockedAccess,
	// ensures locking if required
	typedef LockedPropertyNodeAccess<TypedPropertyNodeImpl<T*>, value_type> LockedAccess;

	const LockedAccess pointer() const {
		return LockedAccess(const_cast<TypedPropertyNodeImpl*>(this), mPointer);
	}

	LockedAccess pointer() {
		return LockedAccess(this, mPointer);
	}

public:
	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(*pointer());
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}
	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(*pointer());
	}
public:
	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		*pointer() = value;
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}
	virtual value_type get() const {
		return *pointer();
	}
protected:
	friend class PropertySerializer;

	/**
	 * Is called by PropertySerializer to update the internal representation
	 * of the value of the property. Not meant to be called anywhere else!
	 */
	void update(T* value) {
		mPointer = value;
	}
private:
	virtual void synchronize();
private:
	value_type mPointer;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Implementation of TypedPropertyNode for Accessors, i.e. getters and setters.
 * @note This class is for internal use by the property serialization framework
 */
template <typename Getter, typename Setter>
class TypedPropertyNodeImpl<Accessor<Getter,Setter>> :
		public TypedPropertyNode<typename Accessor<Getter,Setter>::value_type>
{
	typedef typename Accessor<Getter,Setter>::value_type T;
	typedef Accessor<Getter,Setter> AccessorType;
	typedef TypedPropertyNode<T> Base;
public:
	typedef typename Base::value_type value_type;
public:
	TypedPropertyNodeImpl(const std::string& id, const std::string& name,
	                      const std::string& comment, const AccessorType& value,
	                      bool isReadOnly, bool isVolatile) :
		Base(id, name, comment, isReadOnly, isVolatile), mAccessor(value) {}

protected:
	// all access to mAccessor (except update()) goes through LockedAccess,
	// ensures locking if required
	typedef LockedPropertyNodeAccess<TypedPropertyNodeImpl<Accessor<Getter,Setter>>, AccessorType> LockedAccess;

	const LockedAccess accessor() const {
		return LockedAccess(const_cast<TypedPropertyNodeImpl*>(this), mAccessor);
	}

	LockedAccess accessor() {
		return LockedAccess(this, mAccessor);
	}

public:

	virtual void setFromJSON(const json::Value& node) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		JSONDeserializer ds(node);
		ds.deserialize(*accessor());
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above deserialization
	}

	virtual json::Value getAsJSON() const {
		JSONSerializer s(true);
		return s.serialize(*accessor());
	}

public:

	virtual void set(const value_type& value) {
		if(this->isReadOnly())
			MIRA_THROW(XLogical, "Cannot set value of read-only property.");

		(*accessor()).set(value);
		synchronize(); // synchronizes this node and its children with the
		               // actual underlying data representation that
		               // might have been changed by the above assignment
	}

	virtual value_type get() const {
		return (*accessor()).get();
	}

protected:
	friend class PropertySerializer;

	/**
	 * Is called by PropertySerializer to update the internal representation
	 * of the value of the property. Not meant to be called anywhere else!
	 */
	void update(AccessorType& value) {
		mAccessor = value;
	}

private:
	virtual void synchronize();
private:
	AccessorType mAccessor;
};

///////////////////////////////////////////////////////////////////////////////

// Support for remote property management

/**
 * Special derived class of PropertyNode, that allows to handle "remote
 * properties" transparently. Remote properties, usually are not located
 * in the local process and hence not in the local process memory. Instead,
 * the calls to "get" and "set" need to be delegated to the other process,
 * where the properties are located. Therefore, this class acts as a proxy
 * that mirrors those remote properties in the local process.
 *
 * Derived classes must implement the setFromJSON() and getAsJSON() methods
 * and delegate the calls to the remote side.
 *
 * This class provides a special toTyped() method. It creates a special
 * TypedRemotePropertyNode of the correct type. For details see
 * TypedRemotePropertyNode. The AbstractRemotePropertyNode holds this
 * typed property node and will destroy it automatically.
 *
 * @note This class is for internal use by the property serialization framework
 * @ingroup SerializationModule
 */
class AbstractRemotePropertyNode : public PropertyNode
{
public:
	AbstractRemotePropertyNode(const PropertyNodeInfo& info) :
		PropertyNode(info), mTypedNode(NULL) {}

	virtual ~AbstractRemotePropertyNode() {
		delete mTypedNode;
	}

public:

	/// Adds the specified property node as child node
	void addChild(PropertyNode* child, int index = -1) {
		PropertyNode::addChild(child, index);
	}

public:

	/// Specialized method that creates a TypedRemotePropertyNode
	template <typename T>
	TypedPropertyNode<T>* toTyped();

private:

	/// Pointer to the TypedRemotePropertyNode, or NULL, if we are not typed (yet).
	PropertyNode* mTypedNode;
};

/**
 * Special TypedPropertyNode for remote properties.
 * Instances of this class are created by the toTyped() method of
 * the AbstractRemotePropertyNode class. The instances are also owned by
 * their corresponding AbstractRemotePropertyNode objects.
 *
 * This class stores the PropertyNodeInfo only AND is of the correct type
 * of the underlying property. Instances of this class have no further
 * information concerning the property hierarchy. Calls to the parent() and
 * children() methods are delegated to the associated
 * AbstractRemotePropertyNode class.
 *
 * Besides the sole PropertyNodeInfo this class only provides get() and
 * set(). These methods depend on the data type of the property. They serialize
 * the data into JSON format and delegate the data via setFromJSON() and
 * getAsJSON() to the associated AbstractRemotePropertyNode class
 * (which must handle the remote transport).
 *
 * @note This class is for internal use by the property serialization framework
 * @ingroup SerializationModule
 */
template <typename T>
class TypedRemotePropertyNode : public TypedPropertyNode<T>
{
public:
	typedef TypedPropertyNode<T> Base;
	typedef typename Base::value_type value_type;

public:

	TypedRemotePropertyNode(AbstractRemotePropertyNode* node) :
		Base(*node), mNode(node) {}

public:
	// implementation of PropertyNode that delegates to the "real"
	// RemotePropertyNode
	virtual void setFromJSON(const json::Value& node) { mNode->setFromJSON(node); }
	virtual json::Value getAsJSON() const { return mNode->getAsJSON(); }

public:
	// the following methods of PropertyNode must be delegated to the "real"
	// RemotePropertyNode, that has the parent and the children.
	virtual PropertyNode* parent() { return mNode->parent();}
	virtual const PropertyNode* parent() const { return mNode->parent(); }
	virtual PropertyNode::NodeList& children() { return mNode->children(); }
	virtual const PropertyNode::NodeList& children() const { return mNode->children(); }
	virtual void synchronize() {
		MIRA_THROW(XNotImplemented, "TypedRemotePropertyNode::synchronize() not implemented!");
	}

public:
	// implementation of TypedPropertyNode

	virtual void set(const value_type& value)
	{
		try {
			// serialize the value to json and set it remotely via setFromJSON()
			JSONSerializer s;
			json::Value jsonval = s.serialize(value);
			setFromJSON(jsonval);
		} catch(std::exception& ex) {
			MIRA_THROW(XIO, "Failed to convert value to json: " << ex.what())
		}
	}

	virtual value_type get() const
	{
		try {
			// get the value remotely via getAsJSON
			json::Value jsonval = getAsJSON();
			// and deserialize it into the value
			value_type value;
			JSONDeserializer ds(jsonval);
			ds.deserialize(value);
			return value;
		} catch(std::exception& ex) {
			MIRA_THROW(XIO, "Failed to obtain remote value: " << ex.what())
		}
	}

private:
	AbstractRemotePropertyNode* mNode;
};

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

// implementations of the PropertyNode toTyped methods
template <typename T>
inline TypedPropertyNode<T>* PropertyNode::toTyped() {

	// try a direct cast to the desired type
	TypedPropertyNode<T>* node = dynamic_cast<TypedPropertyNode<T>*>(this);
	if(node!=NULL)
		return node; // success

	// if the above cast failed, the property could still be a remote property
	AbstractRemotePropertyNode* remoteNode = dynamic_cast<AbstractRemotePropertyNode*>(this);
	if(remoteNode!=NULL)
		return remoteNode->toTyped<T>();

	// otherwise we cannot cast
	MIRA_THROW(XBadCast, "Cannot cast PropertyNode which is of "
	                     "type '" << mType << "' to requested type '"
	                     << typeName<T>() << "'");
}

template <typename T>
inline const TypedPropertyNode<T>* PropertyNode::toTyped() const {
	PropertyNode* This = const_cast<PropertyNode*>(this);
	return This->toTyped<T>();
}

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline TypedPropertyNode<T>* AbstractRemotePropertyNode::toTyped()
{
	if(mTypedNode) {
		TypedPropertyNode<T>* node = dynamic_cast<TypedPropertyNode<T>*>(mTypedNode);
		if(node==NULL)
			MIRA_THROW(XBadCast, "Cannot cast PropertyNode which is of "
			                     "type '" << mType << "' to requested type '"
			                     << typeName<T>() << "'");
		return node;
	} else {
		if(mType != typeName<T>())
			MIRA_THROW(XBadCast, "Cannot cast remote PropertyNode which is of "
			                     "type '" << mType << "' to requested type '"
			                     << typeName<T>() << "'");

		TypedRemotePropertyNode<T>* node = new TypedRemotePropertyNode<T>(this);
		mTypedNode = node;

		return node;
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
