/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Deserializer.h
 *    Contains base class for all deserializers.
 *
 * @author Erik Einhorn
 * @date   2010/07/08
 */

#ifndef _MIRA_DESERIALIZER_H_
#define _MIRA_DESERIALIZER_H_

#include <map>
#include <vector>

#include <serialization/RecursiveMemberReflector.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Is a special reflector that is used for deserialization.
 * It usually is used as a base class for concrete Deserializers like the
 * XMLDeserializer or the BinaryDeserialzer.
 *
 * It implements the required visiting methods.
 *
 * If you derive from this class and if you overwrite these methods, you
 * MUST call the method of the Deserializer base class!
 *
 * The Deserializer implements Object-Tracking, as the corresponding
 * Serializer does. It tracks all objects that were deserialized already
 * and allows to resolve pointer references that were serialized by the
 * Serializer.
 *
 * To do so, derived concrete Derserializers should overwrite the pointer()
 * method that is called when a pointer is deserialized. Then they should
 * determine if their Serializer has stored a pointer reference
 * (in the pointerReference() method of the Serializer) or if the full object
 * was stored (which might be indicated by a marker that was serialized by
 * the Serializer). If a pointer reference was stored they need to read the
 * full id of the referenced object and can use the resolve resolveReference()
 * method to obtain the pointer to that object. If the full object was stored
 * they should call the pointer() method of this Deserializer base class to
 * deserialize the full object.
 *
 *
 * <b>Implements from base classes:</b>
 * trackObject(), pushObjectTrackingStore(), popObjectTrackingStore(),
 * pointer(), pointerNormal(), pointerPolymorphic(),
 * pointerAbstract()
 *
 * <b>Methods that should be implemented:</b>
 * pointerClassType()
 *
 * @see @ref SerializationPage
 */
template <typename Derived>
class Deserializer : public RecursiveMemberReflector<Derived>
{
	typedef RecursiveMemberReflector<Derived> Base;
public:

	typedef boost::mpl::bool_<false> isReadOnly;
	typedef boost::mpl::bool_<true> isObjectTrackingSupported;

public:


	/// The constructor.
	Deserializer()
	{}

	/**
	 * Returns either the version number from value of environment variable
	 * 'MIRA_FORCE_DESERIALIZE_VERSION', or -1 (= do not force a version).
	 */
	static int forcedDeserializeVersion()
	{
		// make sure the resolve is only ever done once
		static int version = Base::checkForcedVersion("MIRA_FORCE_DESERIALIZE_VERSION");
		return version;
	}

public:

	/**
	 * Deserializes the specified object value with the given name.
	 * If you overwrite this method in a derived class you
	 * usually SHOULD call this method of the base class.
	 */
	template <typename T>
	void deserialize(const std::string& name, T& value)
	{
		// reset object tracking etc.
		mObjectNameToInstance.clear();
		mObjects.clear();
		mStdSharedPointers.clear();
		mBoostSharedPointers.clear();

		// the Derived was derived from Serializer<Derived>,
		// so we can safely cast to it ...
		this->This()->member(name.c_str(), value, "");
	}

public:

	template<typename T>
	void trackObject(T& member)
	{
		// only if tracking is enabled
		if(this->template isTrackingEnabled<T>())
			addObject(member);
	}

	template<typename T>
	void pointer(T* &pointer)
	{
		// #####################################################################
		// If you get a compiler error here, you tried to serialize a pointer on
		// a fundamental datatype like int, float, etc. For performance reasons
		// this is not allowed. Use a wrapper on the fundamental type
		// to serialize the pointer
		// #####################################################################
		static_assert(!boost::is_fundamental<T>::value,
				"Pointers on fundamental types cannot be serialized");

		// #####################################################################
		// If you get a compiler error here, you tried to serialize a pointer on
		// a pointer. This is not allowed and can not be deserialized. Try to
		// change your code to avoid pointers on pointers!
		// #####################################################################
		static_assert(!boost::is_pointer<T>::value,
				"Pointers on pointers cannot be serialized");

		// explicitly initialize the pointer with NULL, it will be created
		// in one of the methods below.
		pointer=NULL;
		Base::pointer(pointer);
	}

	template<typename T>
	void pointerNormal(T* &pointer, int typeId)
	{
		// create a new instance of T ...
		pointer = new T;
		// and let our RecursiveMemberReflector handle the pointer
		Base::pointerNormal(pointer, typeId);
	}


	template<typename T>
	void pointerPolymorphic(T* &pointer, int typeId)
	{
		// our base class was not able to obtain the typeId, since
		// we initialized the pointer with NULL above
		assert(typeId<0);

		// obtain the class type directly from the deserializer using the
		// pointerClassType() method that must read it from the data source
		std::string type = this->This()->pointerClassType();

		// try to create a new instance of the type
		pointer = T::CLASS().newInstance(type);
		if(pointer==NULL) {
			const ReflectMemberMeta& meta = this->getCurrentMemberMeta();
			MIRA_THROW(XIO, "Cannot deserialize the polymorphic member pointer '"
			          << meta.getName() << "'. The specified class '" << type
			          << "' is not known to the class factory.");
		}

		// obtain the class of the concrete created object
		const Class& c = pointer->getClass();

		// now obtain the real typeid ...
		typeId = c.getTypeId();

		// ... and let our RecursiveMemberReflector handle the pointer
		Base::pointerPolymorphic(pointer, typeId);
	}


	template<typename T>
	void pointerAbstract(T* &pointer, int typeId) {
		// #####################################################################
		// If you get a compiler error here, you tried to deserialize an abstract
		// object that was not inherited from mira::Object. To resolve this issue
		// you should inherit your class from mira::Object using the class
		// factory.
		// #####################################################################
		static_assert(sizeof(T)==0, "You tried to deserialize an abstract "
		              "class that is not an mira::Object");
	}

	/**
	 * Must be overwritten from the concrete deserializer and is called if
	 * the pointer that is deserialized is of a polymorphic type and the
	 * class type needs to be determined in order to create the correct instance
	 * of the object. The corresponding class type should have been stored by
	 * the Serializer::pointerWithClassType method.
	 */
	std::string pointerClassType() { return "";	}

	template<class T>
	void sharedPointerReset(boost::shared_ptr<T>& ptr, T* rawPtr)
	{
		sharedPointerReset( ptr, rawPtr, mBoostSharedPointers );
	}

	template<class T>
	void sharedPointerReset(std::shared_ptr<T>& ptr, T* rawPtr)
	{
		sharedPointerReset( ptr, rawPtr, mStdSharedPointers );
	}

	template<template<class> class SType, class T>
	void sharedPointerReset(SType<T>& ptr, T* rawPtr, std::map<void*,
			SType<void> >& pointerMap)
	{
		// handle simple null pointer case
		if(rawPtr==NULL) {
			ptr.reset();
			return;
		}

		// check if we have a shared pointer for this pointer already

		typedef typename boost::remove_const<T>::type TwithoutConst;
		TwithoutConst* rawConstFreePtr = const_cast<TwithoutConst*>(rawPtr);

		void* p = static_cast<void*>(rawConstFreePtr);
		auto it = pointerMap.find(p);
		if(it==pointerMap.end()) { // no, we don't
			// so create new shared pointer
			ptr.reset(rawPtr);
			// store the shared pointer
			SType<void> voidptr(ptr, p);
			pointerMap.insert(std::make_pair(p, voidptr));
		} else {
			// use existing shared pointer
			ptr = SType<T>(it->second, static_cast<T*>(it->second.get()));
		}
	}

protected:

	/// Returns true, if object tracking is enabled for the type T
	template<typename T>
	bool isTrackingEnabled() const
	{
		if constexpr (!Derived::isObjectTrackingSupported::value) {
			return false;
		}
		else if constexpr (!IsObjectTrackable<T>::value) {
			return false;
		}
		else if constexpr (std::is_base_of<mira::Object, T>::value) {
			return true;
		}
		else {
			return this->template isReflectedAsPointer<T>();
		}
	}

protected:

	typedef std::map<std::string, void*> ObjectNameToInstanceMap;
	/// maps full id names to instance pointers
	ObjectNameToInstanceMap mObjectNameToInstance;

	typedef std::vector<void*> ObjectsVector;
	/// maps the ids to the instance pointers
	ObjectsVector mObjects;

	typedef std::pair<ObjectsVector, ObjectNameToInstanceMap> TrackingState;
	std::stack<TrackingState> mTrackingStack;

public:

	void pushObjectTrackingStore()
	{
		mTrackingStack.push(std::make_pair(mObjects, mObjectNameToInstance));
	}

	void popObjectTrackingStore()
	{
		std::tie(mObjects, mObjectNameToInstance) = mTrackingStack.top();
		mTrackingStack.pop();
	}

protected:

	typedef std::map<void*, boost::shared_ptr<void> > BoostSharedPointerMap;
	BoostSharedPointerMap mBoostSharedPointers;

	typedef std::map<void*, std::shared_ptr<void> > StdSharedPointerMap;
	StdSharedPointerMap mStdSharedPointers;

protected:

	/**
	 * Resolves a pointer reference to a previously stored object
	 * with the specified full id.
	 * This is usually called by the derived concrete deserializer
	 * that has implemented the pointer() method where it detects
	 * a stored pointer reference. Instead of calling the pointer()
	 * method of this base class it then has to call this method
	 * to resolve the pointer reference.
	 * If no object with the given ID exits, an exception is thrown.
	 */
	template<typename T>
	T* resolveReference(int objectId)
	{
		if(!this->template isTrackingEnabled<T>())
			MIRA_THROW(XRuntime, "Cannot use a reference for the object with id'"
			          << objectId << "' where object tracking is disabled");

		// get object with that id

		if((std::size_t)objectId>=mObjects.size())
			MIRA_THROW(XRuntime, "Unresolved reference to unknown object with id '"
			          << objectId << "'");

		return serialization::void_downcast<T>(mObjects[objectId]);
	}

	/**
	 * Same as the above method, but here the object id can be specified
	 * as full human readable id.
	 * @note Can only be used for Reflectors that support human readable IDs,
	 *       otherwise an exception is thrown.
	 */
	template<typename T>
	T* resolveReference(const std::string& fullId)
	{
		if(!this->template isTrackingEnabled<T>())
			MIRA_THROW(XRuntime, "Cannot use a reference for the object '"
			          << fullId << "' where object tracking is disabled");

		if(!this->usesHumanReadableIDs())
			MIRA_THROW(XLogical, "Cannot use a reference for the object '"
			          << fullId << "' since Deserializer does not support "
			          "human readable IDs");

		// get object with that id
		ObjectNameToInstanceMap::iterator it = mObjectNameToInstance.find(fullId);
		if(it==mObjectNameToInstance.end())
			MIRA_THROW(XRuntime, "Unresolved reference to unknown object '"
			          << fullId << "'");

		return serialization::void_downcast<T>(it->second);
	}

private:
	/**
	 * Adds the specified object to the object pointer tracking system.
	 */
	template<typename T>
	void addObject(T& member)
	{
		assert(this->template isTrackingEnabled<T>());

		void* pointer = serialization::void_upcast(&member);
		mObjects.push_back(pointer);

		if(this->usesHumanReadableIDs()) {
			std::string fullId = this->getCurrentMemberFullID();
			//assert(mObjectNameToInstance.count(fullId)==0); // id must be unique,
			mObjectNameToInstance.emplace(fullId, pointer);
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
