/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IntervalFilter.h
 *    IntervalFilter base class/concept.
 *
 * @author Erik Einhorn
 * @date   2010/10/02
 */

#ifndef _MIRA_INTERVALFILTER_H_
#define _MIRA_INTERVALFILTER_H_

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup MathModule
 * Concept and base class for all Interpolators and Filters. Interpolators
 * and Filters can be used within the Transform framework and when obtaining
 * values from Channels, in order to interpolate or filter these values, etc.
 * See the corresponding documentation for more details.
 */
class IntervalFilter
{
public:

	/**
	 * Returns true, if the filter/interpolator is able to extrapolate.
	 */
	bool canExtrapolate() const {
		return false;
	}

	/**
	 * Derived class must return the total number of sample points they require
	 * for their filtering/interpolation.
	 * The caller must guarantee that the apply() method below is called exactly
	 * with the required number of samples.
	 */
	int samples() const {
		return 0;
	}

	/**
	 * Derived class must return the number of sample points that are desired
	 * before the interval.
	 * If canExtrapolate() yields false, then the caller must guarantee that
	 * the apply() method below is called exactly with the required number
	 * of samples before the desired interval.
	 */
	int samplesBefore() const {
		return 0;
	}

	/**
	 * Derived class must return the number of sample points that are desired
	 * after the interval.
	 * If canExtrapolate() yields false, then the caller must guarantee that
	 * the apply() method below is called exactly with the required number
	 * of samples after the desired interval.
	 */
	int samplesAfter() const {
		return 0;
	}

	/**
	 * Filters/interpolates the specified data.
	 * The data is provided as samples of data pairs within the containers x,y.
	 * The method computes the filtered data at the position xx and returns
	 * it.
	 *
	 * The caller must guarantee the following preconditions:
	 * - the containers x and y contain exactly the number of samples that is
	 *   returned by the samples() method
	 * - if canExtrapolate() yields false, the number of samples before the
	 *   interval, xx is located in, must equal exactly the number that is
	 *   returned by samplesBefore().
	 * - if canExtrapolate() yields false, the number of samples after the
	 *   interval, xx is located in, must equal exactly the number that is
	 *   returned by samplesAfter().
	 *
	 * @param[in] x The arguments of the control points
	 * @param[in] y The values of the control points for the above arguments
	 * @param[in] xx The argument where the value is filtered/interpolated for
	 */
	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx) {
		static_assert(sizeof(Tx)==0, "apply() must be implemented in derived class");
	}
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_INTERVALFILTER_H_ */
