/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TClass.h
 *    $In contrast to the VacantClass this is the "real" class specific
 *    implementation which is able to construct the class$.
 *
 * @author Ronny Stricker
 * @date   2010/10/22
 */

#ifndef _MIRA_TCLASS_H_
#define _MIRA_TCLASS_H_

#include <string>

#include <utils/HasMember.h>
#include <utils/Foreach.h>
#include <factory/Class.h>
#include <factory/HasDefaultConstructor.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

MIRA_MEMBER_DETECTOR( mNoPublicDefaultConstructor );

// *****************************************************************************
// *** TClass
// *****************************************************************************

/**
 * @brief The TClass object is the implementation of the class class for classes
 * which are available since the corresponding library is loaded.
 *
 * @ingroup ClassFactoryModule
 */
template<typename CLASS>
class TClass: public Class
{
	friend class ClassFactory;
public:
	/**
	 * @brief The constructor
	 * The constructor calls the addMetaInfo() function of the CLASS object
	 * to add the meta informations to the mMetaInfo map.
	 */
	TClass( std::string const& identifier,
	        std::string const& name,
	        bool libLoaded );

	/**
	 * @brief Return unique id for the class.
	 * @see Class::getTypeId()
	 */
	virtual int getTypeId() const;

	/**
	 * @brief Returns the platform independent C++ typename of the class.
	 * @see Class::getTypename()
	 */
	virtual Typename getTypename() const;

	/**
	 * @brief Return a new instance of the class associated with the class
	 * object. Return true class type with help of covariant return types.
	 * @see Class::newInstance()
	 */
	CLASS* newInstance() const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @see Class::newInstance( int paramCount, ... )
	 */
	virtual Object* newInstance( int paramCount, ... ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * Return class type of this class with help of covariant return types.
	 * @see Class::newInstance( std::string const& childIdentifier )
	 */
	CLASS* newInstance( std::string const& childIdentifier ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * A list of parameters for the constructor can be passed.
	 * @see Class::newInstance( std::string const& childIdentifier, int paramCount, ... )
	 */
	Object* newInstance( std::string const& childIdentifier,
	                     int paramCount, ... ) const;

	/**
	 * @brief Return true if the associated class is abstract.
	 */
	virtual bool isAbstract() const;

protected:

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * A list of parameters for the constructor can be passed as va_list.
	 */
	Object* newVAInstance( std::string const& childIdentifier,
	                       int paramCount,
	                       std::va_list ap ) const;

	/**
	 * @brief Call the specific constructor depending on the number of
	 * parameters passed to the function.
	 */
	Object* newVAInstance( int paramCount, std::va_list ap ) const;

private:
	TClass( )
	{
	}

};

// *****************************************************************************
// *** TClass implementation
// *****************************************************************************

template<typename CLASS>
inline TClass<CLASS>::TClass( std::string const& identifier,
                              std::string const& name, bool libLoaded ) :
	Class( identifier, name, libLoaded )
{
	// add meta info
	CLASS::addMetaInfo( mMetaInfo );
}

template<typename CLASS>
inline int TClass<CLASS>::getTypeId() const
{
	return typeId<CLASS>();
}

template<typename CLASS>
inline Typename TClass<CLASS>::getTypename() const
{
	return typeName<CLASS>();
}

template<typename CLASS>
inline bool TClass<CLASS>::isAbstract() const
{
	return std::is_abstract<CLASS>::value;
}


template<typename CLASS>
CLASS* TClass<CLASS>::newInstance() const
{
	// check if the class is abstract and / or do have a default constructor
	// since the check for the default constructor fails on some compilers,
	// we have to check for the mNoPublicDefaultConstructor member additionally
	typedef typename
		boost::mpl::eval_if<std::is_abstract<CLASS>,
			boost::mpl::identity<ClassFactoryAbstractClassBuilder>,
			// else
			boost::mpl::eval_if< typename HasPublicDefaultConstructor<CLASS>::type,
				boost::mpl::identity< ClassFactoryDefaultConstClassBuilder >,
				boost::mpl::identity< ClassFactoryNoDefaultConstClassBuilder > >
			>::type ClassType;

	return ClassType::template invoke<CLASS>( );
}

template<typename CLASS>
Object* TClass<CLASS>::newInstance( int paramCount, ... ) const
{
	std::va_list ap;
	va_start(ap, paramCount);
	return newVAInstance( paramCount, ap);
}

template<typename CLASS>
CLASS* TClass<CLASS>::newInstance( std::string const& childIdentifier ) const
{
	// handle the case that someone wants to create an instance of US.
	// obtain identifier by function since it might be different from
	// mIdentifier (template classes)
	if( childIdentifier == getIdentifier() )
		return newInstance();
	// check if the desired object is a registered derived child
	if ( mDerivedChildren.find( childIdentifier ) ==
			mDerivedChildren.end() ) {
		MIRA_THROW( XFactoryUnknown, "Unknown class identifier ("
		          + childIdentifier +") calling newInstance on " + getIdentifier()
		          + " !" );
	}
	Object* tObject = mDerivedChildren.find( childIdentifier )->second.newInstance();
	if ( !tObject ) {
		MIRA_THROW( XFactoryLogical, "Class creation failed (" + childIdentifier
		          + ") calling newInstance on " + getIdentifier() + "!" );
	}
	// try to cast object to desired class
	return mira_factoryDynamicCast<CLASS>( tObject );
}

template<typename CLASS>
Object* TClass<CLASS>::newInstance( std::string const& childIdentifier,
                                    int paramCount, ... ) const
{
	std::va_list ap;
	va_start(ap, paramCount);
	
	return newVAInstance( childIdentifier, paramCount, ap );
}

template<typename CLASS>
Object* TClass<CLASS>::newVAInstance( std::string const& childIdentifier,
						int paramCount,
						std::va_list ap ) const
{
	// obtain identifier by function since it might be different from
	// mIdentifier (template classes)
	if(childIdentifier==getIdentifier())
		return newVAInstance(paramCount, ap);
	// check if the desired object is a registered derived child
	if ( mDerivedChildren.find( childIdentifier ) ==
			mDerivedChildren.end() ) {
		MIRA_THROW( XFactoryUnknown, "Unknown class identifier ("
		          + childIdentifier + ") calling newInstance on " + getIdentifier()
		          + "!" );
	}

	Object* tObject = mDerivedChildren.find( childIdentifier )->second
			.newVAInstance(paramCount, ap);
	if ( !tObject ) {
		MIRA_THROW( XFactoryLogical, "Class creation failed (" + childIdentifier
		          + ") calling newInstance on " + getIdentifier() + "!" );
	}
	return tObject;
}

template<typename CLASS>
Object* TClass<CLASS>::newVAInstance( int paramCount, std::va_list ap ) const
{
	// Check if we should use the normal createClass template or the special
	// instantiation for classes with abstract functions
	typedef typename boost::mpl::eval_if<std::is_abstract<CLASS>,
			boost::mpl::identity<ClassFactoryAbstractClassBuilder>,
			boost::mpl::identity<ClassFactoryDefaultConstClassBuilder>
	>::type ClassType;

	switch(paramCount){
	case 0:
		return newInstance();
	case 1:
		return ClassType::template invoke<1,CLASS>( ap);
	case 2:
		return ClassType::template invoke<2,CLASS>( ap);
	case 3:
		return ClassType::template invoke<3,CLASS>( ap);
	case 4:
		return ClassType::template invoke<4,CLASS>( ap);
	default:
		MIRA_THROW( XFactoryLogical, "Invalid number of arguments for "
		            "constructor (" + std::to_string((uint64)paramCount) + " arguments for "
		          + getIdentifier() + ")!");
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_TCLASS_H_ */

