/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IOService.h
 *    Wrapper for boost::asio::io_service.
 *
 * @author Tim Langner
 * @date   2010/07/28
 */

#ifndef _MIRA_IOSERVICE_H_
#define _MIRA_IOSERVICE_H_

#ifndef Q_MOC_RUN
#include <boost/asio.hpp>
#endif

#include <platform/Platform.h>

///////////////////////////////////////////////////////////////////////////////
// Use a forward declaration for boost::thread to reduce compile times.

///@cond INTERNAL
namespace boost {
	class thread;
}
///@endcond INTERNAL

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Wrapper class for boost::asio::io_service.
 * @ingroup CommunicationModule
 */
class IOService
{
public:
	/**
	 * Creates an I/O service object that creates an own instance of
	 * boost::asio::io_service.
	 */
	IOService();

	/**
	 * Creates an I/O service object that uses an existing
	 * boost::asio::io_service instance
	 */
	IOService(boost::asio::io_service& service);

	/// The destructor.
	virtual ~IOService() {}

	/**
	 * Starts a given number of threads on the boost::asio::io_service::run()
	 * method.
	 * @param[in] threadCount Number of threads to start.
	 * @param[in] runInThisThread If true this method will call
	 *            boost::asio::io_service::run() and block until run() returns.
	 */
	void runThreads(std::size_t threadCount = 0, bool runInThisThread = true);

	/**
	 * Wrapper for boost::asio::io_service::run()
	 * This will block until boost::asio::io_service::run() returns.
	 */
	std::size_t run()
	{
		return mService.run();
	}

	/**
	 * Wrapper for boost::asio::io_service::run_one()
	 */
	std::size_t runOne()
	{
		return mService.run_one();
	}

	/**
	 * Wrapper for boost::asio::io_service::stop()
	 */
	void stop()
	{
		mService.stop();
	}

	/**
	 * Wrapper for boost::asio::io_service::reset()
	 */
	void reset()
	{
		mService.reset();
	}

	/**
	 * Operator for implicit casting to boost::asio::io_service
	 */
	operator boost::asio::io_service&()
	{
		return mService;
	}

	template <typename CompletionHandler>
	void post(CompletionHandler handler)
	{
		mService.post(handler);
	}

	/**
	 * Stops and joins all started threads.
	 */
	void join();

private:
	class Pimpl;
	boost::shared_ptr<Pimpl> mPimpl;

	boost::asio::io_service& mService;
	typedef boost::shared_ptr<boost::thread> ThreadPtr;
	std::vector<ThreadPtr> mThreads;
};

/// A shared pointer wrapper for class boost::asio::io_service.
typedef boost::shared_ptr<boost::asio::io_service> ServicePtr;

/// A shared pointer wrapper for class IOService.
typedef boost::shared_ptr<IOService> IOServicePtr;

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
