/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Backend.C
 *
 * @author Erik Einhorn
 * @date   2011/09/17
 */

#include <Backend.h>
#include <iostream>
#include <fstream>

#include <boost/format.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string.hpp>

#include <error/LoggingCore.h>
#include <utils/PathFinder.h>
#include <platform/Environment.h>
#include <platform/Process.h>

#include <QDialog>
#include <ui_SvnDialog.h>

// Boost >= 1.50.0 don't have the namespace boost::filesystem3.
#if BOOST_VERSION < 105000
namespace fs = boost::filesystem3;
#else
namespace fs = boost::filesystem;
#endif

using namespace mira;

Backend::Backend()
{
	mMIRARootDir = detectMIRARootDir();

	if(mMIRARootDir.empty())
		MIRA_THROW(XRuntime, "FATAL: Failed to detect the MIRA root directory. "
		           "Please make sure that your MIRA distribution is installed "
		           "correctly.");
}

Path Backend::detectMIRARootDir()
{
	// infer MIRA root dir from the path of THIS executable
	Path p = getExecutableDirectory();
	p = p.remove_filename();
	for(; !p.empty(); p=p.parent_path())
	{
		Path t = p / "mira.root";
		if(fs::exists(t))
			return p;
	}

	return Path();
}

void Backend::copyTemplate(const std::string& filename, const Path& targetDir,
                           std::string targetName, const TemplateVars& vars,
                           bool append)
{
	Path prefix = mMIRARootDir / "tools/mirawizard/etc/templates";
	Path source = prefix / filename;

	if(!fs::exists(source))
		MIRA_THROW(XIO, "Template file '" << source.string() << "' does not exist");

	if(targetName.empty())
		targetName = filename;

	// make sure the path exist
	fs::create_directories(targetDir);

	Path target = targetDir / targetName;

	if(fs::exists(target) && !append) {
		MIRA_LOG(WARNING) << "File '" << target.string() << "' already exists.";
		return;
	}

	if(vars.empty()) {
		fs::copy(source, target);
		return;
	}

	// do full template processing

	std::ifstream is(source.string());
	if(!is.is_open())
		MIRA_THROW(XIO, "Failed to read template '" << source.string() << "'");

	std::string content((std::istreambuf_iterator<char>(is)),
	                     std::istreambuf_iterator<char>());

	// replace all variables
	foreach(auto p, vars)
		boost::algorithm::replace_all(content, p.first, p.second);

	std::ofstream os(target.string(),
	                 std::ios_base::out |
	                 (append ? std::ios_base::app : std::ios_base::trunc) );
	if(!os.is_open())
		MIRA_THROW(XIO, "Failed to write target file '" << target.string() << "'");

	os << content;
}

boost::tuple<bool, std::string, std::string> Backend::svnAuth(const std::string& url)
{
	std::string username, password;
	while(true)
	{
		if(svn("info " + url, username, password)) {
			// success
			break;
		} else {
			// failure -> ask for login

			QDialog dialog;
			Ui::SvnDialog d;
			d.setupUi(&dialog);

			if(dialog.exec()==QDialog::Rejected) // abort
				return boost::make_tuple(false, "","");

			username = d.leUsername->text().toStdString();
			password = d.lePassword->text().toStdString();
		}
	}

	// if we reach here, we were successful
	return boost::make_tuple(true, username, password);

}

bool Backend::svn(const std::string& command,
                  const std::string& username, const std::string& password)
{
	std::string cmd = "svn --non-interactive --trust-server-cert ";
	if(!username.empty() && !password.empty())
		cmd += "--username " + username + " --password " + password + " ";

	cmd += command;

	return executeProcess(cmd)==0;
}

void Backend::createExternalProject(const std::string& name, const Path& path)
{
	// create the directory structure
	fs::create_directories(path / "toolboxes");
	fs::create_directories(path / "domains");

	TemplateVars vars;
	vars.push_back(std::make_pair("$PROJECTNAME$", name));
	vars.push_back(std::make_pair("$MIRA_ROOT_DIR$", mMIRARootDir.generic_string()));

	// copy the files
	copyTemplate("Makefile", path);

	copyTemplate("CMakeLists.txt", path, vars);

	copyTemplate("CMakeLists.mira", path);
	copyTemplate("FindMIRARoot.cmake", path);
	copyTemplate("CMakeLists.dir", path / "toolboxes", "CMakeLists.txt");
	copyTemplate("CMakeLists.dir", path / "domains", "CMakeLists.txt");
}

void  Backend::createDomainToolbox(const std::string& name, const std::string& desc,
                         const std::string& author, const std::string& tags,
                         const std::string& includeDir, const Path& path,
                         bool createMountDir, const std::string& mountDir,
                         bool isToolbox)
{
	// derive main path with CMakeLists.txt by moving upwards to parent directories
	Path mainPath = path.parent_path();
	while(!mainPath.empty() && !fs::exists(mainPath / "CMakeLists.txt")) {
		copyTemplate("CMakeLists.dir", mainPath, "CMakeLists.txt");
		mainPath = mainPath.parent_path();
	}

	// create the directory structure
	fs::create_directories(path / "etc");
	if (!includeDir.empty())
		fs::create_directories(path / "include" / includeDir);
	fs::create_directories(path / "src");

	TemplateVars vars;
	vars.push_back(std::make_pair("$NAME$", name));
	vars.push_back(std::make_pair("$NAME_UPPER$", boost::to_upper_copy(name)));
	vars.push_back(std::make_pair("$DESCRIPTION$", desc));
	vars.push_back(std::make_pair("$AUTHOR$", author));
	vars.push_back(std::make_pair("$TAGS$", tags));
	vars.push_back(std::make_pair("$MOUNTDIR$", mountDir));

	if(isToolbox) {
		copyTemplate("Package.toolbox", path, name+".package", vars);
		copyTemplate("CMakeLists.toolbox", path, "CMakeLists.txt", vars);
		copyTemplate("README.md", path, "README.md", vars);
		copyTemplate("ToolboxMainPage.dox", path / "doc", "MainPage.dox", vars);
		copyTemplate("components/LibraryExports.h", path / "include" / includeDir, name+"Exports.h", vars);
	} else {
		copyTemplate("Package.domain", path, name+".package", vars);
		copyTemplate("CMakeLists.domain", path, "CMakeLists.txt", vars);
		copyTemplate("README.md", path, "README.md", vars);
		copyTemplate("DomainMainPage.dox", path / "doc", "MainPage.dox", vars);
	}
	copyTemplate("Changelog.changelog", path, name+".changelog", vars);

	if(createMountDir)
		copyTemplate("mountdir.xml", path, "mountdir.xml", vars);
}

void Backend::createComponent(const std::string& componentname, const std::string& ns,
                     const std::string& desc, const std::string& author,
                     const Path& path, ComponentType type)
{
	// derive main path with PACKAGENAME.package by moving upwards to parent directories
	Path mainPath = path;
	PathVector packagefiles = findFiles(mainPath, "*.package");
	while(!mainPath.empty() && packagefiles.empty()) {
		mainPath = mainPath.parent_path();
		packagefiles = findFiles(mainPath, "*.package");
	}

	if(mainPath.empty())
		MIRA_THROW(XRuntime, "Package file not found - package name cannot be inferred");

	if(packagefiles.size() > 1)
		MIRA_THROW(XRuntime, "Multiple package files exist - package name cannot be inferred.");

	Path etcPath = mainPath/"etc";

	Path relativeSrcPath = relativizePath(path, mainPath);
	std::string relativeSrcPathStr = relativeSrcPath.string();

	if(!relativeSrcPathStr.empty() && *relativeSrcPathStr.rbegin()=='.')
		relativeSrcPathStr.resize(relativeSrcPathStr.size()-1);

	// add trailing slash
	if(!relativeSrcPathStr.empty() && *relativeSrcPathStr.rbegin()!='/')
		relativeSrcPathStr.push_back('/');

	TemplateVars vars;
	vars.push_back(std::make_pair("$AUTHOR$", author));
	vars.push_back(std::make_pair("$DESCRIPTION$", desc));

	Time t = Time::now();
	std::string dateStr = (
		boost::format("%d/%02d/%02d")
			% (int)t.date().year()
			% (int)t.date().month()
			% (int)t.date().day()
		).str();

	vars.push_back(std::make_pair("$DATE$", dateStr));
	vars.push_back(std::make_pair("$NAMESPACE$", ns));

	std::string namespaceBegin;
	std::string namespaceEnd;
	if(!ns.empty()) {
		std::vector<std::string> namespaces;
		boost::split( namespaces, ns, boost::is_any_of(":"), boost::token_compress_on);
		foreach(const std::string s, namespaces)
		{
			namespaceBegin += "namespace " + s + " { ";
			namespaceEnd   += "}";
		}
	}

	vars.push_back(std::make_pair("$NAMESPACE_BEGIN$", namespaceBegin));
	vars.push_back(std::make_pair("$NAMESPACE_END$",  namespaceEnd));
	vars.push_back(std::make_pair("$RELATIVE_SRC$", relativeSrcPathStr));

	vars.push_back(std::make_pair("$NAME$", componentname));

	std::string packagename = packagefiles.begin()->stem().string();

	vars.push_back(std::make_pair("$PACKAGE$", packagename));

	// make sure the path exist
	fs::create_directories(path);

	switch(type)
	{
		case UNIT:
			copyTemplate("components/Unit.C", path, componentname+".C", vars);
			copyTemplate("components/Unit.xml", etcPath, componentname+".xml", vars);
			copyTemplate("components/Unit.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case MICROUNIT:
			copyTemplate("components/MicroUnit.C", path, componentname+".C", vars);
			copyTemplate("components/Unit.xml", etcPath, componentname+".xml", vars);
			copyTemplate("components/Unit.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case VISUALIZATION2D:
			copyTemplate("components/Visualization2D.C", path, componentname+"Visualization2D.C", vars);
			copyTemplate("components/Visualization2D.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case VISUALIZATION3D:
			copyTemplate("components/Visualization3D.C", path, componentname+"Visualization3D.C", vars);
			copyTemplate("components/Visualization3D.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case VIEWPART:
			fs::create_directories(path / "private");
			copyTemplate("components/ViewPart.h", path/"private", componentname+".h", vars);
			copyTemplate("components/ViewPart.C", path, componentname+".C", vars);
			copyTemplate("components/ViewEditorPart.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case EDITORPART:
			fs::create_directories(path / "private");
			copyTemplate("components/EditorPart.h", path/"private", componentname+".h", vars);
			copyTemplate("components/EditorPart.C", path, componentname+".C", vars);
			copyTemplate("components/ViewEditorPart.cmake", mainPath, "CMakeLists.txt", vars, true);
			break;

		case PYTHON_UNIT:
			copyTemplate("components/PythonUnit.py", path, componentname+".py", vars);
			copyTemplate("components/PythonUnit.xml", etcPath, componentname+".xml", vars);
			break;

		default:
			break;
	}
}

Backend::EnvVars Backend::getMissingEnvVars(const Path& currentProjectPath) const
{
	EnvVars res;

	// check $MIRA_PATH
	std::string MIRA_PATH;
	try {
		MIRA_PATH=resolveEnvironmentVariable("MIRA_PATH");
	} catch(...) {}

	if(MIRA_PATH.find(mMIRARootDir.string())==std::string::npos)
		res.push_back(std::make_pair("MIRA_PATH", "$MIRA_PATH:"+mMIRARootDir.string()));

	if(!currentProjectPath.empty() && MIRA_PATH.find(currentProjectPath.string())==std::string::npos)
		res.push_back(std::make_pair("MIRA_PATH", "$MIRA_PATH:"+currentProjectPath.string()));

	Path miraBinDir = mMIRARootDir / "bin";
	Path miraLibDir = mMIRARootDir / "lib";
	Path projectLibDir = currentProjectPath / "lib";
#ifdef MIRA_LINUX
	// check $LD_LIBRARY_PATH
	std::string LD_LIBRARY_PATH;
	try {
		LD_LIBRARY_PATH=resolveEnvironmentVariable("LD_LIBRARY_PATH");
	} catch(...) {}
	// OGRE
	const char* ogrePaths[] = {"/usr/lib/OGRE", "/usr/local/lib/OGRE", NULL};
	for(int i=0; ogrePaths[i]!=NULL; ++i)
	{
		std::string p = ogrePaths[i];
		if(fs::exists(p)) {
			if(LD_LIBRARY_PATH.find(p)==std::string::npos)
				res.push_back(std::make_pair("LD_LIBRARY_PATH", "$LD_LIBRARY_PATH:"+p));
			break;
		}
	}
	// MIRA lib dir
	if(LD_LIBRARY_PATH.find(miraLibDir.string())==std::string::npos)
		res.push_back(std::make_pair("LD_LIBRARY_PATH", "$LD_LIBRARY_PATH:"+mMIRARootDir.string()+"/lib"));
	// external project lib dir
	if(LD_LIBRARY_PATH.find(projectLibDir.string())==std::string::npos)
		res.push_back(std::make_pair("LD_LIBRARY_PATH", "$LD_LIBRARY_PATH:"+projectLibDir.string()));
#endif

	// check $PATH
	std::string PATH;
	try {
		PATH=resolveEnvironmentVariable("PATH");
	} catch(...) {}
	// MIRA bin dir
	if(PATH.find(miraBinDir.string())==std::string::npos)
		res.push_back(std::make_pair("PATH", "$PATH:"+mMIRARootDir.string()+"/bin"));

#ifdef MIRA_WINDOWS
	// MIRA lib dir
	if(PATH.find(miraLibDir.string())==std::string::npos)
		res.push_back(std::make_pair("PATH", "$PATH:"+mMIRARootDir.string()+"/lib"));
#endif

	return res;
}


enum Shell {
	ShellBash,
	ShellTcsh,
	ShellWindows,
	ShellUnknown
};

Shell getShell()
{
#ifdef MIRA_WINDOWS
	return ShellWindows;
#else

	try {
		std::string SHELL =resolveEnvironmentVariable("SHELL");
		if(SHELL=="/bin/bash")
			return ShellBash;
		else if(SHELL=="/bin/tcsh" || SHELL=="/bin/csh")
			return ShellTcsh;
	} catch(...) {}
	return ShellUnknown;
#endif
}

bool canSetEnvVars(Shell sh)
{
	if(sh==ShellUnknown)
		return false;

	if(sh==ShellWindows) // not yet supported
		return false;

	Path home;
	try {
		home = getHomeDirectory();
	} catch(...) {
		return false;
	}

	Path p;
	if(sh==ShellBash)
		p = home / ".bashrc";
	else if(sh==ShellTcsh)
		p = home / ".cshrc";

	std::ofstream f(p.string(), std::ios_base::app);
	return f.is_open();

}

std::pair<std::string, bool> Backend::getEnvVarsString(const EnvVars& vars) const
{
	std::string res;

	Shell sh = getShell();
	switch(sh)
	{
	case ShellBash:
		foreach(const auto& p, vars)
			res += "export " + p.first + "=" + p.second  + "\n";
		break;
	case ShellTcsh:
		foreach(const auto& p, vars)
		{
			std::string s = boost::algorithm::replace_all_copy(p.second, ":", "\":\"");
			res += "setenv " + p.first + " " + s + "\n";
		}
		break;
	case ShellWindows:
	case ShellUnknown:
	default:
		foreach(const auto& p, vars)
			res += p.first + " = " + p.second + "\n";
		break;
	}

	return std::make_pair(res,canSetEnvVars(sh));
}

bool Backend::setEnvVarsAutomatically(const EnvVars& vars)
{
	if(vars.empty())
		return true;

	Shell sh = getShell();
	if(sh==ShellUnknown)
		return false;

	if(sh==ShellWindows) // not yet supported
		return false;

	// TODO: implement windows using setx command

	Path home;
	try {
		home = getHomeDirectory();
	} catch(...) {
		return false;
	}

	Path p;
	if(sh==ShellBash)
		p = home / ".bashrc";
	else if(sh==ShellTcsh)
		p = home / ".cshrc";

	std::ofstream f(p.string(), std::ios_base::app);
	if(!f.is_open())
		return false;

	std::string s = getEnvVarsString(vars).first;
	if(!s.empty()) {
		f << std::endl << "# generated by mirawizard on " << Time::now() << std::endl;
		f << s;
	}

	return true;
}
