/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitlabAPI.C
 *
 * @author Thomas Bauer
 * @date   2019/11/22
 */

#include <app/GitlabAPI.h>

#include <error/Exceptions.h>
#include <error/LoggingCore.h>

#include <serialization/JSONSerializer.h>
#include <serialization/Serialization.h>

#include <utils/Singleton.h>
#include <utils/MakeString.h>

#include <boost/algorithm/string.hpp>

namespace mira { namespace gitlab {

///////////////////////////////////////////////////////////////////////////////

class LibCurlSingleton : public LazySingleton<LibCurlSingleton> {
public:
	LibCurlSingleton() {
		auto e = curl_global_init(CURL_GLOBAL_DEFAULT);
		if(e != 0) {
			MIRA_THROW(XRuntime, "GitlabAPI: Could not init libcurl library!");
		}
		MIRA_LOG(NOTICE) << "Initialized libcurl version: " << curl_version();
	}

	virtual ~LibCurlSingleton() {
		curl_global_cleanup();
	}
};

///////////////////////////////////////////////////////////////////////////////

GitlabAPI::GitlabAPI(const std::string& host)
: mInstanceUrl(host), mLoggedIn(false), mCurlHandle(NULL), mAccessToken() {
	LibCurlSingleton::instance();

	mCurlHandle = curl_easy_init();
	if(mCurlHandle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}

	auto tmp = splitUrl(mInstanceUrl);

	if(!tmp[UrlHost].empty()) {
		if(tmp[UrlScheme] == "http") {
			MIRA_LOG(NOTICE) << "Change http to http_s_!";
		}
		mInstanceUrl = "https://" + tmp[UrlHost];
	} else {
		MIRA_THROW(XIO, "No host given: " << mInstanceUrl);
	}
}

GitlabAPI::GitlabAPI(const GitlabAPI& other)
: mInstanceUrl(other.mInstanceUrl), mLoggedIn(other.mLoggedIn),
  mCurlHandle(NULL), mAccessToken(other.mAccessToken) {
	mCurlHandle = curl_easy_init();
	if(mCurlHandle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}
}

GitlabAPI& GitlabAPI::operator=(const GitlabAPI& rhs) {
	mInstanceUrl = rhs.mInstanceUrl;
	mLoggedIn = rhs.mLoggedIn;
	mAccessToken = rhs.mAccessToken;

	mCurlHandle = curl_easy_init();
	if(mCurlHandle == NULL) {
		MIRA_THROW(XRuntime, "GitlabAPI: Could not access curl library!");
	}
	return *this;
}

GitlabAPI::~GitlabAPI() {
	if(mCurlHandle != NULL) {
		curl_easy_cleanup(mCurlHandle);
	}
}

bool GitlabAPI::login(const std::string& username, const std::string& password) {
	if(!loggedIn()) {
		// https://docs.gitlab.com/ee/api/oauth2.html#resource-owner-password-credentials-flow
		// https://tools.ietf.org/html/rfc6749#section-4.3

		// We need to replace some special characters.
		std::string escapedPassword = std::string(curl_easy_escape(mCurlHandle, password.c_str(), password.length()));

		auto responseRaw = post("/oauth/token", "grant_type=password&username=" + username + "&password=" + escapedPassword);

		JSONValue authJSON;
		try {
			json::read(responseRaw, authJSON);
		} catch(XIO& ex) {
			MIRA_LOG(ERROR) << "Server returned non-JSON response: \n" << responseRaw;
			MIRA_RETHROW(ex, "Server response is no valid JSON!")
		}
		JSONDeserializer ds(authJSON);

		gitlab::Error e;
		try {
			ds.deserialize(e);
			MIRA_THROW(XRuntime, e.error << e.error_description);
		} catch(XRuntime&) {
			try {
				ds.deserialize(mAccessToken);
				mLoggedIn = true;
			} catch (XRuntime&) {
				MIRA_LOG(WARNING) << "Could not login. No access token returned from server " << mInstanceUrl;
			}
		}
	}
	return mLoggedIn;
}

bool GitlabAPI::provideApplicationToken(const std::string& token) {
	if(!mLoggedIn) {
		mAccessToken.access_token = token;
		mAccessToken.token_type = "bearer";
		mLoggedIn = true;
	}
	return mLoggedIn;
}

std::vector<Project> GitlabAPI::projects(unsigned int perPage) {
	std::vector<Project> projects;
	if(loggedIn()) {
		requestPaginated(projects, "/projects");
	}
	return projects;
}

std::vector<Project> GitlabAPI::projectsOfGroup(unsigned int groupID, bool subgroups, unsigned int perPage) {
	std::vector<Project> projects;
	projects.reserve(perPage);
	if(loggedIn()) {
		std::string requestString = MakeString()
			<< "/groups/" << groupID << "/projects"
			<< "?include_subgroups=" << (subgroups ? "true" : "false");
		requestPaginated(projects, requestString, perPage);
	}
	MIRA_LOG(NOTICE) << "Found " << projects.size() << " in group " << groupID;

	return projects;
}

std::vector<Namespace> GitlabAPI::namespaces(const std::string& search) {
	std::vector<Namespace> namespaces;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());
		long httpStatus = 0;

		std::string requestPath = "/namespaces";
		if(!search.empty()) {
			requestPath += "?search=" + search;
		}

		auto responseRaw = request(requestPath, httpStatus, headers);

		JSONValue namespacesJSON;
		json::read(responseRaw, namespacesJSON);
		JSONDeserializer ds(namespacesJSON);
		ds.deserialize(namespaces);
	}
	return namespaces;
}

Group GitlabAPI::group(const std::string& path) {
	Group group;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		long httpStatus = 0;
		headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());

		// this request uses the url-encoded path instead of the numerical ID,
		// which is permitted by the API documentation
		auto encodedPath = encodePath(path);
		if(boost::ends_with(encodedPath, "%2F")) {
			encodedPath = encodedPath.substr(0, encodedPath.size()-3);
		}
		std::string requestString = MakeString()
			<< "/groups/" << encodedPath;

		auto responseRaw = request(requestString, httpStatus, headers);

		JSONValue groupsJSON;
		json::read(responseRaw, groupsJSON);
		JSONDeserializer ds(groupsJSON);
		// try to deserialize the group
		try {
			ds.deserialize(group);
		} catch (XRuntime&) {
			// group could not be read, so try to deserialize error message
			Message msg;
			ds.deserialize(msg);
			MIRA_LOG(WARNING) << "Could not get group '" << path << "': " << msg.message;
		}
	}

	return group;
}

std::vector<Group> GitlabAPI::groups(const std::string& search, bool all, unsigned int page) {
	std::vector<Group> groups;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());
		long httpStatus = 0;

		std::string requestPath = "/groups";
		if(all) {
			requestPath += MakeString() << "?all_available=true&per_page=100&page=" << page;
		} else if(!search.empty()) {
			requestPath += MakeString() << "?search=" << search << "&per_page=100&page=" << page;
		}

		auto responseRaw = request(requestPath, httpStatus, headers);

		JSONValue groupsJSON;
		json::read(responseRaw, groupsJSON);
		JSONDeserializer ds(groupsJSON);
		ds.deserialize(groups);
	}
	return groups;
}

std::vector<Group> GitlabAPI::subgroups(const std::string& path) {
	std::vector<Group> groups;
	if(loggedIn()) {
		auto parentGroup = group(path);

		std::string requestPath = MakeString()
			<< "/groups/" << parentGroup.id
			<< "/subgroups"
			<< "?all_available=true";
		requestPaginated(groups, requestPath, 50);
		MIRA_LOG(NOTICE) << "Found " << groups.size()
						<< "subgroups in group " << parentGroup.name;
	}
	return groups;
}

std::vector<Tag> GitlabAPI::tags(unsigned int projectId, const std::string& search) {
	std::vector<Tag> tags;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());
		long httpStatus = 0;

		std::string requestPath = MakeString()
			<< "/projects/" << projectId << "/repository/tags";

		if(!search.empty()) {
			requestPath += "?search=" + search;
		}

		auto responseRaw = request(requestPath, httpStatus, headers);

		JSONValue tagsJSON;
		json::read(responseRaw, tagsJSON);
		JSONDeserializer ds(tagsJSON);
		ds.deserialize(tags);
	}
	return tags;
}

std::string GitlabAPI::rawFile(unsigned int projectId, std::string path, const std::string& ref) {
	std::string responseRaw;
	if(loggedIn()) {
		curl_slist* headers = NULL;
		headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());
		long httpStatus = 0;

		// url encode path
		if(boost::starts_with(path, "/") and path.length() > 1) {
			path = path.substr(1);
		}
		if(path.empty()) {
			MIRA_THROW(XIO, "No path to file.");
		}

		std::string requestString = MakeString()
			<< "/projects/" << projectId
			<< "/repository/files/" << encodePath(path)
			<< "/raw?ref=" << ref;

		responseRaw = request(requestString, httpStatus, headers);

		if(httpStatus == 404) {
			MIRA_THROW(XHTTP404, "File not found.");
		} else if(httpStatus > 200) {
			std::string msg = MakeString() << "Got HTTP " << httpStatus;
			MIRA_THROW(XNetworkError, msg.c_str());
		}
	}
	return responseRaw;
}

std::vector<Tree> GitlabAPI::listFiles(unsigned int projectId, const std::string& path, const std::string& ref) {
	std::vector<Tree> filesTree;

	std::string requestPath = MakeString()
		<< "/projects/" << projectId
		<< "/repository/tree" << encodePath(path)
		<< "?ref=" << ref;
	// try to request files
	try {
		requestPaginated(filesTree, requestPath, 50);
	} catch (XHTTP404&) {} // ignore errors for empty repositories

	return filesTree;
}

std::string GitlabAPI::request(const std::string& path, long& httpStatus, const curl_slist* header) {
	assert(!boost::starts_with(path, apiVersion()));

	std::string ret = "";
	ret.reserve(CURL_MAX_HTTP_HEADER);

	if(mCurlHandle != NULL) {
		auto url = mInstanceUrl + apiVersion() + path;
		MIRA_LOG(DEBUG) << url;
		long _http_status = 0;
		curl_easy_setopt(mCurlHandle, CURLOPT_HTTPHEADER, header);
		curl_easy_setopt(mCurlHandle, CURLOPT_URL, url.c_str());
		curl_easy_setopt(mCurlHandle, CURLOPT_HTTPGET, 1);
		curl_easy_getinfo(mCurlHandle, CURLINFO_RESPONSE_CODE, &httpStatus);

		// activate cookies for this handle
		//curl_easy_setopt(curl_handle, CURLOPT_COOKIEFILE, "");

		// save returned data
		curl_easy_setopt(mCurlHandle, CURLOPT_WRITEFUNCTION, write_callback);
		curl_easy_setopt(mCurlHandle, CURLOPT_WRITEDATA, static_cast<void*>(&ret));

		auto result = curl_easy_perform(mCurlHandle);
		httpStatus = _http_status;

		if(result != CURLE_OK) {
			std::string error_buffer(curl_easy_strerror(result));
			MIRA_LOG(ERROR) << "Request error: " << error_buffer;
			// todo rework in release
			MIRA_THROW(XNetworkError, "Request Error: " << error_buffer);
		}

		// This is too much to keep in production code. Especially if you
		// trigger a thorough reindex, this will absolutely spam your console.
		// MIRA_LOG(DEBUG) << ret;

		// keep going with https://curl.haxx.se/libcurl/c/cookie_interface.html
	} else {
		MIRA_LOG(ERROR) << "Curl handle is not initialized!";
	}

	return ret;
}

template<typename T>
void GitlabAPI::requestPaginated(std::vector<T>& oVec, const std::string& requestPath, unsigned int perPage) {
	if(!loggedIn())
		return;

	// prepare the request for pagination
	bool hasQuerySeparator = boost::contains(requestPath, "?");
	// bool hasQuerySeparator = requestPath.find("?") != std::string::npos;
	std::string paginatedRequest = MakeString()
		<< requestPath
		<< (hasQuerySeparator ? "&" : "?")
		<< "per_page=" << perPage << "&page=";

	curl_slist* headers = NULL;
	headers = curl_slist_append(headers, mAccessToken.authorizationHeader().c_str());
	long httpStatus = 0;

	// TODO: The API documentation recommends to use the rel="..." links in the 
	// response headers to navigate paginated results.
	unsigned int page = 1;
	while (true) {
		std::string paginatedRequestWithPage = paginatedRequest + std::to_string(page++);

		auto responseRaw = request(paginatedRequestWithPage, httpStatus, headers);
		JSONValue responseJson;
		json::read(responseRaw, responseJson);
		JSONDeserializer ds(responseJson);
		try {
			std::vector<T> tmp;
			ds.deserialize(tmp);
			if(tmp.empty()) {
				break;
			}
			oVec.insert(oVec.end(), tmp.begin(), tmp.end());
		}
		catch (XRuntime &) {
			Error e;
			// try to deserialize error
			try {
				ds.deserialize(e);
			} catch (XRuntime&) {
				// maybe it's just a plain message
				Message msg;
				ds.deserialize(msg);
				// copy message as error type
				e.error = msg.message;
			}
			if (boost::starts_with(e.error, "404")) {
				MIRA_THROW(XHTTP404, e.error);
			}
			else {
				MIRA_THROW(XNetworkError, e.error << ": " << e.error_description);
			}
		}
	}
}

std::string GitlabAPI::post(const std::string& path, const std::string& data, const curl_slist* header) {
	std::string ret;

	if(mCurlHandle != NULL) {
		curl_easy_setopt(mCurlHandle, CURLOPT_URL, (mInstanceUrl + path).c_str());
		curl_easy_setopt(mCurlHandle, CURLOPT_HTTPPOST, 1);
		curl_easy_setopt(mCurlHandle, CURLOPT_POSTFIELDS, data.c_str());

		// save received data
		curl_easy_setopt(mCurlHandle, CURLOPT_WRITEFUNCTION, write_callback);
		curl_easy_setopt(mCurlHandle, CURLOPT_WRITEDATA, static_cast<void*>(&ret));

		auto result = curl_easy_perform(mCurlHandle);
		if(result != CURLE_OK) {
			MIRA_LOG(ERROR) << "Could not perform post request: " << curl_easy_strerror(result);
		}
	} else {
		MIRA_LOG(ERROR) << "Curl handle is not initialized!";
	}
	return ret;
}

// *** DEBUG OPTION ***
void GitlabAPI::setVerbose(bool verbose) {
	if(verbose) {
		curl_easy_setopt(mCurlHandle, CURLOPT_VERBOSE, 1);
	} else {
		curl_easy_setopt(mCurlHandle, CURLOPT_VERBOSE, 0);
	}
}

void GitlabAPI::checkSSLCerts(bool check) {
	if(check) {
		curl_easy_setopt(mCurlHandle, CURLOPT_SSL_VERIFYPEER, 1);
	} else {
		curl_easy_setopt(mCurlHandle, CURLOPT_SSL_VERIFYPEER, 0);
	}
}

//std::string GitlabAPI::post(...) {}

size_t write_callback(char* ptr, size_t size, size_t nmemb, void* userdata) {
	auto buffer = static_cast<std::string*>(userdata);

	// if buffer.capacity is insufficient, this will reserve additional space
	buffer->reserve(nmemb);

	if(ptr != NULL) {
		buffer->append(ptr, nmemb);
	}

	return nmemb;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace gitlab
} // namespace mira

MIRA_CLASS_SERIALIZATION(mira::XNetworkError, mira::XRuntime);
MIRA_CLASS_SERIALIZATION(mira::XHTTP404,      mira::XNetworkError);
